@extends('layouts.user')

@section('title', 'Site Details')

@section('content')
    <div class="max-w-7xl mx-auto" x-data="{ activeTab: 'overview' }">
        <!-- Header Section (Dark Blue) -->
        <div class="bg-[#1e293b] text-white px-8 pt-10 pb-0 rounded-t-3xl mb-0 shadow-xl relative overflow-hidden">
            <!-- Subtle background decorative element -->
            <div class="absolute -right-20 -top-20 w-64 h-64 bg-blue-500/10 rounded-full blur-3xl"></div>

            <div class="flex flex-col md:flex-row md:items-center gap-6 mb-8 relative z-10">
                <div class="relative group">
                    <div
                        class="w-20 h-20 rounded-2xl bg-gradient-to-br from-blue-400 to-blue-600 flex items-center justify-center text-3xl font-bold border-4 border-slate-800 shadow-2xl overflow-hidden shadow-blue-500/20">
                        @if($site->logo)
                            <img src="{{ asset('storage/' . $site->logo) }}" class="w-full h-full object-cover">
                        @else
                            <span class="text-white">{{ substr($site->name, 0, 1) }}</span>
                        @endif
                    </div>
                </div>
                <div>
                    <div class="flex items-center gap-3">
                        <h1 class="text-3xl font-extrabold tracking-tight">{{ $site->name }}</h1>
                        <span
                            class="px-3 py-1 bg-blue-500/20 text-blue-300 text-[10px] font-bold uppercase rounded-full border border-blue-500/30">Site
                            Profile</span>
                    </div>
                    <div class="flex flex-wrap items-center gap-4 mt-2 text-sm text-slate-400 font-medium">
                        <span class="flex items-center gap-1.5">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z">
                                </path>
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                            </svg>
                            {{ $site->city }}, {{ $site->zip_code }}
                        </span>
                        <span class="flex items-center gap-1.5">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path>
                            </svg>
                            Code: {{ $site->site_id }}
                        </span>
                        <span class="flex items-center gap-1.5">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z">
                                </path>
                            </svg>
                            Rate: ${{ number_format($site->rate, 2) }}/hr
                        </span>
                    </div>
                </div>

                <div class="md:ml-auto flex items-center gap-2">
                    @can('edit site')
                        <a href="{{ route('sites.edit', $site->id) }}"
                            class="px-5 py-2.5 bg-blue-600 text-white text-xs font-bold rounded-xl shadow-lg shadow-blue-500/30 hover:bg-blue-700 transition-all uppercase tracking-wider flex items-center gap-2">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z">
                                </path>
                            </svg>
                            Edit Site
                        </a>
                    @endcan
                </div>
            </div>

            <!-- Primary Tabs -->
            <div class="flex items-center gap-8 text-sm font-bold relative z-10">
                <a href="#" @click.prevent="activeTab = 'overview'"
                    :class="activeTab === 'overview' ? 'border-white text-white' : 'border-transparent text-slate-400 hover:text-white'"
                    class="pb-4 border-b-4 transition-all">Overview</a>
                @can('view schedules')
                    <a href="#" @click.prevent="activeTab = 'schedules'"
                        :class="activeTab === 'schedules' ? 'border-white text-white' : 'border-transparent text-slate-400 hover:text-white'"
                        class="pb-4 border-b-4 transition-all">Schedules</a>
                @endcan
                <a href="#" @click.prevent="activeTab = 'banned'"
                    :class="activeTab === 'banned' ? 'border-white text-white' : 'border-transparent text-slate-400 hover:text-white'"
                    class="pb-4 border-b-4 transition-all">Banned Employees</a>
                <a href="#" @click.prevent="activeTab = 'checkpoints'"
                    :class="activeTab === 'checkpoints' ? 'border-white text-white' : 'border-transparent text-slate-400 hover:text-white'"
                    class="pb-4 border-b-4 transition-all">Checkpoints</a>
                <a href="#" @click.prevent="activeTab = 'tours'"
                    :class="activeTab === 'tours' ? 'border-white text-white' : 'border-transparent text-slate-400 hover:text-white'"
                    class="pb-4 border-b-4 transition-all">Tour Routes</a>
                <a href="#" @click="activeTab = 'reports'"
                    :class="activeTab === 'reports' ? 'border-white text-white' : 'border-transparent text-slate-400 hover:text-white'"
                    class="pb-4 border-b-4 transition-all">Operational Reports</a>

            </div>
        </div>

        <!-- Actions Bar (White) -->
        <div
            class="bg-white px-8 py-5 border-b border-gray-100 shadow-sm flex items-center justify-between mb-8 rounded-b-3xl">
            <div class="flex gap-10">
                <div>
                    <div class="text-[10px] text-gray-400 font-bold uppercase tracking-widest mb-1">Status</div>
                    <div class="flex items-center gap-2">
                        <span
                            class="w-2 h-2 rounded-full {{ $site->active ? 'bg-green-500 shadow-[0_0_8px_rgba(34,197,94,0.6)]' : 'bg-red-500' }}"></span>
                        <span class="text-xs font-bold text-gray-700">{{ $site->active ? 'Active' : 'Inactive' }}</span>
                    </div>
                </div>
                <div>
                    <div class="text-[10px] text-gray-400 font-bold uppercase tracking-widest mb-1">Account Type</div>
                    <div class="text-xs font-bold text-gray-700">{{ $site->account_type }}</div>
                </div>
                <div>
                    <div class="text-[10px] text-gray-400 font-bold uppercase tracking-widest mb-1">Site ID</div>
                    <div
                        class="text-xs font-bold text-blue-600 bg-blue-50 px-2 py-0.5 rounded border border-blue-100 leading-none inline-block mt-0.5">
                        {{ $site->site_id }}
                    </div>
                </div>
            </div>

            <a href="{{ route('sites.index') }}"
                class="flex items-center gap-2 px-4 py-2 bg-gray-50 text-gray-600 text-[10px] font-black rounded-xl border border-gray-200 hover:bg-white transition-all uppercase tracking-widest">
                <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18">
                    </path>
                </svg>
                Back to List
            </a>
        </div>

        <!-- Tab Content -->
        <div class="space-y-8 pb-12">
            <!-- Overview Tab -->
            <div x-show="activeTab === 'overview'" class="grid grid-cols-1 lg:grid-cols-3 gap-8"
                x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 translate-y-4"
                x-transition:enter-end="opacity-100 translate-y-0">
                <!-- Main Info -->
                <div class="lg:col-span-2 space-y-8">
                    <!-- General Information -->
                    <div class="bg-white rounded-3xl shadow-xl shadow-slate-200/50 border border-gray-100 overflow-hidden">
                        <div class="px-8 py-6 border-b border-gray-50 flex items-center justify-between bg-slate-50/50">
                            <div class="flex items-center gap-3">
                                <div
                                    class="w-10 h-10 rounded-xl bg-blue-600 flex items-center justify-center text-white shadow-lg shadow-blue-500/20">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                    </svg>
                                </div>
                                <h2 class="text-xs font-black text-slate-900 uppercase tracking-widest leading-none">Site
                                    Information</h2>
                            </div>
                        </div>
                        <div class="p-8 grid grid-cols-1 md:grid-cols-2 gap-x-12 gap-y-8">
                            <div>
                                <h3 class="text-[10px] font-bold text-gray-400 uppercase tracking-wider mb-2">Address</h3>
                                <p class="text-sm font-bold text-gray-800 leading-relaxed">{{ $site->address_line_1 }}</p>
                                <p class="text-xs text-gray-500 mt-1">{{ $site->city }}, {{ $site->zip_code }}</p>
                                <p class="text-[11px] font-medium text-gray-400 mt-0.5">{{ $site->state->name ?? '' }},
                                    {{ $site->country->name ?? '' }}
                                </p>
                            </div>
                            <div>
                                <h3 class="text-[10px] font-bold text-gray-400 uppercase tracking-wider mb-2">Geographic
                                    Coordinates</h3>
                                <div class="flex items-center gap-4">
                                    <div>
                                        <div class="text-[10px] text-gray-400">Lat</div>
                                        <div class="text-sm font-bold text-gray-800">{{ $site->latitude }}</div>
                                    </div>
                                    <div class="w-px h-8 bg-gray-100"></div>
                                    <div>
                                        <div class="text-[10px] text-gray-400">Long</div>
                                        <div class="text-sm font-bold text-gray-800">{{ $site->longitude }}</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Contact Details -->
                    <div class="bg-white rounded-3xl shadow-xl shadow-slate-200/50 border border-gray-100 overflow-hidden">
                        <div class="px-8 py-6 border-b border-gray-50 flex items-center justify-between bg-slate-50/50">
                            <div class="flex items-center gap-3">
                                <div
                                    class="w-10 h-10 rounded-xl bg-emerald-600 flex items-center justify-center text-white shadow-lg shadow-emerald-500/20">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z">
                                        </path>
                                    </svg>
                                </div>
                                <h2 class="text-xs font-black text-slate-900 uppercase tracking-widest leading-none">Contact
                                    Person</h2>
                            </div>
                        </div>
                        <div class="p-8 grid grid-cols-1 md:grid-cols-2 gap-x-12 gap-y-8">
                            <div>
                                <h3 class="text-[10px] font-bold text-gray-400 uppercase tracking-wider mb-2">Name</h3>
                                <p class="text-sm font-bold text-gray-800">{{ $site->contact_first_name }}
                                    {{ $site->contact_last_name }}
                                </p>
                                <p class="text-[11px] text-gray-500 mt-1 italic">{{ $site->contact_job_role }}</p>
                            </div>
                            <div class="space-y-4">
                                <div>
                                    <h3 class="text-[10px] font-bold text-gray-400 uppercase tracking-wider mb-1">Phone
                                        Number</h3>
                                    <p class="text-sm font-bold text-gray-800">{{ $site->contact_phone_main }}</p>
                                </div>
                                <div>
                                    <h3 class="text-[10px] font-bold text-gray-400 uppercase tracking-wider mb-1">Email
                                        Address</h3>
                                    <p class="text-sm font-bold text-blue-600">{{ $site->contact_email }}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Side Stats/Info -->
                <div class="space-y-8">
                    <!-- Quick Stats Card -->
                    <div
                        class="bg-gradient-to-br from-blue-600 to-indigo-700 rounded-3xl p-8 text-white shadow-xl shadow-blue-500/30">
                        <h3 class="text-xs font-black uppercase tracking-widest mb-6 opacity-70">Quick Statistics</h3>
                        @php
                            // Total Guards: Unique employees who have ever been scheduled for this site
                            $totalGuards = \Illuminate\Support\Facades\DB::table('employee_schedule')
                                ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
                                ->where('schedules.site_id', $site->id)
                                ->distinct('employee_schedule.employee_id')
                                ->count('employee_schedule.employee_id');

                            // Active Duty: Guards currently checked in (active status) for today's shifts at this site
                            $activeDuty = \Illuminate\Support\Facades\DB::table('employee_schedule')
                                ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
                                ->where('schedules.site_id', $site->id)
                                ->where('schedules.status', 'active')
                                ->whereDate('schedules.schedule_date', now())
                                ->count();

                            $percentage = $totalGuards > 0 ? ($activeDuty / $totalGuards) * 100 : 0;
                        @endphp
                        <div class="space-y-6">
                            <div class="flex items-center justify-between">
                                <span class="text-sm font-medium opacity-80">Total Guards</span>
                                <span class="text-2xl font-black">{{ $totalGuards }}</span>
                            </div>
                            <div class="w-full h-1.5 bg-white/20 rounded-full overflow-hidden">
                                <div class="h-full bg-white shadow-[0_0_10px_rgba(255,255,255,0.4)]"
                                    style="width: {{ $percentage }}%"></div>
                            </div>
                            <div class="flex items-center justify-between pt-2">
                                <span class="text-sm font-medium opacity-80">Active Duty</span>
                                <span class="text-2xl font-black">{{ $activeDuty }}</span>
                            </div>
                        </div>
                    </div>

                    <!-- Financial Summary -->
                    <div class="bg-white rounded-3xl shadow-xl shadow-slate-200/50 border border-gray-100 p-8">
                        <h3
                            class="text-xs font-black text-slate-900 uppercase tracking-widest mb-6 border-l-4 border-blue-600 pl-3">
                            Rate & Billing</h3>
                        <div class="text-center py-6 bg-slate-50 rounded-2xl border border-gray-100 mb-6">
                            <div class="text-[10px] font-bold text-gray-400 uppercase tracking-widest mb-1">Hourly Rate
                            </div>
                            <div class="text-4xl font-black text-slate-900">${{ number_format($site->rate, 2) }}</div>
                            <div class="text-[10px] text-gray-400 mt-1 font-medium italic">Standard Billing Rate</div>
                        </div>
                    </div>
                </div>
            </div>

            @can('view schedules')
                <div x-show="activeTab === 'schedules'" x-data="siteScheduleManager"
                    class="bg-white rounded-3xl shadow-xl shadow-slate-200/50 border border-gray-100 overflow-hidden min-h-[600px] relative"
                    x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 translate-y-4"
                    x-transition:enter-end="opacity-100 translate-y-0">


                    <!-- Tabs Header & Toolbar -->
                    <div class="px-8 border-b border-gray-100 flex items-center justify-between bg-slate-50/50">
                        <!-- Tabs Navigation -->
                        <nav class="flex space-x-8 -mb-px" aria-label="Tabs">
                            <button @click.prevent="isTemplateMode = false"
                                :class="!isTemplateMode ? 'border-blue-600 text-blue-700' : 'border-transparent text-gray-400 hover:text-gray-600 hover:border-gray-300'"
                                class="whitespace-nowrap py-5 px-1 border-b-2 font-bold text-sm transition-colors flex items-center gap-2">
                                <svg class="w-4 h-4" :class="!isTemplateMode ? 'text-blue-500' : 'text-gray-400'" fill="none"
                                    stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                </svg>
                                Live Schedule
                            </button>
                            <button @click.prevent="isTemplateMode = true"
                                :class="isTemplateMode ? 'border-blue-600 text-blue-700' : 'border-transparent text-gray-400 hover:text-gray-600 hover:border-gray-300'"
                                class="whitespace-nowrap py-5 px-1 border-b-2 font-bold text-sm transition-colors flex items-center gap-2">
                                <svg class="w-4 h-4" :class="isTemplateMode ? 'text-blue-500' : 'text-gray-400'" fill="none"
                                    stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10">
                                    </path>
                                </svg>
                                Master Template
                            </button>
                        </nav>

                        <!-- Context Actions -->
                        <div class="flex items-center gap-3 py-3">
                            <!-- Master Template Specific Actions -->
                            <div x-show="isTemplateMode" class="flex items-center gap-3 pl-4 border-l border-gray-200">
                                <!-- Cycle Selector -->
                                <div class="flex items-center gap-2">
                                    <span class="text-[10px] uppercase font-bold text-slate-400">Cycle:</span>
                                    <select x-model="cycleDays"
                                        class="text-xs border-gray-200 rounded-lg focus:ring-blue-500 py-1 pl-2 pr-6">
                                        <option value="7">7 Days</option>
                                        <option value="14">14 Days</option>
                                        <option value="28">28 Days</option>
                                    </select>
                                </div>

                                <!-- Delete Template -->
                                @can('edit schedule')
                                    <button x-show="template"
                                        @click="if(confirm('Are you sure you want to delete this template pattern?')) deleteTemplate()"
                                        class="p-1.5 text-red-400 hover:text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                                        title="Delete Template Pattern">
                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16">
                                            </path>
                                        </svg>
                                    </button>
                                @endcan

                                @can('edit schedule')
                                    <button @click="showRollModal = true"
                                        class="px-3 py-1.5 bg-green-500 text-white text-[10px] font-bold rounded-lg shadow-md shadow-green-500/30 uppercase tracking-widest hover:bg-green-600 transition-all flex items-center gap-2">
                                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z">
                                            </path>
                                        </svg>
                                        Publish Shifts
                                    </button>
                                @endcan
                            </div>

                            <!-- Live Schedule Specific Actions (if any) -->
                            <div x-show="!isTemplateMode">
                                <!-- (Currently the 'New Live Schedule' button is in the content area, which is fine) -->
                            </div>
                        </div>
                    </div>

                    <!-- Live Schedule Content -->
                    <div x-show="!isTemplateMode" class="p-6 bg-white min-h-[600px]">
                        <div class="flex flex-col md:flex-row md:items-end justify-between gap-4 mb-6">
                            <div class="flex flex-wrap items-end gap-3">
                                <div class="w-36">
                                    <label
                                        class="block text-[10px] font-bold text-gray-400 uppercase tracking-wider mb-1">From</label>
                                    <input type="date" x-model="fromDate" @change="fetchLiveSchedules()"
                                        class="w-full text-xs border-gray-200 rounded-lg focus:ring-blue-500 py-1.5 shadow-sm">
                                </div>
                                <div class="w-36">
                                    <label
                                        class="block text-[10px] font-bold text-gray-400 uppercase tracking-wider mb-1">To</label>
                                    <input type="date" x-model="toDate" @change="fetchLiveSchedules()"
                                        class="w-full text-xs border-gray-200 rounded-lg focus:ring-blue-500 py-1.5 shadow-sm">
                                </div>
                                <div class="flex gap-2">
                                    <a :href="`{{ route('schedules.export') }}?site_id={{ $site->id }}&from_date=${fromDate}&to_date=${toDate}`"
                                        title="Export PDF"
                                        class="h-[34px] w-[34px] flex items-center justify-center bg-white border border-gray-200 text-gray-500 rounded-lg hover:bg-gray-50 hover:text-red-600 transition-colors shadow-sm">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z">
                                            </path>
                                        </svg>
                                    </a>
                                    <form :action="`{{ route('schedules.email-client') }}`" method="POST" class="inline">
                                        @csrf
                                        <input type="hidden" name="site_id" value="{{ $site->id }}">
                                        <input type="hidden" name="from_date" :value="fromDate">
                                        <input type="hidden" name="to_date" :value="toDate">
                                        <button type="submit" title="Email to Client"
                                            class="h-[34px] w-[34px] flex items-center justify-center bg-white border border-gray-200 text-gray-500 rounded-lg hover:bg-gray-50 hover:text-blue-600 transition-colors shadow-sm">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z">
                                                </path>
                                            </svg>
                                        </button>
                                    </form>
                                </div>
                            </div>

                            @can('create schedule')
                                <a href="{{ route('schedules.create') }}?site_id={{ $site->id }}"
                                    class="px-4 py-2 bg-blue-600 text-white text-sm font-bold rounded-lg hover:bg-blue-700 shadow-lg shadow-blue-500/30 flex items-center gap-2">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4">
                                        </path>
                                    </svg>
                                    New Schedule
                                </a>
                            @endcan
                        </div>

                        <div x-show="isLoadingLive" class="text-center py-12 text-slate-400">Loading schedules...</div>

                        <div x-show="!isLoadingLive && liveSchedules.length === 0"
                            class="text-center py-12 text-slate-400 italic">
                            No upcoming schedules found for this site.
                        </div>

                        <div x-show="!isLoadingLive && liveSchedules.length > 0" class="overflow-x-auto">
                            <table class="w-full text-left text-sm text-slate-600">
                                <thead class="bg-slate-50 text-xs uppercase font-bold text-slate-400">
                                    <tr>
                                        <th class="px-4 py-3 rounded-l-lg">Date</th>
                                        <th class="px-4 py-3">Time</th>
                                        <th class="px-4 py-3">Employee</th>
                                        <th class="px-4 py-3">Status</th>
                                        <th class="px-4 py-3 rounded-r-lg"></th>
                                    </tr>
                                </thead>
                                <tbody class="divide-y divide-gray-100">
                                    <template x-for="schedule in liveSchedules" :key="schedule.id">
                                        <tr class="hover:bg-slate-50 transition-colors">
                                            <td class="px-4 py-3 font-bold text-slate-800" x-text="schedule.schedule_date"></td>
                                            <td class="px-4 py-3">
                                                <span x-text="schedule.from_datetime.substring(11,16)"></span> -
                                                <span x-text="schedule.to_datetime.substring(11,16)"></span>
                                            </td>
                                            <td class="px-4 py-3">
                                                <div class="flex items-center gap-2">
                                                    <template x-for="emp in schedule.employees" :key="emp.id">
                                                        <span
                                                            class="bg-blue-50 text-blue-700 px-2 py-0.5 rounded text-xs font-bold"
                                                            x-text="emp.first_name + ' ' + emp.last_name"></span>
                                                    </template>
                                                    <span x-show="!schedule.employees || schedule.employees.length === 0"
                                                        class="text-xs text-red-400 italic">Unassigned</span>
                                                </div>
                                            </td>
                                            <td class="px-4 py-3">
                                                <span class="px-2 py-1 rounded text-xs font-bold uppercase"
                                                    :class="{
                                                                                                                                                                                                                                                                                                                                                                    'bg-green-100 text-green-700': schedule.status === 'active' || schedule.status === 'completed',
                                                                                                                                                                                                                                                                                                                                                                    'bg-yellow-100 text-yellow-700': schedule.status === 'draft',
                                                                                                                                                                                                                                                                                                                                                                    'bg-red-100 text-red-700': schedule.status === 'cancelled'
                                                                                                                                                                                                                                                                                                                                                                }"
                                                    x-text="schedule.status"></span>
                                            </td>
                                            <td class="px-4 py-3 text-right">
                                                <a :href="`/schedules/${schedule.id}/edit`"
                                                    class="text-blue-600 hover:text-blue-800 font-bold text-xs">Edit</a>
                                            </td>
                                        </tr>
                                    </template>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- Template Content Wrapper -->
                    <div x-show="isTemplateMode" class="flex min-h-[600px] items-stretch">
                        <!-- Sidebar: Template List -->
                        <div class="w-64 flex-none border-r border-gray-100 bg-slate-50 p-4">
                            <div class="flex items-center justify-between mb-4 px-2">
                                <h3 class="text-xs font-black text-slate-400 uppercase tracking-wider">Templates</h3>
                                @can('edit schedule')
                                    <button @click="createNewTemplate()"
                                        class="text-blue-600 hover:bg-blue-100 p-1.5 rounded-lg transition-colors"
                                        title="Create New Template">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M12 4v16m8-8H4">
                                            </path>
                                        </svg>
                                    </button>
                                @endcan
                            </div>

                            <div class="space-y-1">
                                <template x-for="t in templates" :key="t.id">
                                    <button @click="selectTemplate(t)"
                                        class="w-full text-left px-3 py-2.5 rounded-lg text-xs font-bold transition-all flex items-center justify-between group"
                                        :class="template && template.id === t.id ? 'bg-white shadow-sm text-blue-700 ring-1 ring-blue-100' : 'text-slate-600 hover:bg-white hover:shadow-sm'">
                                        <span x-text="t.name"></span>
                                        <span x-show="template && template.id === t.id" class="text-blue-500">&bull;</span>
                                    </button>
                                </template>
                                <div x-show="templates.length === 0" class="text-center py-8 text-xs text-gray-400 italic">
                                    No templates yet.
                                </div>
                            </div>
                        </div>

                        <!-- Main Content: Grid -->
                        <div class="flex-1 p-6 bg-white">
                            <div x-show="!template && templates.length === 0" class="text-center py-12">
                                <div class="text-gray-400 mb-2">No Master Template set up.</div>
                                @can('edit schedule')
                                    <button @click="createNewTemplate()"
                                        class="bg-blue-600 text-white px-4 py-2 rounded-xl text-xs font-bold shadow-lg shadow-blue-500/30">Create
                                        First Template</button>
                                @endcan
                            </div>

                            <div x-show="!template && templates.length > 0" class="text-center py-12 text-gray-400">
                                Select a template from the left to view.
                            </div>

                            <div x-show="template">
                                <div class="mb-4 flex items-center justify-between">
                                    <h3 class="text-lg font-bold text-slate-800" x-text="template ? template.name : ''"></h3>
                                    <div class="flex items-center gap-2">
                                        <span class="text-xs text-slate-500 font-bold">Cycle Duration:</span>
                                        <select x-model="cycleDays" @change="saveSettings()"
                                            class="text-xs border-gray-200 rounded-lg focus:ring-blue-500 bg-slate-50">
                                            <option value="7">7 Days</option>
                                            <option value="14">14 Days</option>
                                            <option value="28">28 Days</option>
                                        </select>
                                    </div>
                                </div>

                                <div class="grid grid-cols-7 gap-4">
                                    <template x-for="day in parseInt(cycleDays)" :key="day">
                                        <div
                                            class="border border-gray-100 rounded-xl p-3 min-h-[150px] bg-slate-50 hover:border-blue-200 transition-colors group relative">
                                            <div
                                                class="text-[10px] font-black text-slate-400 uppercase mb-2 flex justify-between">
                                                <span x-text="'Day ' + day"></span>
                                                @can('edit schedule')
                                                    <button @click="openAddModal(day)"
                                                        class="opacity-0 group-hover:opacity-100 text-blue-600 font-bold text-[14px] leading-none"
                                                        title="Quick Add Shift">+</button>
                                                @endcan
                                            </div>

                                            <!-- Shifts for this day -->
                                            <template x-if="template && template.shifts">
                                                <div class="space-y-2">
                                                    <template x-for="shift in template.shifts.filter(s => s.day_offset == day)"
                                                        :key="shift.id">
                                                        <div
                                                            class="bg-white p-2 rounded-lg border border-gray-200 shadow-sm text-xs group/shift relative hover:border-blue-300 transition-colors">
                                                            <div class="font-bold text-slate-700"
                                                                x-text="shift.job_role ? shift.job_role.name : 'Unknown'"></div>
                                                            <div class="text-gray-500 text-[10px]">
                                                                <span x-text="shift.start_time.substring(0,5)"></span> - <span
                                                                    x-text="shift.end_time.substring(0,5)"></span>
                                                            </div>
                                                            <div x-show="shift.employee"
                                                                class="text-[9px] text-blue-600 truncate mt-1"
                                                                x-text="shift.employee ? shift.employee.first_name : ''"></div>

                                                            @can('edit schedule')
                                                                <button
                                                                    @click="fetch(`/schedule-templates/shifts/${shift.id}`, {method: 'DELETE', headers: {'X-CSRF-TOKEN': '{{ csrf_token() }}'}}).then(() => fetchTemplates())"
                                                                    class="absolute top-1 right-1 opacity-0 group-hover/shift:opacity-100 text-gray-300 hover:text-red-500 transition-all p-1">
                                                                    <svg class="w-3 h-3" fill="none" stroke="currentColor"
                                                                        viewBox="0 0 24 24">
                                                                        <path stroke-linecap="round" stroke-linejoin="round"
                                                                            stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                                                    </svg>
                                                                </button>
                                                            @endcan
                                                        </div>
                                                    </template>
                                                </div>
                                            </template>
                                        </div>
                                    </template>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Add Shift Modal Overlay -->
                    <div x-show="showAddModal"
                        class="absolute inset-0 z-50 flex items-center justify-center bg-slate-900/40 backdrop-blur-sm" x-cloak>
                        <div @click.outside="showAddModal = false"
                            class="bg-white rounded-2xl shadow-2xl w-full max-w-sm p-6 transform transition-all">
                            <h3 class="text-lg font-bold text-slate-800 mb-4">Add Shift (Day <span
                                    x-text="newShift.day_offset"></span>)</h3>

                            <div class="space-y-3">
                                <div>
                                    <label class="text-xs font-bold text-gray-500 uppercase block mb-1">Time Range</label>
                                    <div class="flex gap-2">
                                        <input type="time" x-model="newShift.start_time"
                                            class="w-full text-sm border-gray-200 rounded-lg">
                                        <input type="time" x-model="newShift.end_time"
                                            class="w-full text-sm border-gray-200 rounded-lg">
                                    </div>
                                </div>

                                <div>
                                    <label class="text-xs font-bold text-gray-500 uppercase block mb-1">Position (Role)</label>
                                    <select x-model="newShift.job_role_id" class="w-full text-sm border-gray-200 rounded-lg">
                                        <option value="">Select Role</option>
                                        @foreach($jobRoles as $role)
                                            <option value="{{ $role->id }}">{{ $role->name }}</option>
                                        @endforeach
                                    </select>
                                </div>

                                <div>
                                    <label class="text-xs font-bold text-gray-500 uppercase block mb-1">Assign Guard
                                        (Optional)</label>
                                    <select x-model="newShift.employee_id" class="w-full text-sm border-gray-200 rounded-lg">
                                        <option value="">Unassigned</option>
                                        @foreach($employees as $employee)
                                            <option value="{{ $employee->id }}">{{ $employee->first_name }}
                                                {{ $employee->last_name }}
                                            </option>
                                        @endforeach
                                    </select>
                                </div>

                                <div>
                                    <label class="text-xs font-bold text-gray-500 uppercase block mb-1">Unpaid Break
                                        (Min)</label>
                                    <input type="number" x-model="newShift.break_duration_minutes"
                                        class="w-full text-sm border-gray-200 rounded-lg" placeholder="0">
                                </div>
                            </div>

                            <div class="mt-6 flex justify-end gap-3">
                                <button @click="showAddModal = false"
                                    class="px-3 py-2 text-sm text-gray-500 hover:bg-gray-100 rounded-lg font-bold">Cancel</button>
                                <button @click="saveShift()"
                                    class="px-4 py-2 bg-blue-600 text-white text-sm rounded-lg font-bold hover:bg-blue-700">Add
                                    Shift</button>
                            </div>
                        </div>
                    </div>

                    <!-- Roll Schedule Modal -->
                    <div x-show="showRollModal"
                        class="absolute inset-0 z-50 flex items-center justify-center bg-slate-900/40 backdrop-blur-sm" x-cloak>
                        <div @click.outside="showRollModal = false"
                            class="bg-white rounded-2xl shadow-2xl w-full max-w-sm p-6 transform transition-all">
                            <h3 class="text-lg font-bold text-slate-800 mb-4">Roll Out Schedule</h3>

                            <div class="mb-4 p-3 bg-blue-50 border border-blue-100 rounded-lg">
                                <label class="text-xs font-bold text-blue-800 uppercase block mb-2">Target Templates</label>
                                <div class="max-h-32 overflow-y-auto space-y-1">
                                    <template x-for="t in templates" :key="t.id">
                                        <label class="flex items-center gap-2 cursor-pointer hover:bg-blue-100/50 p-1 rounded">
                                            <input type="checkbox" :value="t.id" x-model="selectedTemplatesToRoll"
                                                class="rounded text-blue-600 focus:ring-blue-500 border-gray-300 w-4 h-4">
                                            <span class="text-xs font-bold text-slate-700" x-text="t.name"></span>
                                        </label>
                                    </template>
                                </div>
                                <div class="text-[10px] text-blue-400 mt-2 italic"
                                    x-show="selectedTemplatesToRoll.length === 0">
                                    Defaults to currently selected template only.
                                </div>
                            </div>

                            <div class="space-y-3">
                                <div>
                                    <label class="text-xs font-bold text-gray-500 uppercase block mb-1">Start Date</label>
                                    <input type="date" x-model="rollStartDate"
                                        class="w-full text-sm border-gray-200 rounded-lg">
                                </div>
                                <div>
                                    <label class="text-xs font-bold text-gray-500 uppercase block mb-1">End Date</label>
                                    <input type="date" x-model="rollEndDate" class="w-full text-sm border-gray-200 rounded-lg">
                                </div>
                            </div>

                            <div class="mt-6 flex justify-end gap-3">
                                <button @click="showRollModal = false"
                                    class="px-3 py-2 text-sm text-gray-500 hover:bg-gray-100 rounded-lg font-bold">Cancel</button>
                                <button @click="rollTemplate()"
                                    class="px-4 py-2 bg-green-500 text-white text-sm rounded-lg font-bold hover:bg-green-600">Generate
                                    Shifts</button>
                            </div>
                        </div>
                    </div>
            @endcan
            </div>
        </div>

        @can('view schedules')
            <script>
                document.addEventListener('alpine:init', () => {
                    Alpine.data('siteScheduleManager', () => ({
                        isTemplateMode: false,
                        templates: [],
                        template: null,
                        isLoading: false,
                        cycleDays: 7,
                        showAddModal: false,
                        showRollModal: false,
                        rollStartDate: '',
                        rollEndDate: '',
                        selectedTemplatesToRoll: [],
                        newShift: { day_offset: 1, start_time: '09:00', end_time: '17:00', job_role_id: '', employee_id: '', break_duration_minutes: 0 },
                        liveSchedules: [],
                        isLoadingLive: false,
                        fromDate: '{{ now()->startOfMonth()->format('Y-m-d') }}',
                        toDate: '{{ now()->endOfMonth()->format('Y-m-d') }}',
                        init() {
                            this.fetchTemplates();
                            this.fetchLiveSchedules();
                        },
                        async fetchLiveSchedules() {
                            this.isLoadingLive = true;
                            try {
                                // Note: ensure route supports JSON response
                                let res = await fetch(`{{ route('schedules.index') }}?site_id={{ $site->id }}&from_date=${this.fromDate}&to_date=${this.toDate}&wantsJson=1`, {
                                    headers: { 'Accept': 'application/json' }
                                });
                                if (res.ok) {
                                    let data = await res.json();
                                    this.liveSchedules = data.schedules.data ? data.schedules.data : data.schedules;
                                }
                            } catch (e) {
                                console.error('Error fetching live schedules:', e);
                            }
                            this.isLoadingLive = false;
                        },
                        openAddModal(day) {
                            this.newShift.day_offset = day;
                            this.showAddModal = true;
                        },
                        selectTemplate(t) {
                            this.template = t;
                            this.cycleDays = t.cycle_duration_days;
                        },
                        createNewTemplate() {
                            let name = prompt('Enter name for new template:', 'New Schedule');
                            if (name) {
                                this.template = null; // Prepare for new
                                this.saveSettings(name);
                            }
                        },
                        async fetchTemplates() {
                            this.isLoading = true;
                            try {
                                let res = await fetch(`{{ route('schedule-templates.index', $site->id) }}`);
                                let data = await res.json();
                                this.templates = data.templates;
                                // Auto-select first if none selected
                                if (!this.template && this.templates.length > 0) {
                                    this.selectTemplate(this.templates[0]);
                                }
                                // If current template exists but updated, refresh it
                                if (this.template) {
                                    let fresh = this.templates.find(t => t.id === this.template.id);
                                    if (fresh) this.template = fresh;
                                }
                            } catch (e) {
                                console.error(e);
                            }
                            this.isLoading = false;
                        },
                        async saveSettings(newName = null) {
                            let payload = {
                                site_id: {{ $site->id }},
                                name: newName || (this.template ? this.template.name : 'Master Schedule'),
                                cycle_duration_days: this.cycleDays,
                                settings: {}
                            };
                            if (this.template && !newName) {
                                payload.id = this.template.id;
                            }

                            let res = await fetch(`{{ route('schedule-templates.store') }}`, {
                                method: 'POST',
                                headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': '{{ csrf_token() }}' },
                                body: JSON.stringify(payload)
                            });
                            if (res.ok) {
                                let data = await res.json();
                                this.fetchTemplates();
                                alert('Template saved!');
                            }
                        },
                        async saveShift() {
                            if (!this.template) { alert('Please create or select a template first.'); return; }
                            let res = await fetch(`/schedule-templates/${this.template.id}/shifts`, {
                                method: 'POST',
                                headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': '{{ csrf_token() }}' },
                                body: JSON.stringify(this.newShift)
                            });
                            if (res.ok) {
                                this.showAddModal = false;
                                this.fetchTemplates();
                            } else {
                                alert('Error saving shift');
                            }
                        },
                        async rollTemplate() {
                            let targets = this.selectedTemplatesToRoll.length > 0 ? this.selectedTemplatesToRoll : (this.template ? [this.template.id] : []);
                            if (targets.length === 0) { alert('No template selected to roll.'); return; }

                            let errorMessages = [];

                            for (let tId of targets) {
                                let res = await fetch(`/schedule-templates/${tId}/roll`, {
                                    method: 'POST',
                                    headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': '{{ csrf_token() }}' },
                                    body: JSON.stringify({
                                        start_date: this.rollStartDate,
                                        end_date: this.rollEndDate
                                    })
                                });
                                if (!res.ok) {
                                    try {
                                        let data = await res.json();
                                        if (res.status === 422 && data.errors) {
                                            errorMessages.push(`Template #${tId}: ` + data.errors.join(', '));
                                        } else {
                                            errorMessages.push(`Template #${tId}: ` + (data.message || 'Error'));
                                        }
                                    } catch (e) {
                                        errorMessages.push(`Template #${tId}: Unknown error`);
                                    }
                                }
                            }

                            if (errorMessages.length === 0) {
                                this.showRollModal = false;
                                this.selectedTemplatesToRoll = []; // specific reset
                                alert('Selected schedules successfully rolled out!');
                            } else {
                                alert(' Issues detected with some rollouts:\n- ' + errorMessages.join(' \n- '));
                            }
                        },
                        async deleteTemplate() {
                            if (!this.template) return;
                            if (!confirm('Delete "' + this.template.name + '"?')) return;
                            let res = await fetch(`/schedule-templates/${this.template.id}`, {
                                method: 'DELETE',
                                headers: { 'X-CSRF-TOKEN': '{{ csrf_token() }}' }
                            });
                            if (res.ok) {
                                this.template = null;
                                this.fetchTemplates(); // Refresh list
                                alert('Template deleted.');
                            } else {
                                alert('Error deleting template.');
                            }
                        }
                    }));
                });
            </script>
        @endcan


        <!-- Banned Employees Tab -->
        <div x-show="activeTab === 'banned'"
            class="bg-white rounded-3xl shadow-xl shadow-slate-200/50 border border-gray-100 overflow-hidden space-y-6"
            x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 translate-y-4"
            x-transition:enter-end="opacity-100 translate-y-0" x-cloak>
            <div class="p-8">
                <h2 class="text-2xl font-black text-slate-900 tracking-tight mb-6">Banned Employees</h2>

                @if($site->bannedEmployees->isEmpty())
                    <div class="p-12 text-center bg-gray-50 border-2 border-dashed border-gray-200 rounded-3xl">
                        <div class="text-gray-400 font-medium">No employees are currently banned from this site.</div>
                    </div>
                @else
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        @foreach($site->bannedEmployees as $employee)
                            <div class="bg-red-50 rounded-2xl border border-red-100 p-6 relative overflow-hidden group">
                                <div class="flex items-center gap-4 mb-4 relative z-10">
                                    <div
                                        class="w-12 h-12 rounded-xl bg-white flex items-center justify-center text-red-500 font-bold border border-red-200 shadow-sm">
                                        @if($employee->profile_picture)
                                            <img src="{{ asset('storage/' . $employee->profile_picture) }}"
                                                class="w-full h-full object-cover rounded-xl">
                                        @else
                                            {{ substr($employee->first_name, 0, 1) }}{{ substr($employee->last_name, 0, 1) }}
                                        @endif
                                    </div>
                                    <div>
                                        <h3 class="font-bold text-slate-900">{{ $employee->first_name }} {{ $employee->last_name }}
                                        </h3>
                                        <p class="text-[10px] font-bold text-slate-400 uppercase tracking-widest">
                                            {{ $employee->employee_id }}
                                        </p>
                                    </div>
                                </div>

                                <div class="relative z-10 space-y-3">
                                    <div class="bg-white/60 p-3 rounded-xl border border-red-200/50 backdrop-blur-sm">
                                        <p class="text-[10px] font-bold text-red-400 uppercase tracking-wider mb-1">Reason</p>
                                        <p class="text-xs text-red-900 font-medium italic">
                                            "{{ $employee->pivot->reason ?? 'No reason provided' }}"</p>
                                    </div>

                                    <div class="flex items-center justify-between pt-2">
                                        <span class="text-[10px] font-bold text-red-400 uppercase">Banned Since</span>
                                        <span
                                            class="text-xs font-bold text-red-700">{{ \Carbon\Carbon::parse($employee->pivot->banned_at)->format('M d, Y') }}</span>
                                    </div>

                                    @can('edit site')
                                        <form action="{{ route('sites.unban', ['site' => $site, 'employee' => $employee]) }}"
                                            method="POST" onsubmit="return confirm('Are you sure you want to unban this employee?');"
                                            class="pt-2">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit"
                                                class="w-full py-2 bg-white border border-red-200 text-red-600 text-xs font-bold rounded-xl shadow-sm hover:bg-red-600 hover:text-white transition-all uppercase tracking-wider">
                                                Unban Employee
                                            </button>
                                        </form>
                                    @endcan
                                </div>
                            </div>
                        @endforeach
                    </div>
                @endif
            </div>
        </div>

        <!-- Operational Reports Tab -->
        <div x-show="activeTab === 'reports'" class="space-y-8 pb-12" x-cloak
            x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 translate-y-4"
            x-transition:enter-end="opacity-100 translate-y-0">
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <!-- Shift Summary Card -->
                <a href="{{ route('reports.operational.index', ['site_id' => $site->id]) }}" class="block group">
                    <div
                        class="bg-white rounded-3xl p-8 shadow-xl shadow-slate-200/50 border border-gray-100 transition-all group-hover:-translate-y-1 group-hover:shadow-blue-500/10 h-full flex flex-col justify-between">
                        <div>
                            <div
                                class="w-14 h-14 rounded-2xl bg-blue-50 text-blue-600 flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                                <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-3 7h3m-3 4h3m-6-4h.01M9 16h.01">
                                    </path>
                                </svg>
                            </div>
                            <h3 class="text-xl font-bold text-slate-800 mb-2 group-hover:text-blue-600 transition-colors">
                                Shift
                                Summary</h3>
                            <p class="text-sm text-slate-500 font-medium leading-relaxed">View detailed logs of all shifts
                                for
                                this site, including check-in/out times and status.</p>
                        </div>
                        <div class="mt-8 flex items-center text-blue-600 font-bold text-xs uppercase tracking-wider">
                            View Report <svg class="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform" fill="none"
                                stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M17 8l4 4m0 0l-4 4m4-4H3"></path>
                            </svg>
                        </div>
                    </div>
                </a>

                <!-- Payout Summary Card -->
                <a href="{{ route('reports.operational.payouts', ['site_id' => $site->id]) }}" class="block group">
                    <div
                        class="bg-white rounded-3xl p-8 shadow-xl shadow-slate-200/50 border border-gray-100 transition-all group-hover:-translate-y-1 group-hover:shadow-green-500/10 h-full flex flex-col justify-between">
                        <div>
                            <div
                                class="w-14 h-14 rounded-2xl bg-green-50 text-green-600 flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                                <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z">
                                    </path>
                                </svg>
                            </div>
                            <h3 class="text-xl font-bold text-slate-800 mb-2 group-hover:text-green-600 transition-colors">
                                Payout Summary</h3>
                            <p class="text-sm text-slate-500 font-medium leading-relaxed">Analyze completed hours and
                                estimated
                                payouts for employees assigned to this site.</p>
                        </div>
                        <div class="mt-8 flex items-center text-green-600 font-bold text-xs uppercase tracking-wider">
                            View Report <svg class="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform" fill="none"
                                stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M17 8l4 4m0 0l-4 4m4-4H3"></path>
                            </svg>
                        </div>
                    </div>
                </a>

                <!-- Incident Summary Card -->
                <a href="{{ route('reports.operational.incidents', ['site_id' => $site->id]) }}" class="block group">
                    <div
                        class="bg-white rounded-3xl p-8 shadow-xl shadow-slate-200/50 border border-gray-100 transition-all group-hover:-translate-y-1 group-hover:shadow-red-500/10 h-full flex flex-col justify-between">
                        <div>
                            <div
                                class="w-14 h-14 rounded-2xl bg-red-50 text-red-600 flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                                <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                    </path>
                                </svg>
                            </div>
                            <h3 class="text-xl font-bold text-slate-800 mb-2 group-hover:text-red-600 transition-colors">
                                Incident Summary</h3>
                            <p class="text-sm text-slate-500 font-medium leading-relaxed">Review all incidents and security
                                reports filed for this location.</p>
                        </div>
                        <div class="mt-8 flex items-center text-red-600 font-bold text-xs uppercase tracking-wider">
                            View Report <svg class="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform" fill="none"
                                stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M17 8l4 4m0 0l-4 4m4-4H3"></path>
                            </svg>
                        </div>
                    </div>
                </a>

                <!-- Checkpoint Summary Card -->
                <a href="{{ route('reports.operational.index', ['site_id' => $site->id, 'type' => 'scans_history']) }}"
                    class="block group">
                    <div
                        class="bg-white rounded-3xl p-8 shadow-xl shadow-slate-200/50 border border-gray-100 transition-all group-hover:-translate-y-1 group-hover:shadow-purple-500/10 h-full flex flex-col justify-between">
                        <div>
                            <div
                                class="w-14 h-14 rounded-2xl bg-purple-50 text-purple-600 flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                                <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4">
                                    </path>
                                </svg>
                            </div>
                            <h3 class="text-xl font-bold text-slate-800 mb-2 group-hover:text-purple-600 transition-colors">
                                Checkpoint Summary</h3>
                            <p class="text-sm text-slate-500 font-medium leading-relaxed">Detailed history of all checkpoint
                                scans performed at this site.</p>
                        </div>
                        <div class="mt-8 flex items-center text-purple-600 font-bold text-xs uppercase tracking-wider">
                            View Report <svg class="w-4 h-4 ml-2 group-hover:translate-x-1 transition-transform" fill="none"
                                stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M17 8l4 4m0 0l-4 4m4-4H3"></path>
                            </svg>
                        </div>
                    </div>
                </a>
            </div>

            <!-- Quick Preview Table for Incidents -->
            <div class="bg-white rounded-3xl shadow-xl shadow-slate-200/50 border border-gray-100 overflow-hidden">
                <div class="px-8 py-6 border-b border-gray-50 flex items-center justify-between bg-slate-50/50">
                    <h3 class="text-xs font-black text-slate-900 uppercase tracking-widest leading-none">Recent Incidents at
                        {{ $site->name }}
                    </h3>
                    <a href="{{ route('reports.operational.incidents', ['site_id' => $site->id]) }}"
                        class="text-xs font-bold text-blue-600 hover:text-blue-700">View All</a>
                </div>
                <div class="overflow-x-auto">
                    <table class="w-full text-left">
                        <thead class="bg-slate-50 text-[10px] uppercase font-bold text-slate-400">
                            <tr>
                                <th class="px-8 py-4">Date</th>
                                <th class="px-8 py-4">Subject</th>
                                <th class="px-8 py-4">Employee</th>
                                <th class="px-8 py-4 text-right">Action</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-100">
                            @php
                                $recentIncidents = \Illuminate\Support\Facades\DB::table('incidents')
                                    ->join('schedules', 'incidents.schedule_id', '=', 'schedules.id')
                                    ->join('employees', 'incidents.employee_id', '=', 'employees.id')
                                    ->where('schedules.site_id', $site->id)
                                    ->orderBy('incidents.created_at', 'desc')
                                    ->limit(5)
                                    ->select('incidents.*', 'employees.first_name', 'employees.last_name')
                                    ->get();
                             @endphp
                            @forelse($recentIncidents as $idx => $inc)
                                <tr class="hover:bg-gray-50/50 transition-colors">
                                    <td class="px-8 py-4 text-xs font-bold text-gray-600">
                                        {{ \Carbon\Carbon::parse($inc->created_at)->format('M d, Y') }}
                                    </td>
                                    <td class="px-8 py-4 text-sm font-bold text-gray-800">{{ $inc->subject }}</td>
                                    <td class="px-8 py-4 text-xs font-medium text-gray-500">{{ $inc->first_name }}
                                        {{ $inc->last_name }}
                                    </td>
                                    <td class="px-8 py-4 text-right">
                                        <a href="#" class="text-xs font-bold text-blue-600 hover:underline">Details</a>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="4" class="px-8 py-8 text-center text-xs text-gray-400 italic">No recent
                                        incidents
                                        recorded.</td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>


        </div>

        <!-- Checkpoints Tab -->
        <div x-show="activeTab === 'checkpoints'" x-data="checkpointsManager({{ $site->id }})"
            class="bg-white rounded-3xl shadow-xl shadow-slate-200/50 border border-gray-100 overflow-hidden"
            x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 translate-y-4"
            x-transition:enter-end="opacity-100 translate-y-0" x-cloak>
            <div class="p-8">
                <div class="flex items-center justify-between mb-6">
                    <h2 class="text-2xl font-black text-slate-900 tracking-tight">Checkpoints</h2>
                    <button @click="openCreateModal"
                        class="px-4 py-2 bg-blue-600 text-white text-sm font-bold rounded-lg hover:bg-blue-700 shadow-lg shadow-blue-500/30">
                        + Create Checkpoint
                    </button>
                </div>

                <div x-show="checkpoints.length === 0"
                    class="p-12 text-center bg-gray-50 border-2 border-dashed border-gray-200 rounded-3xl">
                    <div class="text-gray-400 font-medium">No checkpoints created yet.</div>
                </div>

                <div x-show="checkpoints.length > 0" class="overflow-x-auto">
                    <table class="w-full text-left text-sm">
                        <thead class="bg-slate-50 text-xs uppercase font-bold text-slate-400">
                            <tr>
                                <th class="px-4 py-3">Name</th>
                                <th class="px-4 py-3">Checkpoint Type</th>
                                <th class="px-4 py-3">Monitoring</th>
                                <th class="px-4 py-3">Checkpoint ID</th>
                                <th class="px-4 py-3">Scan Action</th>
                                <th class="px-4 py-3">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-100">
                            <template x-for="checkpoint in checkpoints" :key="checkpoint.id">
                                <tr class="hover:bg-slate-50">
                                    <td class="px-4 py-3 font-bold text-slate-800" x-text="checkpoint.name"></td>
                                    <td class="px-4 py-3" x-text="checkpoint.scan_type"></td>
                                    <td class="px-4 py-3" x-text="checkpoint.monitoring_option"></td>
                                    <td class="px-4 py-3 font-mono text-xs" x-text="checkpoint.checkpoint_id_code || 'N/A'">
                                    </td>
                                    <td class="px-4 py-3">
                                        <div class="flex items-center gap-2">
                                            <span
                                                class="px-2 py-0.5 rounded-full text-[9px] font-black uppercase tracking-wider"
                                                :class="{
                                                                                    'bg-slate-100 text-slate-500': !checkpoint.extra_scan_options || (typeof checkpoint.extra_scan_options === 'string' ? JSON.parse(checkpoint.extra_scan_options).type : checkpoint.extra_scan_options.type) === 'log_only',
                                                                                    'bg-blue-100 text-blue-600': (typeof checkpoint.extra_scan_options === 'string' ? JSON.parse(checkpoint.extra_scan_options).type : checkpoint.extra_scan_options?.type) === 'message',
                                                                                    'bg-emerald-100 text-emerald-600': (typeof checkpoint.extra_scan_options === 'string' ? JSON.parse(checkpoint.extra_scan_options).type : checkpoint.extra_scan_options?.type) === 'photo'
                                                                                }"
                                                x-text="(typeof checkpoint.extra_scan_options === 'string' ? JSON.parse(checkpoint.extra_scan_options).type : checkpoint.extra_scan_options?.type)?.replace('_', ' ') || 'Log only'">
                                            </span>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3">
                                        <button @click="openEditModal(checkpoint)"
                                            class="text-blue-600 hover:text-blue-800 font-bold text-xs mr-3">Edit</button>
                                        <button @click="deleteCheckpoint(checkpoint.id)"
                                            class="text-red-600 hover:text-red-800 font-bold text-xs">Delete</button>
                                    </td>
                                </tr>
                            </template>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Create/Edit Checkpoint Modal -->
            <div x-show="showModal"
                class="fixed inset-0 z-50 flex items-center justify-center bg-slate-900/40 backdrop-blur-sm" x-cloak>
                <div @click.outside="showModal = false"
                    class="bg-white rounded-2xl shadow-2xl w-full max-w-2xl p-6 max-h-[90vh] overflow-y-auto">
                    <h3 class="text-lg font-bold text-slate-800 mb-4"
                        x-text="editingCheckpoint ? 'Edit Checkpoint' : 'Create Checkpoint'"></h3>

                    <form @submit.prevent="saveCheckpoint" class="space-y-4">
                        <div class="space-y-1.5">
                            <label
                                class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Checkpoint
                                Name *</label>
                            <input type="text" x-model="form.name" required
                                class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 outline-none focus:border-blue-500 transition-all"
                                placeholder="e.g. Main Gate">
                        </div>

                        <div class="space-y-1.5">
                            <label
                                class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Special
                                Instructions</label>
                            <textarea x-model="form.special_instructions" rows="2"
                                class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-medium text-slate-600 outline-none focus:border-blue-500 transition-all"
                                placeholder="Add specific instructions..."></textarea>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Checkpoint
                                    Type (Scan method) *</label>
                                <select x-model="form.scan_type" required
                                    class="w-full px-4 py-3 bg-slate-50 border-slate-200 rounded-xl text-sm font-bold text-slate-700 outline-none focus:border-blue-500 transition-all cursor-pointer">
                                    <option value="all">All (QR & NFC)</option>
                                    <option value="qr">QR Code Only</option>
                                    <option value="nfc">NFC Tag Only</option>
                                    <option value="manual">Manual Entry Only</option>
                                </select>
                            </div>
                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Monitoring
                                    Options *</label>
                                <select x-model="form.monitoring_option" required
                                    class="w-full px-4 py-3 bg-slate-50 border-slate-200 rounded-xl text-sm font-bold text-slate-700 outline-none focus:border-blue-500 transition-all cursor-pointer">
                                    <option value="regular">Request a scan on a regular interval</option>
                                </select>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Scan
                                    Interval (Min)</label>
                                <input type="number" x-model="form.scan_interval_minutes"
                                    class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 outline-none focus:border-blue-500 transition-all"
                                    placeholder="60">
                            </div>
                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Grace
                                    Period (Min)</label>
                                <input type="number" x-model="form.grace_period_minutes"
                                    class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 outline-none focus:border-blue-500 transition-all"
                                    placeholder="0">
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <div class="space-y-1.5 md:col-span-1">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">ID
                                    Method</label>
                                <div class="flex gap-1 p-1 bg-slate-100 rounded-xl mt-1">
                                    <button type="button" @click="form.id_method = 'code'"
                                        :class="form.id_method === 'code' ? 'bg-white shadow-sm text-blue-600' : 'text-slate-500'"
                                        class="flex-1 py-2 text-[9px] font-black uppercase tracking-wider rounded-lg transition-all">Code</button>
                                    <button type="button" @click="form.id_method = 'image'"
                                        :class="form.id_method === 'image' ? 'bg-white shadow-sm text-blue-600' : 'text-slate-500'"
                                        class="flex-1 py-2 text-[9px] font-black uppercase tracking-wider rounded-lg transition-all">Image</button>
                                </div>
                            </div>
                            <div class="space-y-1.5 md:col-span-2">
                                <div x-show="form.id_method === 'code'">
                                    <label
                                        class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Checkpoint
                                        ID Code</label>
                                    <input type="text" x-model="form.checkpoint_id_code"
                                        class="w-full mt-1 px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-mono text-slate-700 outline-none focus:border-blue-500 transition-all"
                                        placeholder="e.g. NFC7823901">
                                </div>
                                <div x-show="form.id_method === 'image'">
                                    <label
                                        class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Upload
                                        NFC Image</label>
                                    <div class="relative group mt-1">
                                        <input type="file" @change="handleImageUpload($event)" accept="image/*"
                                            class="absolute inset-0 w-full h-full opacity-0 cursor-pointer z-10">
                                        <div
                                            class="w-full px-4 py-3 bg-slate-50 border border-dashed border-slate-300 rounded-xl text-[10px] font-bold text-slate-400 group-hover:border-blue-500 group-hover:text-blue-500 transition-all flex items-center justify-center gap-2">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                            </svg>
                                            <span x-text="form.image_name || 'Select Image'"></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Extra Scan Options Section -->
                        <div class="pt-4 border-t border-slate-100">
                            <label class="block text-[10px] font-black text-slate-400 uppercase tracking-widest mb-4">Extra
                                Scan Options</label>

                            <div class="mb-4">
                                <select x-model="form.extra_scan_options.type"
                                    class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 outline-none focus:border-blue-500 transition-all cursor-pointer">
                                    <option value="log_only">Log only (default)</option>
                                    <option value="message">Display a message</option>
                                    <option value="photo">Upload photos (multiple)</option>
                                </select>
                            </div>

                            <!-- Conditional Content for Message -->
                            <div x-show="form.extra_scan_options.type === 'message'" x-transition
                                class="space-y-1.5 p-4 bg-blue-50/50 rounded-2xl border border-blue-100">
                                <label
                                    class="block text-[10px] font-black text-blue-600 uppercase tracking-widest leading-none">Custom
                                    Message Text</label>
                                <textarea x-model="form.extra_scan_options.message_content" rows="3"
                                    class="w-full px-4 py-3 bg-white border border-blue-100 rounded-xl text-sm font-medium text-slate-600 outline-none focus:border-blue-500 transition-all"
                                    placeholder="Enter the message you want the guard to see upon scanning..."></textarea>
                            </div>

                            <!-- Conditional Content for Photo -->
                            <div x-show="form.extra_scan_options.type === 'photo'" x-transition
                                class="space-y-4 p-4 bg-emerald-50/50 rounded-2xl border border-emerald-100">
                                <div class="flex items-center justify-between">
                                    <label
                                        class="block text-[10px] font-black text-emerald-600 uppercase tracking-widest leading-none">Photo
                                        Options</label>
                                    <div class="flex items-center gap-2">
                                        <input type="checkbox" x-model="form.extra_scan_options.allow_multiple_photos"
                                            id="multiple_photos" class="rounded text-emerald-600 border-emerald-200">
                                        <label for="multiple_photos"
                                            class="text-[10px] font-bold text-emerald-700 uppercase">Allow Multiple
                                            Photos</label>
                                    </div>
                                </div>
                                <div class="space-y-1.5">
                                    <label
                                        class="block text-[10px] font-black text-emerald-600 uppercase tracking-widest leading-none">Instruction/Description</label>
                                    <input type="text" x-model="form.extra_scan_options.photo_description"
                                        class="w-full px-4 py-3 bg-white border border-emerald-100 rounded-xl text-sm font-medium text-slate-600 outline-none focus:border-emerald-500 transition-all"
                                        placeholder="e.g. Please take a clear photo of the fire extinguisher pressure gauge">
                                </div>
                            </div>
                        </div>

                        <div class="pt-6 border-t border-slate-100 flex justify-end gap-3">
                            <button type="button" @click="showModal = false"
                                class="px-6 py-2.5 text-xs font-black uppercase tracking-widest text-slate-400 hover:text-slate-600 transition-colors">Discard</button>
                            <button type="submit"
                                class="px-8 py-2.5 bg-blue-600 text-white text-xs font-bold uppercase tracking-widest rounded-xl shadow-lg shadow-blue-500/40 hover:bg-blue-700 hover:-translate-y-0.5 transition-all">Save
                                Checkpoint</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Tour Routes Tab -->
        <div x-show="activeTab === 'tours'" x-data="tourRoutesManager({{ $site->id }})"
            class="bg-white rounded-3xl shadow-xl shadow-slate-200/50 border border-gray-100 overflow-hidden"
            x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 translate-y-4"
            x-transition:enter-end="opacity-100 translate-y-0" x-cloak>
            <div class="p-8">
                <div class="flex items-center justify-between mb-6">
                    <h2 class="text-2xl font-black text-slate-900 tracking-tight">Tour Routes</h2>
                    <button @click="openCreateModal"
                        class="px-4 py-2 bg-blue-600 text-white text-sm font-bold rounded-lg hover:bg-blue-700 shadow-lg shadow-blue-500/30">
                        + Create Tour Route
                    </button>
                </div>

                <div x-show="tourRoutes.length === 0"
                    class="p-12 text-center bg-gray-50 border-2 border-dashed border-gray-200 rounded-3xl">
                    <div class="text-gray-400 font-medium">No tour routes created yet.</div>
                </div>

                <div x-show="tourRoutes.length > 0" class="overflow-x-auto">
                    <table class="w-full text-left text-sm">
                        <thead class="bg-slate-50 text-xs uppercase font-bold text-slate-400">
                            <tr>
                                <th class="px-4 py-3">Description</th>
                                <th class="px-4 py-3">Assigned To</th>
                                <th class="px-4 py-3">Recurrence</th>
                                <th class="px-4 py-3">Duration</th>
                                <th class="px-4 py-3">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-100">
                            <template x-for="route in tourRoutes" :key="route.id">
                                <tr class="hover:bg-slate-50">
                                    <td class="px-4 py-3 font-bold text-slate-800" x-text="route.description"></td>
                                    <td class="px-4 py-3" x-text="route.assigned_to"></td>
                                    <td class="px-4 py-3" x-text="route.recurrence_type"></td>
                                    <td class="px-4 py-3" x-text="route.estimated_duration_minutes + ' min'"></td>
                                    <td class="px-4 py-3">
                                        <button @click="openEditModal(route)"
                                            class="text-blue-600 hover:text-blue-800 font-bold text-xs mr-3">Edit</button>
                                        <button @click="deleteTourRoute(route.id)"
                                            class="text-red-600 hover:text-red-800 font-bold text-xs">Delete</button>
                                    </td>
                                </tr>
                            </template>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Create/Edit Tour Route Modal -->
            <div x-show="showModal"
                class="fixed inset-0 z-50 flex items-center justify-center bg-slate-900/40 backdrop-blur-sm" x-cloak>
                <div @click.outside="showModal = false"
                    class="bg-white rounded-[2rem] shadow-2xl w-full max-w-2xl p-8 max-h-[90vh] overflow-y-auto border border-blue-50/50 relative">
                    <div class="flex items-center justify-between mb-8">
                        <div>
                            <h3 class="text-2xl font-black text-slate-800 tracking-tight"
                                x-text="editingRoute ? 'Edit Tour Route' : 'Create Tour Route'"></h3>
                            <p class="text-xs font-bold text-slate-400 uppercase tracking-widest mt-1">Configure patrol path
                                and schedule</p>
                        </div>
                        <button @click="showModal = false"
                            class="p-2 bg-slate-50 text-slate-400 hover:text-slate-600 rounded-xl transition-all hover:rotate-90">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5"
                                    d="M6 18L18 6M6 6l12 12" />
                            </svg>
                        </button>
                    </div>

                    <form @submit.prevent="saveTourRoute" class="space-y-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Route
                                    Description *</label>
                                <input type="text" x-model="form.description" required
                                    class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 focus:border-blue-500 focus:ring-4 focus:ring-blue-500/5 transition-all outline-none"
                                    placeholder="e.g. Night Shift Perimeter Walk">
                            </div>

                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Assigned
                                    To *</label>
                                <select x-model="form.assigned_to" required
                                    class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 focus:border-blue-500 transition-all outline-none cursor-pointer">
                                    <option value="all">Every Guard on Site</option>
                                    <option value="specific">Specific Guard (V3.8)</option>
                                </select>
                            </div>
                        </div>

                        <div class="space-y-1.5">
                            <label
                                class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Special
                                Instructions</label>
                            <textarea x-model="form.special_instructions" rows="2"
                                class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-medium text-slate-600 focus:border-blue-500 transition-all outline-none"
                                placeholder="Instructions/Reminders for this specific route..."></textarea>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Recurrence
                                    Type *</label>
                                <select x-model="form.recurrence_type" required
                                    class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 focus:border-blue-500 transition-all outline-none cursor-pointer">
                                    <option value="weekly">Weekly Schedule</option>
                                    <option value="monthly">Monthly Cycle</option>
                                </select>
                            </div>
                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Est.
                                    Duration (Min)</label>
                                <input type="number" x-model="form.estimated_duration_minutes"
                                    class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 focus:border-blue-500 transition-all outline-none"
                                    placeholder="30">
                            </div>

                            <div class="space-y-1.5">
                                <label
                                    class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Grace
                                    Period (Min)</label>
                                <input type="number" x-model="form.grace_period_minutes"
                                    class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-bold text-slate-700 focus:border-blue-500 transition-all outline-none"
                                    placeholder="10">
                            </div>
                        </div>

                        <!-- Weekly Schedule -->
                        <div x-show="form.recurrence_type === 'weekly'"
                            class="space-y-3 p-5 bg-slate-50 rounded-2xl border border-slate-100" x-transition>
                            <label class="block text-[10px] font-black text-slate-400 uppercase tracking-widest">Active
                                Days</label>
                            <div class="flex flex-wrap gap-2">
                                <template
                                    x-for="day in ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday']"
                                    :key="day">
                                    <label class="relative cursor-pointer group">
                                        <input type="checkbox" :value="day" x-model="form.schedule_days"
                                            class="sr-only peer">
                                        <div
                                            class="px-4 py-2 bg-white border border-slate-200 rounded-xl text-xs font-bold text-slate-500 transition-all peer-checked:bg-blue-600 peer-checked:text-white peer-checked:border-blue-600 hover:border-blue-300 shadow-sm">
                                            <span x-text="day.substring(0,3)"></span>
                                        </div>
                                    </label>
                                </template>
                            </div>
                        </div>

                        <!-- Monthly Schedule -->
                        <div x-show="form.recurrence_type === 'monthly'"
                            class="space-y-3 p-5 bg-slate-50 rounded-2xl border border-slate-100" x-transition>
                            <label class="block text-[10px] font-black text-slate-400 uppercase tracking-widest">Monthly Day
                                Selection</label>
                            <div class="grid grid-cols-7 gap-1.5">
                                <template x-for="i in 31" :key="i">
                                    <label class="relative cursor-pointer">
                                        <input type="checkbox" :value="i" x-model="form.schedule_days" class="sr-only peer">
                                        <div class="w-8 h-8 flex items-center justify-center bg-white border border-slate-200 rounded-lg text-[10px] font-bold text-slate-500 transition-all peer-checked:bg-blue-600 peer-checked:text-white peer-checked:border-blue-600 hover:border-blue-300 shadow-sm"
                                            x-text="i"></div>
                                    </label>
                                </template>
                            </div>
                        </div>

                        <!-- Checkpoint Selection -->
                        <div class="space-y-3">
                            <label
                                class="block text-[10px] font-black text-slate-400 uppercase tracking-widest leading-none">Checkpoints
                                to include *</label>
                            <div
                                class="grid grid-cols-1 sm:grid-cols-2 gap-3 max-h-60 overflow-y-auto p-1 custom-scrollbar">
                                <template x-for="cp in availableCheckpoints" :key="cp.id">
                                    <label
                                        class="flex items-center gap-3 p-4 bg-white border border-slate-100 rounded-2xl cursor-pointer transition-all hover:border-blue-200 hover:shadow-xl hover:shadow-blue-500/5 group relative border-l-4"
                                        :class="form.checkpoint_ids.includes(cp.id) ? 'border-l-blue-600' : 'border-l-transparent'">
                                        <input type="checkbox" :value="cp.id" x-model="form.checkpoint_ids"
                                            class="w-4 h-4 rounded text-blue-600 border-slate-300 focus:ring-blue-500 transition-all">
                                        <div class="flex-1 min-w-0">
                                            <div class="text-xs font-black text-slate-700 truncate" x-text="cp.name"></div>
                                            <div class="text-[9px] font-bold text-slate-400 uppercase tracking-tight"
                                                x-text="cp.checkpoint_id_code || 'ID not set'"></div>
                                        </div>
                                    </label>
                                </template>
                                <template x-if="availableCheckpoints.length === 0">
                                    <div
                                        class="col-span-full text-center py-10 bg-slate-50 rounded-2xl border border-dashed border-slate-200">
                                        <div
                                            class="w-12 h-12 bg-white rounded-full flex items-center justify-center mx-auto mb-3 shadow-sm">
                                            <svg class="w-6 h-6 text-slate-300" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                                            </svg>
                                        </div>
                                        <div class="text-xs font-bold text-slate-400 italic uppercase tracking-widest">No
                                            site checkpoints found</div>
                                        <p class="text-[10px] text-slate-300 mt-1">Please create checkpoints for this site
                                            first.</p>
                                    </div>
                                </template>
                            </div>
                        </div>

                        <div class="pt-6 border-t border-slate-100 flex justify-end gap-3">
                            <button type="button" @click="showModal = false"
                                class="px-6 py-2.5 text-xs font-black uppercase tracking-widest text-slate-400 hover:text-slate-600 transition-colors">Discard</button>
                            <button type="submit"
                                class="px-8 py-2.5 bg-blue-600 text-white text-xs font-black uppercase tracking-widest rounded-xl shadow-lg shadow-blue-500/40 hover:bg-blue-700 hover:-translate-y-0.5 transition-all">Save
                                Tour Route</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    </div>

    <script>
        // Checkpoints Manager
        function checkpointsManager(siteId) {
            return {
                checkpoints: [],
                showModal: false,
                editingCheckpoint: null,
                form: {
                    name: '',
                    special_instructions: '',
                    scan_type: 'all',
                    monitoring_option: 'regular',
                    scan_interval_minutes: null,
                    checkpoint_type: '',
                    checkpoint_id_code: '',
                    allow_manual_scan: false,
                    grace_period_minutes: 0,
                    checkpoint_id_image: null,
                    image_name: '',
                    extra_scan_options: {
                        type: 'log_only',
                        message_content: '',
                        photo_description: '',
                        allow_multiple_photos: false
                    }
                },
                init() {
                    this.fetchCheckpoints();
                },
                async fetchCheckpoints() {
                    try {
                        const res = await fetch(`/sites/${siteId}/checkpoints`, {
                            headers: { 'Accept': 'application/json' }
                        });
                        const data = await res.json();
                        this.checkpoints = data.checkpoints;
                    } catch (e) {
                        console.error('Error fetching checkpoints:', e);
                    }
                },
                openCreateModal() {
                    this.editingCheckpoint = null;
                    this.resetForm();
                    this.showModal = true;
                },
                openEditModal(checkpoint) {
                    this.editingCheckpoint = checkpoint;
                    this.form = {
                        ...checkpoint,
                        id_method: checkpoint.checkpoint_id_image ? 'image' : 'code',
                        image_name: checkpoint.checkpoint_id_image ? 'Existing Image' : '',
                        extra_scan_options: checkpoint.extra_scan_options ? (typeof checkpoint.extra_scan_options === 'string' ? JSON.parse(checkpoint.extra_scan_options) : checkpoint.extra_scan_options) : {
                            type: 'log_only',
                            message_content: '',
                            photo_description: '',
                            allow_multiple_photos: false
                        }
                    };
                    this.showModal = true;
                },
                resetForm() {
                    this.form = {
                        name: '',
                        special_instructions: '',
                        scan_type: 'all',
                        monitoring_option: 'regular',
                        scan_interval_minutes: null,
                        checkpoint_type: '',
                        checkpoint_id_code: '',
                        allow_manual_scan: false,
                        grace_period_minutes: 0,
                        id_method: 'code',
                        image_name: '',
                        extra_scan_options: {
                            type: 'log_only',
                            message_content: '',
                            photo_description: '',
                            allow_multiple_photos: false
                        }
                    };
                },
                handleImageUpload(e) {
                    const file = e.target.files[0];
                    if (file) {
                        this.form.checkpoint_id_image = file;
                        this.form.image_name = file.name;
                    }
                },
                async saveCheckpoint() {
                    try {
                        const formData = new FormData();
                        Object.keys(this.form).forEach(key => {
                            if (key === 'checkpoint_id_image' && this.form[key] instanceof File) {
                                formData.append(key, this.form[key]);
                            } else if (key === 'extra_scan_options') {
                                formData.append(key, JSON.stringify(this.form[key]));
                            } else if (this.form[key] !== null) {
                                formData.append(key, this.form[key]);
                            }
                        });

                        const url = this.editingCheckpoint
                            ? `/sites/${siteId}/checkpoints/${this.editingCheckpoint.id}`
                            : `/sites/${siteId}/checkpoints`;

                        // Laravel spoofing for PUT if editing
                        if (this.editingCheckpoint) {
                            formData.append('_method', 'PUT');
                        }

                        const res = await fetch(url, {
                            method: 'POST', // Always POST for FormData with file support
                            headers: {
                                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                'Accept': 'application/json'
                            },
                            body: formData
                        });

                        if (res.ok) {
                            this.showModal = false;
                            this.fetchCheckpoints();
                            alert(this.editingCheckpoint ? 'Checkpoint updated successfully' : 'Checkpoint created successfully');
                        } else {
                            const data = await res.json();
                            alert('Error: ' + (data.message || 'Failed to save checkpoint'));
                        }
                    } catch (e) {
                        console.error('Error saving checkpoint:', e);
                        alert('Error saving checkpoint');
                    }
                },
                async deleteCheckpoint(id) {
                    if (!confirm('Are you sure you want to delete this checkpoint?')) return;

                    try {
                        const res = await fetch(`/sites/${siteId}/checkpoints/${id}`, {
                            method: 'DELETE',
                            headers: {
                                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                'Accept': 'application/json'
                            }
                        });

                        if (res.ok) {
                            this.fetchCheckpoints();
                            alert('Checkpoint deleted successfully');
                        }
                    } catch (e) {
                        console.error('Error deleting checkpoint:', e);
                        alert('Error deleting checkpoint');
                    }
                }
            };
        }

        // Tour Routes Manager
        function tourRoutesManager(siteId) {
            return {
                tourRoutes: [],
                showModal: false,
                editingRoute: null,
                form: {
                    description: '',
                    assigned_to: 'all',
                    special_instructions: '',
                    estimated_duration_minutes: 0,
                    grace_period_minutes: 0,
                    recurrence_type: 'weekly',
                    schedule_days: [],
                    checkpoint_ids: []
                },
                availableCheckpoints: [],
                init() {
                    this.fetchTourRoutes();
                    this.fetchAvailableCheckpoints();
                },
                async fetchAvailableCheckpoints() {
                    try {
                        const res = await fetch(`/sites/${siteId}/checkpoints`, {
                            headers: { 'Accept': 'application/json' }
                        });
                        const data = await res.json();
                        this.availableCheckpoints = data.checkpoints;
                    } catch (e) {
                        console.error('Error fetching checkpoints for tours:', e);
                    }
                },
                async fetchTourRoutes() {
                    try {
                        const res = await fetch(`/sites/${siteId}/tour-routes`, {
                            headers: { 'Accept': 'application/json' }
                        });
                        const data = await res.json();
                        this.tourRoutes = data.tourRoutes;
                    } catch (e) {
                        console.error('Error fetching tour routes:', e);
                    }
                },
                openCreateModal() {
                    this.editingRoute = null;
                    this.resetForm();
                    this.fetchAvailableCheckpoints();
                    this.showModal = true;
                },
                openEditModal(route) {
                    this.editingRoute = route;
                    this.fetchAvailableCheckpoints();
                    this.form = {
                        ...route,
                        schedule_days: route.schedule_days || [],
                        checkpoint_ids: route.checkpoints ? route.checkpoints.map(c => c.id) : []
                    };
                    this.showModal = true;
                },
                resetForm() {
                    this.form = {
                        description: '',
                        assigned_to: 'all',
                        special_instructions: '',
                        estimated_duration_minutes: 0,
                        grace_period_minutes: 0,
                        recurrence_type: 'weekly',
                        schedule_days: [],
                        checkpoint_ids: []
                    };
                },
                async saveTourRoute() {
                    try {
                        const url = this.editingRoute
                            ? `/sites/${siteId}/tour-routes/${this.editingRoute.id}`
                            : `/sites/${siteId}/tour-routes`;
                        const method = this.editingRoute ? 'PUT' : 'POST';

                        const res = await fetch(url, {
                            method,
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                'Accept': 'application/json'
                            },
                            body: JSON.stringify(this.form)
                        });

                        if (res.ok) {
                            this.showModal = false;
                            this.fetchTourRoutes();
                            alert(this.editingRoute ? 'Tour route updated successfully' : 'Tour route created successfully');
                        } else {
                            const data = await res.json();
                            alert('Error: ' + (data.message || 'Failed to save tour route'));
                        }
                    } catch (e) {
                        console.error('Error saving tour route:', e);
                        alert('Error saving tour route');
                    }
                },
                async deleteTourRoute(id) {
                    if (!confirm('Are you sure you want to delete this tour route?')) return;

                    try {
                        const res = await fetch(`/sites/${siteId}/tour-routes/${id}`, {
                            method: 'DELETE',
                            headers: {
                                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                'Accept': 'application/json'
                            }
                        });

                        if (res.ok) {
                            this.fetchTourRoutes();
                            alert('Tour route deleted successfully');
                        }
                    } catch (e) {
                        console.error('Error deleting tour route:', e);
                        alert('Error deleting tour route');
                    }
                }
            };
        }
    </script>
@endsection