<!DOCTYPE html>
<html lang="{{ str_replace('_', '-', app()->getLocale()) }}" class="h-full bg-[#eef1f5]">

<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>@yield('title', 'Dashboard') - Shift Matrix</title>
    <meta name="csrf-token" content="{{ csrf_token() }}">
    @vite(['resources/css/app.css', 'resources/js/app.js'])
    @php
        $globalSettings = \App\Models\Setting::whereIn('key', ['font_size', 'font_family', 'font_weight'])->pluck('value', 'key');
        $fontFamily = $globalSettings['font_family'] ?? 'Inter';
        $fontSize = $globalSettings['font_size'] ?? '32px';
        $fontWeight = $globalSettings['font_weight'] ?? '700';
    @endphp
    <link
        href="https://fonts.googleapis.com/css2?family={{ str_replace(' ', '+', $fontFamily) }}:wght@{{ $fontWeight }}&display=swap"
        rel="stylesheet">
    <script defer src="https://unpkg.com/alpinejs@3.x.x/dist/cdn.min.js"></script>
    <style>
        body {
            font-family: 'Inter', sans-serif;
        }

        h1,
        h2,
        h3,
        h4,
        h5,
        h6 {
            font-family: '{{ $fontFamily }}', sans-serif !important;
            font-weight:
                {{ $fontWeight }}
                !important;
        }

        h1 {
            font-size:
                {{ $fontSize }}
                !important;
        }

        h2 {
            font-size: calc({{ $fontSize }} * 0.8) !important;
        }

        h3 {
            font-size: calc({{ $fontSize }} * 0.7) !important;
        }

        h4 {
            font-size: calc({{ $fontSize }} * 0.6) !important;
        }

        h5 {
            font-size: calc({{ $fontSize }} * 0.5) !important;
        }

        h6 {
            font-size: calc({{ $fontSize }} * 0.45) !important;
        }
    </style>
</head>

<body class="h-full flex flex-col">
    <!-- Top Header -->
    <header class="bg-[#1e293b] h-14 flex items-center justify-between px-4 flex-shrink-0">
        <div class="flex items-center gap-8">
            <a href="{{ route('dashboard') }}" class="flex items-center gap-2 py-2">
                <img src="{{ asset('images/logo.png') }}" alt="{{ config('app.name') }}"
                    class="h-16 w-auto brightness-0 invert object-contain scale-110">
            </a>
            @if(session('selected_company_id'))
                @php
                    $selectedCompany = \App\Models\Company::find(session('selected_company_id'));
                    $userCompanyCount = Auth::user()->companies()->where('companies.active', true)->count();
                @endphp
                @if($selectedCompany)
                    <div class="flex items-center ml-4 gap-2">
                        <span class="px-2 py-0.5 rounded text-xs font-semibold bg-blue-600 text-white border border-blue-500">
                            {{ $selectedCompany->name }}
                        </span>
                        @if($userCompanyCount > 1)
                            <a href="{{ route('select.company', ['manual' => 1]) }}"
                                class="text-[10px] text-gray-400 hover:text-white underline hover:no-underline transition-colors uppercase tracking-wider font-medium">Switch</a>
                        @endif
                    </div>
                @endif
            @endif
        </div>
        <!-- Search Bar -->
        <form action="{{ route('global.search') }}" method="GET"
            class="hidden md:flex items-center bg-[#334155] rounded px-3 py-1.5 w-80 shadow-inner border border-slate-600 focus-within:ring-2 focus-within:ring-blue-500/50 transition-all">
            <button type="submit" class="text-gray-400 hover:text-white transition-colors">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                </svg>
            </button>
            <input type="text" name="query" value="{{ request('query') }}"
                placeholder="Search employees, sites, & more..."
                class="bg-transparent border-none text-sm text-gray-200 focus:ring-0 w-full ml-2 placeholder-gray-500 font-medium h-full py-0">
        </form>
        </div>
        <div class="flex items-center gap-4 text-gray-300">
            <!-- Notification Bell -->
            <div class="relative" x-data="{ open: false }">
                <button @click="open = !open" class="hover:text-white relative p-1" title="Notifications">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9">
                        </path>
                    </svg>
                    @php
                        $unreadCount = \App\Models\Notification::where('user_id', Auth::id())->whereNull('read_at')->count();
                    @endphp
                    @if($unreadCount > 0)
                        <span
                            class="absolute -top-1 -right-1 inline-flex items-center justify-center px-1.5 py-0.5 text-[10px] font-bold leading-none text-white transform bg-red-600 rounded-full">
                            {{ $unreadCount > 99 ? '99+' : $unreadCount }}
                        </span>
                    @endif
                </button>

                <!-- Notification Dropdown -->
                <div x-show="open" @click.away="open = false" x-cloak
                    class="absolute right-0 mt-2 w-80 bg-white rounded-lg shadow-lg border border-gray-200 z-50 max-h-96 overflow-y-auto text-gray-800">
                    <div class="p-4 border-b border-gray-200 flex justify-between items-center">
                        <h3 class="text-sm font-bold text-gray-900">Notifications</h3>
                        <a href="{{ route('notifications.index') }}"
                            class="text-xs text-blue-600 hover:text-blue-800">View All</a>
                    </div>
                    @php
                        $notifications = \App\Models\Notification::where('user_id', Auth::id())
                            ->orderBy('created_at', 'desc')
                            ->limit(5)
                            ->get();
                    @endphp
                    @forelse($notifications as $notification)
                        <a href="{{ route('notifications.read', $notification->id) }}"
                            class="block p-4 hover:bg-gray-50 border-b border-gray-100 {{ is_null($notification->read_at) ? 'bg-blue-50' : '' }}">
                            <div class="flex items-start gap-3">
                                <div class="flex-shrink-0">
                                    <div
                                        class="w-2 h-2 bg-blue-600 rounded-full {{ is_null($notification->read_at) ? '' : 'opacity-0' }}">
                                    </div>
                                </div>
                                <div class="flex-1 min-w-0">
                                    <p class="text-sm font-semibold text-gray-900 truncate">{{ $notification->title }}</p>
                                    <p class="text-xs text-gray-600 mt-1 line-clamp-2">{{ $notification->message }}</p>
                                    <p class="text-xs text-gray-400 mt-1">{{ $notification->created_at->diffForHumans() }}
                                    </p>
                                </div>
                            </div>
                        </a>
                    @empty
                        <div class="p-8 text-center">
                            <p class="text-sm text-gray-400">No new notifications</p>
                        </div>
                    @endforelse
                    @if($notifications->count() > 0)
                        <div class="p-2 text-center border-t border-gray-200">
                            <a href="{{ route('notifications.index') }}"
                                class="text-xs font-semibold text-blue-600 hover:text-blue-700">
                                View History
                            </a>
                        </div>
                    @endif
                </div>
            </div>

            <!-- Broadcast Button -->
            <div x-data="{ open: false, target: 'all', search: '' }" class="relative">
                <button @click="open = true" class="hover:text-white" title="Broadcast Message">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M11 5.882V19.24a1.76 1.76 0 01-3.417.592l-2.147-6.15M18 13a3 3 0 100-6M5.436 13.683A4.001 4.001 0 017 6h1.832c4.1 0 7.625-1.234 9.168-3v14c-1.543-1.766-5.067-3-9.168-3H7a3.988 3.988 0 01-1.564-.317z">
                        </path>
                    </svg>
                </button>

                <!-- Broadcast Modal -->
                <div x-show="open" class="fixed inset-0 z-50 overflow-y-auto" style="display: none;">
                    <div
                        class="flex items-center justify-center min-h-screen px-4 pt-4 pb-20 text-center sm:block sm:p-0">
                        <div x-show="open" class="fixed inset-0 transition-opacity" @click="open = false">
                            <div class="absolute inset-0 bg-gray-500 opacity-75"></div>
                        </div>
                        <span class="hidden sm:inline-block sm:align-middle sm:h-screen">&#8203;</span>
                        <div x-show="open" @click.stop
                            class="relative z-50 inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg w-full">
                            <form action="{{ route('notifications.store') }}" method="POST" class="p-6">
                                @csrf
                                <div class="flex justify-between items-center mb-4">
                                    <h3 class="text-lg font-medium text-black">Send Notification</h3>
                                    <button type="button" @click="open = false"
                                        class="text-gray-400 hover:text-gray-500">
                                        <svg class="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                    </button>
                                </div>

                                <div class="space-y-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700">Send To</label>
                                        <select name="target_type" x-model="target"
                                            class="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md text-gray-900">
                                            <option value="all">All Employees</option>
                                            <option value="selected">Select Employees</option>
                                        </select>
                                    </div>

                                    <div x-show="target === 'selected'" class="space-y-2">
                                        <input type="text" x-model="search" placeholder="Search employees..."
                                            class="w-full px-3 py-1.5 text-xs border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500 text-gray-900">
                                        <div class="max-h-40 overflow-y-auto border border-gray-200 rounded-md p-2">
                                            @if(session('selected_company_id'))
                                                @foreach(\App\Models\Employee::where('company_id', session('selected_company_id'))->orderBy('first_name')->get() as $emp)
                                                    <div class="flex items-center py-0.5"
                                                        x-show="search === '' || '{{ strtolower($emp->employee_id . ' ' . $emp->first_name . ' ' . $emp->last_name) }}'.includes(search.toLowerCase())">
                                                        <input type="checkbox" name="employees[]" value="{{ $emp->id }}"
                                                            class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                                                        <label class="ml-2 block text-sm text-gray-900">
                                                            <span
                                                                class="font-bold text-blue-600">[{{ $emp->employee_id }}]</span>
                                                            {{ $emp->first_name }} {{ $emp->last_name }}
                                                        </label>
                                                    </div>
                                                @endforeach
                                            @endif
                                        </div>
                                    </div>

                                    <div>
                                        <label class="block text-sm font-medium text-gray-700">Title</label>
                                        <input type="text" name="title" required
                                            class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 sm:text-sm p-2 border text-gray-900">
                                    </div>

                                    <div>
                                        <label class="block text-sm font-medium text-gray-700">Message</label>
                                        <textarea name="message" rows="4" required
                                            class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 sm:text-sm p-2 border text-gray-900"></textarea>
                                    </div>

                                    <div>
                                        <label class="block text-sm font-medium text-gray-700">Redirection URL
                                            (Optional)</label>
                                        <input type="url" name="url" placeholder="https://..."
                                            class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 sm:text-sm p-2 border text-gray-900">
                                        <p class="text-[10px] text-gray-400 mt-1">If provided, clicking the notification
                                            will redirect the employee to this link.</p>
                                    </div>
                                </div>

                                <div class="mt-5 sm:mt-6 flex justify-end gap-2">
                                    <button type="button" @click="open = false"
                                        class="inline-flex justify-center px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md shadow-sm hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                        Cancel
                                    </button>
                                    <button type="submit"
                                        class="inline-flex justify-center px-4 py-2 text-sm font-medium text-white bg-blue-600 border border-transparent rounded-md shadow-sm hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                        Send
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>

            <a href="{{ route('user.help') }}" class="hover:text-white" title="Help Guide"><svg class="w-5 h-5"
                    fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z">
                    </path>
                </svg></a>
            <div class="flex items-center gap-2 border-l border-gray-600 pl-4 ml-2">
                <div
                    class="w-8 h-8 rounded-full bg-blue-500 flex items-center justify-center text-white text-xs font-bold">
                    {{ strtoupper(substr(Auth::user()->name, 0, 1)) }}
                </div>
                <span class="text-sm font-medium hidden sm:block">{{ Auth::user()->name }}</span>
                <form action="{{ route('logout') }}" method="POST" class="inline">
                    @csrf
                    <button type="submit" class="hover:text-white"><svg class="w-4 h-4" fill="none"
                            stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1">
                            </path>
                        </svg></button>
                </form>
            </div>
        </div>
    </header>

    <div class="flex flex-1 overflow-hidden">
        <!-- Slim Sidebar -->
        <aside class="w-12 bg-[#1e293b] flex flex-col items-center py-4 gap-6 flex-shrink-0">
            <a href="{{ route('dashboard') }}" class="sidebar-icon" title="Dashboard">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6">
                    </path>
                </svg>
            </a>
            @if(Auth::user()->email === 'thapar.advocate@gmail.com')
                <a href="{{ route('user.analytics') }}" class="sidebar-icon" title="Performance Analytics">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z">
                        </path>
                    </svg>
                </a>
            @endif
            @can('view employees')
                @if(Route::has('employees.index'))
                    <a href="{{ route('employees.index') }}" class="sidebar-icon" title="Employees">
                        <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z">
                            </path>
                        </svg>
                    </a>
                @endif
            @endcan
            @can('view sites')
                <a href="{{ route('sites.index') }}" class="sidebar-icon" title="Sites">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"></path>
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                    </svg>
                </a>
            @endcan
            <a href="{{ route('vehicles.index') }}" class="sidebar-icon" title="Vehicle Management">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M8 7v8a2 2 0 002 2h6M8 7V5a2 2 0 012-2h4.586a1 1 0 01.707.293l4.414 4.414a1 1 0 01.293.707V15a2 2 0 01-2 2h-2M8 7H6a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2v-2">
                    </path>
                </svg>
            </a>
            {{-- @can('view email templates') --}}
            <a href="{{ route('email-templates.index') }}" class="sidebar-icon" title="Emails">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z">
                    </path>
                </svg>
            </a>
            {{-- @endcan --}}
            <a href="{{ route('email-campaigns.index') }}" class="sidebar-icon" title="Campaigns">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M11 5.882V19.24a1.76 1.76 0 01-3.417.592l-2.147-6.15M18 13a3 3 0 100-6M5.436 13.683A4.001 4.001 0 017 6h1.832c4.1 0 7.625-1.234 9.168-3v14c-1.543-1.766-5.067-3-9.168-3H7a3.988 3.988 0 01-1.564-.317z">
                    </path>
                </svg>
            </a>
            <a href="{{ route('employee-requests.index') }}" class="sidebar-icon" title="Requests">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z">
                    </path>
                </svg>
            </a>

            @if(Auth::user()->role === 'user' || Auth::user()->can('view leads'))
                <a href="{{ route('leads.index') }}" class="sidebar-icon" title="Leads">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M13 10V3L4 14h7v7l9-11h-7z"></path>
                    </svg>
                </a>
            @endif
            @can('view quotations')
                <a href="{{ route('quotations.index') }}" class="sidebar-icon" title="Quotations">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z">
                        </path>
                    </svg>
                </a>
            @endcan

            <a href="{{ route('activity-logs.index') }}" class="sidebar-icon" title="Activity Logs">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                </svg>
            </a>
            <a href="{{ route('quotations.index') }}" class="sidebar-icon" title="Quotations">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z">
                    </path>
                </svg>
            </a>


            <a href="{{ route('profile.password') }}" class="sidebar-icon" title="Change Password">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z">
                    </path>
                </svg>
            </a>

            <a href="{{ route('analytics.index') }}" class="sidebar-icon" title="Analytics">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z">
                    </path>
                </svg>
            </a>


            <a href="{{ route('developer.settings') }}" class="sidebar-icon" title="Developer Settings"
                onclick="return confirm('Developer Access Only\nThese commands should only be executed by authorized developers. Use with caution as they can affect system performance and data.')">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M10 20l4-16m4 4l4 4-4 4M6 16l-4-4 4-4"></path>
                </svg>
            </a>
        </aside>

        <!-- Main Body -->
        <div class="flex-1 flex flex-col overflow-hidden">
            <!-- Sub Header / Tabs -->
            <div class="bg-white border-b border-gray-200 h-10 flex items-center px-4">
                <nav class="flex h-full">
                    <a href="{{ route('dashboard') }}"
                        class="px-4 flex items-center text-xs font-semibold {{ request()->routeIs('dashboard') ? 'text-gray-900 border-b-2 border-red-500 bg-gray-50' : 'text-gray-500 hover:text-gray-700 hover:bg-gray-50' }}">Live
                        Dashboard</a>
                    <div class="relative group h-full">
                        <button
                            class="px-4 h-full flex items-center gap-1 text-xs font-semibold text-gray-500 hover:text-gray-700 hover:bg-gray-50 uppercase">
                            Masters
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </button>
                        <div
                            class="absolute left-0 top-full w-48 bg-white border border-gray-200 shadow-lg py-1 z-50 hidden group-hover:block">
                            @can('view sites')
                                <a href="{{ route('sites.index') }}"
                                    class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Sites/Clients</a>
                            @endcan

                            @can('view routes')
                                <a href="{{ route('routes.index') }}"
                                    class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Route
                                    Masters</a>
                            @endcan

                            @can('view employees')
                                @if(Route::has('employees.index'))
                                    <a href="{{ route('employees.index') }}"
                                        class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Employees</a>
                                @endif
                            @endcan

                            @can('view quotations')
                                <a href="{{ route('quotations.index') }}"
                                    class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium border-t border-gray-100 mt-1">Quotations</a>
                            @endcan

                            <a href="{{ route('user.uniforms.index') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium border-t border-gray-100 mt-1">Uniforms
                                Inventory</a>
                        </div>
                    </div>

                    <div class="relative group h-full">
                        <a href="{{ route('leads.index') }}"
                            class="px-4 h-full flex items-center gap-1 text-xs font-semibold {{ request()->routeIs('leads.*') ? 'text-gray-900 border-b-2 border-red-500 bg-gray-50' : 'text-gray-500 hover:text-gray-700 hover:bg-gray-50' }} uppercase transition-colors">
                            Leads
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </a>
                        <div
                            class="absolute left-0 top-full w-48 bg-white border border-gray-200 shadow-lg py-1 z-50 hidden group-hover:block">
                            @if(Auth::user()->role === 'user' || Auth::user()->can('view leads'))
                                <a href="{{ route('leads.index') }}"
                                    class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Lead
                                    List</a>
                                <a href="{{ route('leads.index', ['status' => 'new']) }}"
                                    class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">New
                                    Leads</a>
                            @endif
                        </div>
                    </div>
                    <div class="relative group h-full">
                        <button
                            class="px-4 h-full flex items-center gap-1 text-xs font-semibold text-gray-500 hover:text-gray-700 hover:bg-gray-50 uppercase">
                            Schedules & Attendance
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </button>
                        <div
                            class="absolute left-0 top-full w-56 bg-white border border-gray-200 shadow-lg py-1 z-50 hidden group-hover:block">
                            @can('view schedules')
                                <a href="{{ route('schedules.index') }}"
                                    class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Guard
                                    Schedule</a>

                                <a href="{{ route('patroller-schedules.index') }}"
                                    class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Patroller
                                    Schedule</a>

                                <a href="{{ route('addon-schedules.index') }}"
                                    class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Addon
                                    Schedule</a>
                            @endcan
                        </div>
                    </div>
                    <div class="relative group h-full">
                        <button
                            class="px-4 h-full flex items-center gap-1 text-xs font-semibold {{ request()->routeIs('reports.operational.*') ? 'text-gray-900 border-b-2 border-red-500 bg-gray-50' : 'text-gray-500 hover:text-gray-700 hover:bg-gray-50' }} uppercase">
                            Operational Reports
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </button>
                        <div
                            class="absolute left-0 top-full w-56 bg-white border border-gray-200 shadow-lg py-1 z-50 hidden group-hover:block">
                            <a href="{{ route('reports.operational.index') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Shift
                                Summary</a>
                            <a href="{{ route('reports.operational.payouts') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Payout
                                Summary</a>
                            <a href="{{ route('reports.operational.guard_payouts') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Guard
                                Payout Summary</a>
                            <a href="{{ route('reports.operational.consolidated') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Consolidated
                                Report</a>
                            <a href="{{ route('reports.operational.site_payouts') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Site
                                Payout Summary</a>
                            <a href="{{ route('reports.operational.site_margin') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Site
                                Margin Report</a>
                            <a href="{{ route('reports.operational.late_checkin') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Late
                                Check-in Report</a>
                            <a href="{{ route('reports.operational.banned') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">
                                Banned Employees</a>
                            <a href="{{ route('reports.operational.passwords') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">
                                Employee Credentials</a>
                            <a href="{{ route('job-notes.index') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium border-t border-gray-100">
                                Job Mission Notes</a>
                        </div>
                    </div>
                    <div class="relative group h-full">
                        <button
                            class="px-4 h-full flex items-center gap-1 text-xs font-semibold {{ request()->routeIs('admin.reports.patroller.*') ? 'text-gray-900 border-b-2 border-red-500 bg-gray-50' : 'text-gray-500 hover:text-gray-700 hover:bg-gray-50' }} uppercase">
                            Patroller Reports
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </button>
                        <div
                            class="absolute left-0 top-full w-56 bg-white border border-gray-200 shadow-lg py-1 z-50 hidden group-hover:block">
                            <a href="{{ route('reports.patroller.index') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">In/Out
                                Report</a>
                            <a href="{{ route('reports.patroller.tickets') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Ticket
                                Reports</a>
                            <a href="{{ route('reports.patroller.monthly_tickets') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Monthly
                                Ticket Chart</a>
                            <a href="{{ route('reports.patroller.payout') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Patroller
                                Payout</a>
                            <a href="{{ route('reports.patroller.route_payout') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Route
                                Payout</a>
                            <a href="{{ route('reports.patroller.route_margin') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Route
                                Margin</a>
                        </div>
                    </div>
                    <a href="{{ route('attendance.chart') }}"
                        class="px-4 h-full flex items-center text-xs font-semibold {{ request()->routeIs('attendance.chart') ? 'text-gray-900 border-b-2 border-red-500 bg-gray-50' : 'text-gray-500 hover:text-gray-700 hover:bg-gray-50' }} uppercase">
                        Attendance Chart
                    </a>
                    <a href="{{ route('care.active-guards') }}"
                        class="px-4 h-full flex items-center text-xs font-semibold {{ request()->routeIs('care.*') ? 'text-gray-900 border-b-2 border-red-500 bg-gray-50' : 'text-gray-500 hover:text-gray-700 hover:bg-gray-50' }} uppercase">
                        Customer Care
                    </a>
                    <div class="relative group h-full">
                        <button
                            class="px-4 h-full flex items-center gap-1 text-xs font-semibold {{ request()->routeIs('reports.operational.incidents*') ? 'text-gray-900 border-b-2 border-red-500 bg-gray-50' : 'text-gray-500 hover:text-gray-700 hover:bg-gray-50' }} uppercase">
                            Incident Reports
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M19 9l-7 7-7-7"></path>
                            </svg>
                        </button>
                        <div
                            class="absolute left-0 top-full w-56 bg-white border border-gray-200 shadow-lg py-1 z-50 hidden group-hover:block">
                            <a href="{{ route('reports.operational.incidents') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Report
                                Summary</a>
                            <a href="{{ route('reports.operational.incident_counts') }}"
                                class="block px-4 py-2 text-[11px] text-gray-700 hover:bg-gray-50 hover:text-blue-600 font-medium">Incident
                                Counts</a>
                        </div>
                    </div>
                </nav>
            </div>

            <!-- Content Area -->
            <main class="flex-1 overflow-y-auto p-4">
                @if (session('success'))
                    <div x-data="{ show: true }" x-show="show" x-init="setTimeout(() => show = false, 5000)"
                        class="mb-4 flex items-center p-4 text-sm text-green-800 border border-green-300 rounded-lg bg-green-50 dark:bg-gray-800 dark:text-green-400 dark:border-green-800"
                        role="alert">
                        <svg class="flex-shrink-0 inline w-4 h-4 me-3" aria-hidden="true" xmlns="http://www.w3.org/2000/svg"
                            fill="currentColor" viewBox="0 0 20 20">
                            <path
                                d="M10 .5a9.5 9.5 0 1 0 9.5 9.5A9.51 9.51 0 0 0 10 .5ZM9.5 4a1.5 1.5 0 1 1 0 3 1.5 1.5 0 0 1 0-3ZM12 15H8a1 1 0 0 1 0-2h1v-3H8a1 1 0 0 1 0-2h2a1 1 0 0 1 1 1v4h1a1 1 0 0 1 0 2Z" />
                        </svg>
                        <span class="sr-only">Success</span>
                        <div>
                            <span class="font-medium">Success!</span> {{ session('success') }}
                        </div>
                        <button type="button" @click="show = false"
                            class="ms-auto -mx-1.5 -my-1.5 bg-green-50 text-green-500 rounded-lg focus:ring-2 focus:ring-green-400 p-1.5 hover:bg-green-200 inline-flex items-center justify-center h-8 w-8 dark:bg-gray-800 dark:text-green-400 dark:hover:bg-gray-700">
                            <span class="sr-only">Close</span>
                            <svg class="w-3 h-3" aria-hidden="true" xmlns="http://www.w3.org/2000/svg" fill="none"
                                viewBox="0 0 14 14">
                                <path stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="m1 1 6 6m0 0 6 6M7 7l6-6M7 7l-6 6" />
                            </svg>
                        </button>
                    </div>
                @endif

                @if (session('error'))
                    <div x-data="{ show: true }" x-show="show" x-init="setTimeout(() => show = false, 5000)"
                        class="mb-4 flex items-center p-4 text-sm text-red-800 border border-red-300 rounded-lg bg-red-50 dark:bg-gray-800 dark:text-red-400 dark:border-red-800"
                        role="alert">
                        <svg class="flex-shrink-0 inline w-4 h-4 me-3" aria-hidden="true" xmlns="http://www.w3.org/2000/svg"
                            fill="currentColor" viewBox="0 0 20 20">
                            <path
                                d="M10 .5a9.5 9.5 0 1 0 9.5 9.5A9.51 9.51 0 0 0 10 .5ZM9.5 4a1.5 1.5 0 1 1 0 3 1.5 1.5 0 0 1 0-3ZM12 15H8a1 1 0 0 1 0-2h1v-3H8a1 1 0 0 1 0-2h2a1 1 0 0 1 1 1v4h1a1 1 0 0 1 0 2Z" />
                        </svg>
                        <span class="sr-only">Error</span>
                        <div>
                            <span class="font-medium">Error!</span> {{ session('error') }}
                        </div>
                        <button type="button" @click="show = false"
                            class="ms-auto -mx-1.5 -my-1.5 bg-red-50 text-red-500 rounded-lg focus:ring-2 focus:ring-red-400 p-1.5 hover:bg-red-200 inline-flex items-center justify-center h-8 w-8 dark:bg-gray-800 dark:text-red-400 dark:hover:bg-gray-700">
                            <span class="sr-only">Close</span>
                            <svg class="w-3 h-3" aria-hidden="true" xmlns="http://www.w3.org/2000/svg" fill="none"
                                viewBox="0 0 14 14">
                                <path stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="m1 1 6 6m0 0 6 6M7 7l6-6M7 7l-6 6" />
                            </svg>
                        </button>
                    </div>
                @endif
                @yield('content')
            </main>
        </div>
    </div>
    <!-- Global Page Loader -->
    <div id="page-loader"
        class="fixed inset-0 bg-white/90 z-[9999] hidden flex-col items-center justify-center backdrop-blur-md transition-all duration-300">
        <div class="relative w-16 h-16">
            <div class="absolute inset-0 border-4 border-blue-100 rounded-full"></div>
            <div
                class="absolute inset-0 border-4 border-blue-600 rounded-full border-t-transparent animate-spin shadow-lg">
            </div>
        </div>
        <div class="mt-6 text-gray-500 font-bold text-xs tracking-[0.2em] animate-pulse">
            LOADING
        </div>
    </div>

    <script>
        document.addEventListener('alpine:init', () => {
            Alpine.data('searchableSelect', (config) => ({
                options: config.options,
                value: config.value,
                placeholder: config.placeholder,
                search: '',
                open: false,

                get filteredOptions() {
                    if (this.search === '') return this.options;
                    return this.options.filter(option =>
                        option.label.toLowerCase().includes(this.search.toLowerCase())
                    );
                },

                get selectedLabel() {
                    if (!this.value) return this.placeholder;
                    // Loose equality for string/int match
                    const selected = this.options.find(o => o.value == this.value);
                    return selected ? selected.label : this.placeholder;
                },

                select(option) {
                    this.value = option.value;
                    this.open = false;
                    this.search = '';
                }
            }));
        });
    </script>

    <script>
        document.addEventListener('DOMContentLoaded', () => {
            const loader = document.getElementById('page-loader');
            const showLoader = () => {
                loader.classList.remove('hidden');
                loader.classList.add('flex');
            };
            const hideLoader = () => {
                loader.classList.add('hidden');
                loader.classList.remove('flex');
            };

            // Attach to links via delegation
            document.addEventListener('click', (e) => {
                const link = e.target.closest('a');
                if (link) {
                    const href = link.getAttribute('href');
                    const target = link.getAttribute('target');

                    if (href &&
                        !href.startsWith('#') &&
                        !href.startsWith('javascript:') &&
                        target !== '_blank' &&
                        !e.ctrlKey && !e.metaKey && !e.shiftKey &&
                        !link.hasAttribute('download') &&
                        !href.includes('export') &&
                        !href.includes('/pdf') &&
                        !href.toLowerCase().includes('.pdf') &&
                        !href.includes('/zip')
                    ) {
                        // For PDF/Excel exports, the browser stays on the page, so loader would hang forever.
                        if (href.includes('export') || href.includes('/pdf') || href.toLowerCase().includes('.pdf')) return;

                        showLoader();
                    }
                }
            });

            // Attach to forms
            document.addEventListener('submit', (e) => {
                if (!e.defaultPrevented) {
                    // Check if form is an export form? (e.g. guard payouts export is just a link, but if it were a form?)
                    // Usually forms redirect. 
                    showLoader();
                }
            });

            // Hide on back/forward cache restoration
            window.addEventListener('pageshow', (event) => {
                hideLoader();
            });
        });
    </script>

    @stack('scripts')
</body>

</html>