@php
    $notesEnabled = \App\Models\Setting::where('key', 'notes_enabled')->value('value') ?? 0;
    $notesInterval = \App\Models\Setting::where('key', 'notes_interval')->value('value') ?? 2;
    $employee = auth('employee')->user();
    $activeJob = null;
    $jobType = null;

    if ($employee) {
        // Check Guard Job
        $activeGuardJob = $employee->schedules()->where('schedules.status', 'active')->first();
        if ($activeGuardJob) {
            $activeJob = $activeGuardJob;
            $jobType = 'guard';
        } else {
            // Check Patroller Job
            $activePatrollerJob = $employee->patrollerSchedules()->where('patroller_schedules.job_status', 'in_progress')->first();
            if ($activePatrollerJob) {
                $activeJob = $activePatrollerJob;
                $jobType = 'patroller';
            }
        }
    }
@endphp

@if ($notesEnabled && $activeJob)
    <div x-data="jobNotesPopup()" x-init="init()"
        class="fixed inset-0 z-[9999] flex items-center justify-center p-4 bg-slate-900/60 backdrop-blur-sm" x-show="show"
        x-cloak @keydown.escape.window="closePopup()" @open-job-notes.window="openPopup()">

        <div class="relative bg-white rounded-2xl max-w-md w-full p-8 shadow-2xl border border-slate-100"
            @click.away="closePopup()">
            <div class="flex items-center gap-4 mb-6">
                <div class="w-12 h-12 bg-blue-100 rounded-xl flex items-center justify-center text-blue-600">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z">
                        </path>
                    </svg>
                </div>
                <div>
                    <h3 class="text-xl font-bold text-slate-900">Add Mission Note</h3>
                    <div class="flex items-center gap-2">
                        <p class="text-xs text-slate-500 font-medium">Regular protocol check-in for Duty
                            #{{ $activeJob->duty_number }}</p>
                        <button type="button" @click="togglePrevious()"
                            class="text-[10px] font-bold text-blue-600 uppercase tracking-tight hover:underline">
                            <span x-show="!showPrevious">View History</span>
                            <span x-show="showPrevious">Back to Form</span>
                        </button>
                    </div>
                </div>
            </div>

            <div x-show="!showPrevious">
                <form @submit.prevent="submitNote()">
                    <div class="mb-6">
                        <label class="block text-[10px] font-black text-slate-400 uppercase tracking-widest mb-2">Note
                            Content</label>
                        <textarea x-model="note" required
                            class="w-full px-4 py-3 bg-slate-50 border border-slate-200 rounded-xl text-sm font-medium focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all placeholder:text-slate-400"
                            rows="4" placeholder="Describe current status, observations, or any issues..."></textarea>
                    </div>

                    <div class="mb-6">
                        <label class="block text-[10px] font-black text-slate-400 uppercase tracking-widest mb-2">Evidence
                            Photo (Optional)</label>
                        <input type="file" @change="imageFile = $event.target.files[0]" accept="image/*"
                            class="w-full text-xs text-slate-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-xs file:font-bold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100 transition-all" />
                    </div>

                    <div class="flex gap-3">
                        <button type="button" @click="closePopup()"
                            class="flex-1 px-4 py-3 bg-slate-100 text-slate-600 rounded-xl text-xs font-bold uppercase tracking-wider hover:bg-slate-200 transition-all">
                            Skip
                        </button>
                        <button type="submit" :disabled="loading"
                            class="flex-[2] px-4 py-3 bg-blue-600 text-white rounded-xl text-xs font-bold uppercase tracking-wider hover:bg-blue-700 transition-all shadow-lg shadow-blue-500/30 flex items-center justify-center gap-2">
                            <template x-if="!loading">
                                <span>Submit Note</span>
                            </template>
                            <template x-if="loading">
                                <svg class="animate-spin h-4 w-4 text-white" fill="none" viewBox="0 0 24 24">
                                    <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor"
                                        stroke-width="4">
                                    </circle>
                                    <path class="opacity-75" fill="currentColor"
                                        d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z">
                                    </path>
                                </svg>
                            </template>
                        </button>
                    </div>
                </form>
            </div>

            <!-- Previous Notes View -->
            <div x-show="showPrevious" class="space-y-4 max-h-[400px] overflow-y-auto pr-2 custom-scrollbar">
                <template x-if="loadingNotes">
                    <div class="flex flex-col items-center justify-center py-10 opacity-50">
                        <svg class="animate-spin h-6 w-6 text-blue-600 mb-3" fill="none" viewBox="0 0 24 24">
                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4">
                            </circle>
                            <path class="opacity-75" fill="currentColor"
                                d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z">
                            </path>
                        </svg>
                        <p class="text-xs font-bold text-slate-400 uppercase tracking-widest">Decrypting Logs...</p>
                    </div>
                </template>

                <template x-if="!loadingNotes && previousNotes.length === 0">
                    <div class="text-center py-10">
                        <p class="text-xs font-bold text-slate-400 uppercase tracking-widest">No Previous Records</p>
                    </div>
                </template>

                <template x-for="note in previousNotes" :key="note.id">
                    <div class="bg-slate-50 rounded-xl p-4 border border-slate-100">
                        <div class="flex items-center justify-between mb-2">
                            <span class="text-[10px] font-black text-blue-600 uppercase tracking-widest"
                                x-text="formatDate(note.created_at)"></span>
                        </div>
                        <p class="text-xs text-slate-700 font-medium leading-relaxed" x-text="note.note"></p>
                        <template x-if="note.image">
                            <div class="mt-3">
                                <a :href="'/storage/' + note.image" target="_blank">
                                    <img :src="'/storage/' + note.image"
                                        class="w-20 h-20 object-cover rounded-lg border border-slate-200 shadow-sm hover:scale-105 transition-transform" />
                                </a>
                            </div>
                        </template>
                    </div>
                </template>
            </div>
        </div>
    </div>

    <script>
        function jobNotesPopup() {
            return {
                show: false,
                note: '',
                imageFile: null,
                loading: false,
                loadingNotes: false,
                showPrevious: false,
                previousNotes: [],
                interval: {{ $notesInterval }} * 60 * 1000,
                jobId: {{ $activeJob->id }},
                jobType: '{{ $jobType }}',
                timerId: null,

                init() {
                    // Set initial timer
                    this.startTimer();

                    // Request notification permission
                    if ('Notification' in window && Notification.permission === 'default') {
                        Notification.requestPermission();
                    }
                },

                startTimer() {
                    if (this.timerId) clearTimeout(this.timerId);
                    this.timerId = setTimeout(() => {
                        this.show = true;
                        this.sendReminderNotify();
                    }, this.interval);
                },

                async sendReminderNotify() {
                    // Play sound alert
                    try {
                        let audio = new Audio('https://assets.mixkit.co/active_storage/sfx/2869/2869-preview.mp3');
                        audio.play();
                    } catch (e) {
                        console.log('Audio play blocked');
                    }

                    // 1. Browser Notification
                    if ('Notification' in window && Notification.permission === 'granted') {
                        new Notification('Mission Note Reminder', {
                            body: 'Please submit your mission note for Duty #{{ $activeJob->duty_number }}.',
                            icon: '/favicon.ico'
                        });
                    }

                    // 2. Database Notification via AJAX
                    try {
                        await fetch('/employee/notifications/reminder', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': '{{ csrf_token() }}'
                            },
                            body: JSON.stringify({
                                job_id: this.jobId,
                                job_type: this.jobType,
                                duty_number: '{{ $activeJob->duty_number }}'
                            })
                        });
                    } catch (e) {
                        console.error('Failed to send DB reminder:', e);
                    }
                },

                openPopup() {
                    if (this.timerId) clearTimeout(this.timerId);
                    this.show = true;
                },

                closePopup() {
                    if (this.timerId) clearTimeout(this.timerId);
                    this.show = false;
                    this.note = '';
                    this.imageFile = null;
                    if (document.querySelector('input[type="file"]')) {
                        document.querySelector('input[type="file"]').value = '';
                    }
                    this.showPrevious = false;
                    this.startTimer();
                },

                togglePrevious() {
                    this.showPrevious = !this.showPrevious;
                    if (this.showPrevious) {
                        this.fetchNotes();
                    }
                },

                async fetchNotes() {
                    this.loadingNotes = true;
                    try {
                        let url = this.jobType === 'guard'
                            ? `/employee/jobs/${this.jobId}/notes`
                            : `/employee/patroller-jobs/${this.jobId}/notes`;

                        const response = await fetch(url);
                        const data = await response.json();
                        if (data.success) {
                            this.previousNotes = data.notes;
                        }
                    } catch (e) {
                        console.error('Failed to fetch notes:', e);
                    } finally {
                        this.loadingNotes = false;
                    }
                },

                formatDate(dateStr) {
                    const date = new Date(dateStr);
                    return date.toLocaleString('en-US', {
                        month: 'short',
                        day: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit',
                        hour12: true
                    });
                },

                async submitNote() {
                    if (!this.note.trim()) return;

                    this.loading = true;
                    try {
                        let url = this.jobType === 'guard'
                            ? `/employee/jobs/${this.jobId}/note`
                            : `/employee/patroller-jobs/${this.jobId}/note`;

                        const formData = new FormData();
                        formData.append('note', this.note);
                        formData.append('job_type', this.jobType);
                        if (this.imageFile) {
                            formData.append('image', this.imageFile);
                        }

                        const response = await fetch(url, {
                            method: 'POST',
                            headers: {
                                'X-CSRF-TOKEN': '{{ csrf_token() }}'
                            },
                            body: formData
                        });

                        const data = await response.json();
                        if (data.success) {
                            this.closePopup();
                        } else {
                            alert(data.message || 'Failed to save note.');
                        }
                    } catch (error) {
                        console.error('Error saving note:', error);
                        alert('A communication error occurred.');
                    } finally {
                        this.loading = false;
                    }
                }
            }
        }
    </script>
@endif