@extends('layouts.employee')

@section('title', 'Patroller Job')

@section('content')
    <div class="p-6" x-data="patrollerJob()">
        <div class="mb-6 flex justify-between items-end">
            <div>
                <h1 class="text-2xl font-bold text-gray-900">[PATROLLER] {{ $schedule->route->name }}</h1>
                <p class="text-gray-600">Duty #{{ $schedule->duty_number }}</p>
                <p class="text-sm text-gray-500">{{ $schedule->from_time->format('h:i A') }} -
                    {{ $schedule->to_time->format('h:i A') }}
                </p>
            </div>
            <a href="{{ route('employee.patroller-jobs.report', $schedule->id) }}"
                class="bg-purple-100 text-purple-700 px-4 py-2 rounded-lg font-bold text-sm hover:bg-purple-200 transition-colors flex items-center gap-2">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M9 17v-2m3 2v-4m3 4v-6m2 10H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 011.414.586l4 4a1 1 0 01.586 1.414V19a2 2 0 01-2 2z">
                    </path>
                </svg>
                View Report
            </a>
        </div>

        {{-- Job Stats --}}
        @php
            $totalTickets = $schedule->jobSites->sum(fn($js) => $js->issueTickets->count());
            $uniqueVisitedSites = $schedule->jobSites->where('status', '!=', 'pending')->unique('site_id')->count();
            $totalSites = $schedule->route->sites->count();
        @endphp
        <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
            <div class="bg-white p-4 rounded-lg border border-gray-200">
                <p class="text-xs text-gray-500 uppercase font-bold">Progress</p>
                <p class="text-2xl font-bold text-purple-600">{{ $uniqueVisitedSites }} / {{ $totalSites }} <span
                        class="text-sm text-gray-400 font-normal">Sites</span></p>
            </div>
            <div class="bg-white p-4 rounded-lg border border-gray-200">
                <p class="text-xs text-gray-500 uppercase font-bold">Total Issues</p>
                <p class="text-2xl font-bold text-red-600">{{ $totalTickets }} <span
                        class="text-sm text-gray-400 font-normal">Tickets</span></p>
            </div>
            <div class="bg-white p-4 rounded-lg border border-gray-200">
                <p class="text-xs text-gray-500 uppercase font-bold">Gas Allowance</p>
                <p class="text-2xl font-bold text-green-600">${{ $schedule->gas ?? '0.00' }}</p>
            </div>
            <div class="bg-white p-4 rounded-lg border border-gray-200">
                <p class="text-xs text-gray-500 uppercase font-bold">Reference</p>
                <p class="text-sm font-bold text-gray-800 break-words">{{ $schedule->customer_reference ?? 'N/A' }}</p>
            </div>
            @if($schedule->comments)
                <div class="col-span-2 md:col-span-4 bg-white p-4 rounded-lg border border-gray-200">
                    <p class="text-xs text-gray-500 uppercase font-bold mb-1">Comments</p>
                    <p class="text-sm text-gray-700 italic">"{{ $schedule->comments }}"</p>
                </div>
            @endif
        </div>

        {{-- Patroller Team Status --}}
        <div class="mb-6">
            <h3 class="text-lg font-bold text-gray-900 mb-3 px-1">Team Status</h3>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
                @foreach($schedule->employees as $emp)
                    @php
                        $isFinished = $emp->pivot->actual_end_at !== null;
                        $isMe = $emp->id === auth('employee')->id();
                    @endphp
                    <div
                        class="bg-white p-3 rounded-lg border {{ $isFinished ? 'border-emerald-200 bg-emerald-50' : 'border-blue-200 bg-blue-50' }} flex items-center justify-between">
                        <div class="flex items-center gap-3">
                            <div
                                class="w-8 h-8 rounded-full border {{ $isFinished ? 'border-emerald-300 bg-emerald-100 text-emerald-700' : 'border-blue-300 bg-blue-100 text-blue-700' }} flex items-center justify-center font-bold text-xs">
                                {{ substr($emp->first_name, 0, 1) }}
                            </div>
                            <div>
                                <p class="text-sm font-bold text-gray-900 flex items-center gap-2">
                                    {{ $emp->first_name }} {{ $emp->last_name }}
                                    @if($isMe)
                                        <span
                                            class="text-[10px] bg-slate-100 text-slate-600 px-1.5 rounded uppercase tracking-wide">You</span>
                                    @endif
                                </p>
                                <p class="text-xs {{ $isFinished ? 'text-emerald-600' : 'text-blue-600' }}">
                                    {{ $isFinished ? 'Completed at ' . \Carbon\Carbon::parse($emp->pivot->actual_end_at)->format('h:i A') : 'Active on Patrol' }}
                                </p>
                            </div>
                        </div>
                        <div>
                            @if($isFinished)
                                <svg class="w-5 h-5 text-emerald-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                </svg>
                            @else
                                <div class="w-2.5 h-2.5 rounded-full bg-blue-500 animate-pulse"></div>
                            @endif
                        </div>
                    </div>
                @endforeach
            </div>
        </div>

        {{-- Timeline --}}
        @php
            $timelineEvents = $schedule->jobSites
                ->where('status', '!=', 'pending')
                ->sortBy('checked_in_at');

            // Check if current user has ended shift
            $currentUserPivot = $schedule->employees->find(auth('employee')->id())->pivot;
            $hasEndedShift = $currentUserPivot->actual_end_at !== null;
        @endphp

        @if($timelineEvents->isNotEmpty())
            <div class="mb-8">
                <h3 class="text-lg font-bold text-gray-900 mb-4 px-1">Patrol Timeline</h3>
                <div class="ml-2 pl-6 border-l-2 border-purple-200 space-y-6 py-2">
                    @foreach($timelineEvents as $event)
                        <div class="relative">
                            <div
                                class="absolute -left-[31px] top-1 w-4 h-4 rounded-full border-2 border-white shadow-sm
                                                                                                                                                                                                {{ $event->status === 'checked_in' ? 'bg-purple-600 ring-4 ring-purple-100' : 'bg-green-500' }}">
                            </div>

                            <div class="bg-white p-3 rounded-lg border border-gray-100 shadow-sm relative top-[-10px]">
                                <div class="flex justify-between items-start mb-1">
                                    <span class="text-xs font-bold text-purple-600">
                                        {{ $event->checked_in_at ? $event->checked_in_at->format('h:i A') : '' }}
                                    </span>
                                    @if($event->status === 'checked_in')
                                        <span
                                            class="text-[10px] bg-purple-100 text-purple-700 px-2 py-0.5 rounded-full font-bold uppercase">Active</span>
                                    @else
                                        <span class="text-[10px] text-gray-400">
                                            {{ $event->checked_out_at ? $event->checked_out_at->format('h:i A') : '' }}
                                        </span>
                                    @endif
                                </div>

                                <p class="font-bold text-gray-800 text-sm">{{ $event->site->name }}</p>

                                @if($event->status === 'checked_out' && $event->checked_in_at && $event->checked_out_at)
                                    <p class="text-xs text-gray-500 mt-1">
                                        Duration: {{ $event->checked_in_at->diffInMinutes($event->checked_out_at) }} mins
                                    </p>
                                @endif

                                @if($event->issueTickets->count() > 0)
                                    <div class="mt-2 flex items-center gap-1">
                                        <span class="text-red-500">
                                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                                </path>
                                            </svg>
                                        </span>
                                        <span class="text-xs font-bold text-red-600">{{ $event->issueTickets->count() }} Issues</span>
                                    </div>
                                @endif
                            </div>
                        </div>
                    @endforeach
                </div>
            </div>
        @endif

        {{-- Sites List --}}
        <div class="space-y-4">
            @foreach($schedule->route->sites as $site)
                @php
                    // Get all visits for this site, ordered by latest first
                    $visits = $schedule->jobSites->where('site_id', $site->id)->sortByDesc('created_at');
                    $currentVisit = $visits->first();

                    $status = $currentVisit ? $currentVisit->status : 'pending';

                    $canCheckIn = ($status === 'pending' || $status === 'checked_out');
                    $isCheckedIn = ($status === 'checked_in');
                    $hasPriorImages = $visits->whereNotNull('checkin_images')->isNotEmpty();
                @endphp

                <div class="bg-white border border-gray-200 rounded-lg p-6">
                    <div class="flex justify-between items-center mb-4 gap-4">
                        <div class="min-w-0">
                            <h3 class="text-lg font-bold text-gray-900 truncate">{{ $site->name }}</h3>
                            <p class="text-sm text-gray-600 truncate">{{ $site->address_line_1 }}</p>
                        </div>

                        <div class="flex-shrink-0">
                            @if($canCheckIn)
                                @if(isset($activeCheckIn) && $activeCheckIn->site_id !== $site->id)
                                    <div class="flex flex-col items-end">
                                        <button disabled
                                            class="bg-gray-100 text-gray-400 font-bold py-2 px-4 rounded-lg flex items-center gap-2 text-xs uppercase tracking-wider shadow-sm border border-gray-200 whitespace-nowrap cursor-not-allowed">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z">
                                                </path>
                                            </svg>
                                            Check In Locked
                                        </button>
                                        <p class="text-[10px] text-red-500 font-bold mt-1 text-right">
                                            Checkout from<br>
                                            <span class="text-gray-900">{{ Str::limit($activeCheckIn->site->name, 20) }}</span> first
                                        </p>
                                    </div>
                                @else
                                    @if($schedule->job_status === 'in_progress')
                                        <button @click="openCheckinModal({{ $site->id }}, '{{ $site->name }}')"
                                            class="bg-purple-600 text-white font-bold py-2 px-4 rounded-lg hover:bg-purple-700 flex items-center gap-2 text-xs uppercase tracking-wider shadow-sm transition-all whitespace-nowrap">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z">
                                                </path>
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                            </svg>
                                            {{ $status === 'checked_out' ? 'Check In Again' : 'Check In' }}
                                        </button>
                                    @else
                                        <span
                                            class="px-3 py-1 bg-gray-100 text-gray-500 rounded-lg text-xs font-bold uppercase tracking-wider border border-gray-200">
                                            Job Not Started
                                        </span>
                                    @endif
                                @endif
                            @else
                                <span
                                    class="px-3 py-1 rounded-full text-xs font-bold uppercase
                                                                                                                                                                                            {{ $status === 'checked_out' ? 'bg-green-100 text-green-700' : '' }}
                                                                                                                                                                                            {{ $status === 'checked_in' ? 'bg-blue-100 text-blue-700' : '' }}
                                                                                                                                                                                            {{ $status === 'pending' ? 'bg-gray-100 text-gray-700' : '' }}">
                                    {{ $status === 'checked_out' ? 'Visited' : $status }}
                                </span>
                            @endif
                        </div>
                    </div>

                    @if($isCheckedIn)
                        <div class="space-y-3">
                            <div class="bg-blue-50 p-4 rounded-lg flex justify-between items-center">
                                <div>
                                    <p class="text-sm font-bold text-blue-800">Currently Checked In</p>
                                    <p class="text-xs text-blue-600">Since {{ $currentVisit->checked_in_at->format('h:i A') }}</p>
                                </div>
                            </div>

                            <div class="flex gap-2">
                                <button @click="openTicketModal({{ $site->id }})"
                                    class="flex-1 bg-red-600 text-white font-bold py-3 rounded-lg hover:bg-red-700">
                                    Issue Tickets
                                </button>
                                <button @click="checkout({{ $site->id }})"
                                    class="flex-1 bg-green-600 text-white font-bold py-3 rounded-lg hover:bg-green-700">
                                    Check Out
                                </button>
                            </div>
                        </div>
                    @endif

                    {{-- Visit History & Tickets --}}
                    @if($visits->count() > 0)
                        <div class="mt-6 pt-4 border-t border-gray-100">
                            <p class="text-xs font-bold text-gray-500 uppercase mb-3 text-center">Activity Log</p>
                            <div class="space-y-3">
                                @foreach($visits as $visit)
                                    @if($visit->status !== 'pending')
                                        <div class="bg-gray-50 rounded p-3 text-sm">
                                            <div class="flex justify-between text-gray-700 mb-2">
                                                <div>
                                                    <p class="text-xs text-gray-500">Check In</p>
                                                    <p class="font-bold">
                                                        {{ $visit->checked_in_at ? $visit->checked_in_at->format('h:i A') : '-' }}
                                                    </p>
                                                </div>
                                                <div class="text-right">
                                                    <p class="text-xs text-gray-500">Check Out</p>
                                                    <p class="font-bold">
                                                        {{ $visit->checked_out_at ? $visit->checked_out_at->format('h:i A') : 'Active' }}
                                                    </p>
                                                </div>
                                            </div>

                                            {{-- Tickets for this visit --}}
                                            @if($visit->issueTickets->count() > 0)
                                                <div class="mt-2 pt-2 border-t border-gray-200">
                                                    <p class="text-xs font-bold text-red-600 mb-2">{{ $visit->issueTickets->count() }} Issue
                                                        Ticket(s)</p>
                                                    <div class="space-y-2 pl-2 border-l-2 border-red-200">
                                                        @foreach($visit->issueTickets as $ticket)
                                                            <div>
                                                                <div class="flex items-center gap-2">
                                                                    <span
                                                                        class="text-xs font-mono bg-white border border-gray-200 px-1 rounded">{{ $ticket->ticket_number }}</span>
                                                                    <span
                                                                        class="text-xs {{ $ticket->status === 'bad' ? 'text-red-600 font-bold' : 'text-green-600' }}">
                                                                        {{ strtoupper($ticket->status) }}
                                                                    </span>
                                                                    @if(!empty($ticket->images))
                                                                        <span class="text-purple-600" title="Has images">
                                                                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                                                    d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z">
                                                                                </path>
                                                                            </svg>
                                                                        </span>
                                                                    @endif
                                                                </div>
                                                                <p class="text-xs text-gray-600 truncate">{{ $ticket->description }}</p>
                                                            </div>
                                                        @endforeach
                                                    </div>
                                                </div>
                                            @else
                                                <div class="mt-1">
                                                    <span class="text-xs text-green-600 flex items-center gap-1">
                                                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                                d="M5 13l4 4L19 7"></path>
                                                        </svg>
                                                        No issues reported
                                                    </span>
                                                </div>
                                            @endif
                                        </div>
                                    @endif
                                @endforeach
                            </div>
                        </div>
                    @endif
                </div>
            @endforeach
        </div>

        {{-- End Job Button --}}
        {{-- Job Actions --}}
        <div class="mt-8 mb-12">
            @if($schedule->job_status === 'pending')
                @php
                    $startTime = $schedule->from_time->copy()->subMinutes($visibilityMinutes ?? 0);
                    $canStart = now()->greaterThanOrEqualTo($startTime);
                    $isMissed = $schedule->scheduled_date->lt(now()->startOfDay());
                @endphp

                @if($isMissed)
                    <div
                        class="bg-red-50 text-red-800 p-4 rounded-lg text-center font-bold border border-red-200 uppercase tracking-widest">
                        Job Missed
                    </div>
                @elseif($canStart)
                    <button @click="startJob()"
                        class="w-full bg-green-600 text-white font-bold py-4 rounded-lg hover:bg-green-700 shadow-lg flex items-center justify-center gap-2">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z">
                            </path>
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                        Start Patrol Job
                    </button>
                @else
                    <div class="bg-yellow-50 text-yellow-800 p-4 rounded-lg text-center font-bold border border-yellow-200">
                        <p>Job starts at {{ $schedule->from_time->format('h:i A') }}</p>
                        <p class="text-xs font-normal mt-1">You can start this job {{ $startTime->diffForHumans() }}</p>
                    </div>
                @endif
            @elseif($schedule->job_status === 'in_progress')
                @if(!$hasEndedShift)
                    <button @click="fetchReliever()"
                        class="w-full mb-3 bg-indigo-600 text-white font-bold py-4 rounded-lg hover:bg-indigo-700 shadow-lg flex items-center justify-center gap-2">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z">
                            </path>
                        </svg>
                        Reliever Info
                    </button>
                    <button @click="endJob()"
                        class="w-full bg-red-600 text-white font-bold py-4 rounded-lg hover:bg-red-700 shadow-lg">
                        End Patrol Job
                    </button>
                    <p class="text-xs text-center text-gray-500 mt-3">Only ends your shift. Job completes when all patrollers
                        finish.</p>
                @else
                    <div class="bg-gray-100 text-gray-500 p-4 rounded-lg text-center font-bold border border-gray-200">
                        <p>You have ended your shift.</p>
                        <p class="text-xs font-normal mt-1">Job status: In Progress (Waiting for others)</p>
                    </div>
                @endif
            @endif
        </div>

        {{-- Reliever Modal --}}
        <div x-show="showRelieverModal"
            class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4" style="display: none;"
            x-cloak>
            <div class="bg-white rounded-lg p-6 max-w-sm w-full shadow-2xl relative">
                <button @click="showRelieverModal = false" class="absolute top-4 right-4 text-gray-400 hover:text-gray-600">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                        </path>
                    </svg>
                </button>
                <h3 class="text-lg font-bold text-gray-900 mb-4">Next Scheduled Patrol</h3>

                <div x-show="loadingReliever" class="flex justify-center py-8">
                    <svg class="animate-spin h-8 w-8 text-blue-600" xmlns="http://www.w3.org/2000/svg" fill="none"
                        viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor"
                            d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z">
                        </path>
                    </svg>
                </div>

                <div x-show="!loadingReliever && reliever">
                    <template x-if="reliever && reliever.employee">
                        <div class="text-center">
                            <img :src="reliever.employee.photo"
                                class="w-24 h-24 rounded-full mx-auto object-cover border-4 border-slate-100 mb-4">
                            <h4 class="text-xl font-bold text-gray-900" x-text="reliever.employee.name"></h4>
                            <p class="text-sm text-gray-500 font-medium mb-1">Starts at <span x-text="reliever.start_time"
                                    class="text-gray-900 font-bold"></span></p>
                            <p class="text-xs text-slate-400 mb-6" x-text="reliever.site_name"></p>

                            <a :href="'tel:' + reliever.employee.phone"
                                class="block w-full py-3 bg-blue-600 text-white font-bold rounded-xl hover:bg-blue-700 transition-all text-center uppercase tracking-widest shadow-lg shadow-blue-500/30 flex items-center justify-center gap-2">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z">
                                    </path>
                                </svg>
                                Call Reliever
                            </a>
                        </div>
                    </template>
                    <template x-if="reliever && !reliever.employee && !reliever.self_relief">
                        <div class="py-4 text-center">
                            <p
                                class="text-amber-800 font-bold text-sm bg-amber-50 p-3 rounded-lg border border-amber-100 mb-4">
                                No specific employee assigned yet.<br><span class="text-xs font-normal">Next shift starts at
                                    <span x-text="reliever.start_time" class="font-bold"></span></span></p>
                            <button @click="showRelieverModal = false"
                                class="block w-full py-3 bg-slate-100 text-slate-600 font-bold rounded-xl hover:bg-slate-200 transition-all uppercase tracking-widest">Close</button>
                        </div>
                    </template>
                    <template x-if="reliever && reliever.self_relief">
                        <div class="py-4 text-center">
                            <p
                                class="text-blue-800 font-bold text-sm bg-blue-50 p-3 rounded-lg border border-blue-100 mb-4">
                                You are scheduled for the next shift.<br><span class="text-xs font-normal">Good luck!</span>
                            </p>
                            <button @click="showRelieverModal = false"
                                class="block w-full py-3 bg-slate-100 text-slate-600 font-bold rounded-xl hover:bg-slate-200 transition-all uppercase tracking-widest">Close</button>
                        </div>
                    </template>
                </div>
                <div x-show="!loadingReliever && !reliever" class="text-center py-8">
                    <p class="text-gray-500 font-medium">No next schedule found.</p>
                    <button @click="showRelieverModal = false"
                        class="mt-4 px-6 py-2 bg-slate-100 text-slate-600 font-bold rounded-lg hover:bg-slate-200 transition-all uppercase tracking-wider text-xs">Close</button>
                </div>
            </div>
        </div>

        {{-- Check-in Modal --}}
        <div x-show="showCheckinModal"
            class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4" style="display: none;">
            <div class="bg-white rounded-lg p-6 max-w-md w-full shadow-2xl">
                <h3 class="text-xl font-bold mb-1">Check In</h3>
                <p class="text-gray-500 mb-4 text-sm" x-text="checkinSiteName"></p>

                <form @submit.prevent="submitCheckin">
                    <div class="mb-6">
                        <div class="p-4 bg-purple-50 text-purple-700 rounded-lg text-sm font-medium">
                            <p>Please confirm your arrival at this site.</p>
                            <p class="text-xs mt-1 text-purple-600">Your location will be recorded.</p>
                        </div>
                    </div>
                    <div class="flex gap-3">
                        <button type="button" @click="showCheckinModal = false"
                            class="flex-1 bg-gray-100 text-gray-700 font-bold py-3 rounded-lg hover:bg-gray-200">
                            Cancel
                        </button>
                        <button type="submit"
                            class="flex-1 bg-purple-600 text-white font-bold py-3 rounded-lg hover:bg-purple-700 shadow">
                            Confirm Check In
                        </button>
                    </div>
                </form>
            </div>
        </div>

        {{-- Ticket Modal --}}
        <div x-show="showTicketModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4"
            style="display: none;">
            <div class="bg-white rounded-lg p-6 max-w-md w-full shadow-2xl">
                <h3 class="text-xl font-bold mb-4">Create Issue Ticket</h3>
                <form @submit.prevent="submitTicket">
                    <div class="mb-4">
                        <label class="block text-sm font-bold mb-2">Ticket Number <span
                                class="text-red-500">*</span></label>
                        <input type="text" x-model="ticketForm.ticket_number"
                            class="w-full bg-gray-50 border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none transition-all"
                            required placeholder="Enter ticket #" autofocus>
                    </div>

                    <div class="mb-4">
                        <label class="block text-sm font-bold mb-2">Issue Type</label>
                        <div class="grid grid-cols-2 gap-3">
                            <button type="button" @click="ticketForm.status = 'good'"
                                :class="{'bg-green-100 border-green-500 text-green-700': ticketForm.status === 'good', 'bg-white border-gray-200 text-gray-600': ticketForm.status !== 'good'}"
                                class="border rounded-lg p-3 text-center font-bold transition-all">
                                Good
                            </button>
                            <button type="button" @click="ticketForm.status = 'bad'"
                                :class="{'bg-red-100 border-red-500 text-red-700': ticketForm.status === 'bad', 'bg-white border-gray-200 text-gray-600': ticketForm.status !== 'bad'}"
                                class="border rounded-lg p-3 text-center font-bold transition-all">
                                Bad
                            </button>
                        </div>
                    </div>

                    <div class="mb-4">
                        <label class="block text-sm font-bold mb-2">Description</label>
                        <textarea x-model="ticketForm.description" rows="3"
                            class="w-full bg-gray-50 border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none transition-all"
                            required placeholder="Describe the issue..."></textarea>
                    </div>

                    <div class="mb-6">
                        <label class="block text-sm font-bold mb-2">Photos (Optional)</label>

                        {{-- Hidden Inputs --}}
                        <input type="file" x-ref="galleryInput" @change="handleTicketImages" multiple accept="image/*"
                            class="hidden">
                        <input type="file" x-ref="cameraInput" @change="handleTicketImages" accept="image/*"
                            capture="environment" class="hidden">

                        {{-- Action Buttons --}}
                        <div class="grid grid-cols-2 gap-3 mb-3">
                            <button type="button" @click="$refs.cameraInput.click()"
                                class="flex items-center justify-center gap-2 bg-slate-900 text-white p-3 rounded-lg font-bold text-sm hover:bg-slate-800 transition-all">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M3 9a2 2 0 012-2h.93a2 2 0 001.664-.89l.812-1.22A2 2 0 0110.07 4h3.86a2 2 0 011.664.89l.812 1.22A2 2 0 0018.07 7H19a2 2 0 012 2v9a2 2 0 01-2 2H5a2 2 0 01-2-2V9z">
                                    </path>
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M15 13a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                </svg>
                                Take Photo
                            </button>
                            <button type="button" @click="$refs.galleryInput.click()"
                                class="flex items-center justify-center gap-2 bg-gray-100 text-gray-700 p-3 rounded-lg font-bold text-sm hover:bg-gray-200 transition-all border border-gray-200">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z">
                                    </path>
                                </svg>
                                Upload
                            </button>
                        </div>

                        {{-- Image List --}}
                        <div x-show="ticketForm.images.length > 0" class="space-y-2">
                            <template x-for="(image, index) in ticketForm.images" :key="index">
                                <div
                                    class="flex items-center justify-between p-2 bg-gray-50 rounded border border-gray-100">
                                    <div class="flex items-center gap-2 overflow-hidden">
                                        <svg class="w-4 h-4 text-gray-400 flex-shrink-0" fill="none" stroke="currentColor"
                                            viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z">
                                            </path>
                                        </svg>
                                        <span class="text-xs text-gray-600 truncate" x-text="image.name"></span>
                                        <span class="text-[10px] text-gray-400"
                                            x-text="(image.size / 1024).toFixed(0) + 'KB'"></span>
                                    </div>
                                    <button type="button" @click="removeTicketImage(index)"
                                        class="text-red-500 hover:text-red-700">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                    </button>
                                </div>
                            </template>
                        </div>
                    </div>

                    <div class="flex gap-3">
                        <button type="button" @click="showTicketModal = false"
                            class="flex-1 bg-gray-100 text-gray-700 font-bold py-3 rounded-lg hover:bg-gray-200">
                            Cancel
                        </button>
                        <button type="submit"
                            class="flex-1 bg-blue-600 text-white font-bold py-3 rounded-lg hover:bg-blue-700 shadow">
                            Create Ticket
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        function patrollerJob() {
            return {
                showRelieverModal: false,
                reliever: null,
                loadingReliever: false,
                fetchReliever() {
                    this.loadingReliever = true;
                    this.showRelieverModal = true;
                    fetch(`/employee/patroller-jobs/{{ $schedule->id }}/reliever`)
                        .then(res => res.json())
                        .then(data => {
                            this.reliever = data;
                            this.loadingReliever = false;
                        })
                        .catch(err => {
                            console.error(err);
                            this.loadingReliever = false;
                        });
                },
                showCheckinModal: false,
                showTicketModal: false,
                currentSiteId: null,
                checkinSiteName: '',

                checkinForm: {
                    images: []
                },
                ticketForm: {
                    ticket_number: '',
                    description: '',
                    status: 'good',
                    images: []
                },

                openCheckinModal(siteId, siteName) {
                    this.currentSiteId = siteId;
                    this.checkinSiteName = siteName;
                    this.showCheckinModal = true;
                },

                openTicketModal(siteId) {
                    this.currentSiteId = siteId;
                    this.ticketForm.ticket_number = '';
                    this.ticketForm.description = '';
                    this.ticketForm.status = 'good';
                    this.ticketForm.images = [];
                    this.showTicketModal = true;
                },

                handleTicketImages(event) {
                    if (event.target.files.length > 0) {
                        this.ticketForm.images = [...this.ticketForm.images, ...Array.from(event.target.files)];
                        event.target.value = '';
                    }
                },

                removeTicketImage(index) {
                    this.ticketForm.images.splice(index, 1);
                },

                async submitCheckin() {
                    if (!navigator.geolocation) {
                        alert('Geolocation is not supported');
                        return;
                    }

                    const submitBtn = document.querySelector('button[type="submit"]');
                    const originalText = submitBtn.innerText;
                    submitBtn.innerText = 'Checking in...';
                    submitBtn.disabled = true;

                    navigator.geolocation.getCurrentPosition(async (position) => {
                        const formData = new FormData();
                        formData.append('latitude', position.coords.latitude);
                        formData.append('longitude', position.coords.longitude);

                        try {
                            const response = await fetch(`/employee/patroller-jobs/{{ $schedule->id }}/sites/${this.currentSiteId}/checkin`, {
                                method: 'POST',
                                headers: {
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                },
                                body: formData
                            });

                            if (response.ok) {
                                location.reload();
                            } else {
                                const data = await response.json();
                                alert(data.message || 'Check-in failed');
                                submitBtn.innerText = originalText;
                                submitBtn.disabled = false;
                            }
                        } catch (error) {
                            alert('Error: ' + error.message);
                            submitBtn.innerText = originalText;
                            submitBtn.disabled = false;
                        }
                    }, (error) => {
                        alert('Location error: ' + error.message);
                        submitBtn.innerText = originalText;
                        submitBtn.disabled = false;
                    });
                },

                async submitTicket() {
                    if (!navigator.geolocation) {
                        alert('Geolocation is not supported');
                        return;
                    }

                    const submitBtn = document.querySelector('#ticketSubmitBtn'); // Adjust selector as needed or use event

                    navigator.geolocation.getCurrentPosition(async (position) => {
                        const formData = new FormData();
                        formData.append('latitude', position.coords.latitude);
                        formData.append('longitude', position.coords.longitude);
                        formData.append('ticket_number', this.ticketForm.ticket_number);
                        formData.append('description', this.ticketForm.description);
                        formData.append('status', this.ticketForm.status);

                        this.ticketForm.images.forEach((image, index) => {
                            formData.append(`images[${index}]`, image);
                        });

                        try {
                            const response = await fetch(`/employee/patroller-jobs/{{ $schedule->id }}/sites/${this.currentSiteId}/tickets`, {
                                method: 'POST',
                                headers: {
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                },
                                body: formData
                            });

                            if (response.ok) {
                                location.reload();
                            } else {
                                const data = await response.json();
                                alert(data.message || 'Ticket creation failed');
                            }
                        } catch (error) {
                            alert('Error: ' + error.message);
                        }
                    });
                },

                async checkout(siteId) {
                    if (!navigator.geolocation) {
                        alert('Geolocation is not supported');
                        return;
                    }

                    if (!confirm('Are you sure you want to check out from this site?')) {
                        return;
                    }

                    navigator.geolocation.getCurrentPosition(async (position) => {
                        try {
                            const response = await fetch(`/employee/patroller-jobs/{{ $schedule->id }}/sites/${siteId}/checkout`, {
                                method: 'POST',
                                headers: {
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                    'Content-Type': 'application/json'
                                },
                                body: JSON.stringify({
                                    latitude: position.coords.latitude,
                                    longitude: position.coords.longitude
                                })
                            });

                            if (response.ok) {
                                location.reload();
                            } else {
                                const data = await response.json();
                                alert(data.message || 'Check-out failed');
                            }
                        } catch (error) {
                            alert('Error: ' + error.message);
                        }
                    }, (error) => {
                        alert('Location error: ' + error.message);
                        // Optional: Allow checkout without location if strictly necessary, but usually better to enforce
                    }, {
                        enableHighAccuracy: true,
                        timeout: 10000,
                        maximumAge: 0
                    });
                },

                async startJob() {
                    if (!confirm('Are you set to start this patrol job?')) {
                        return;
                    }

                    if (!navigator.geolocation) {
                        alert('Geolocation is not supported by your browser');
                        return;
                    }

                    navigator.geolocation.getCurrentPosition(async (position) => {
                        try {
                            const response = await fetch(`/employee/patroller-jobs/{{ $schedule->id }}/start`, {
                                method: 'POST',
                                headers: {
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                    'Content-Type': 'application/json'
                                },
                                body: JSON.stringify({
                                    latitude: position.coords.latitude,
                                    longitude: position.coords.longitude
                                })
                            });

                            if (response.ok) {
                                location.reload();
                            } else {
                                const data = await response.json();
                                alert(data.message || 'Failed to start job');
                            }
                        } catch (error) {
                            alert('Error: ' + error.message);
                        }
                    }, (error) => {
                        alert('Location check failed: ' + error.message + '. Please ensure location services are enabled.');
                    }, {
                        enableHighAccuracy: true,
                        timeout: 10000,
                        maximumAge: 0
                    });
                },

                async endJob() {
                    if (!confirm('Are you sure you want to end this patrol job?')) {
                        return;
                    }

                    try {
                        const response = await fetch(`/employee/patroller-jobs/{{ $schedule->id }}/end`, {
                            method: 'POST',
                            headers: {
                                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                'Content-Type': 'application/json'
                            }
                        });

                        if (response.ok) {
                            window.location.href = '/employee/dashboard';
                        } else {
                            const data = await response.json();
                            alert(data.message || 'Failed to end job');
                        }
                    } catch (error) {
                        alert('Error: ' + error.message);
                    }
                }
            }
        }
    </script>
@endsection