<?php

namespace App\Http\Controllers;

use App\Models\Site;
use App\Models\IdSetting;
use App\Models\Country;
use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

use Barryvdh\DomPDF\Facade\Pdf;

class SiteController extends Controller
{
    public function exportPDF(Request $request)
    {
        // Increase memory limit for PDF generation
        ini_set('memory_limit', '256M');

        $companyId = session('selected_company_id');
        $query = Site::where('company_id', $companyId)->with(['country', 'state']);

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('site_id', 'like', '%' . $request->search . '%');
            });
        }

        $sites = $query->get();

        $headers = [
            'Name',
            'Site ID',
            'Rate',
            'Contact Name',
            'Phone Main',
            'Email',
            'Address 1',
            'Address 2',
            'State',
            'City',
            'Zip',
            'Lat',
            'Long'
        ];

        $data = $sites->map(function ($site) {
            return [
                $site->name,
                $site->site_id,
                $site->rate,
                trim($site->contact_first_name . ' ' . $site->contact_last_name),
                $site->contact_phone_main,
                $site->contact_email,
                $site->address_line_1,
                $site->address_line_2,
                $site->state->name ?? '',
                $site->city,
                $site->zip_code,
                $site->latitude,
                $site->longitude,
            ];
        });

        // Get company information
        $company = \App\Models\Company::find($companyId);

        $pdf = Pdf::loadView('pdf.sites_all', [
            'title' => 'Sites List',
            'headers' => $headers,
            'data' => $data,
            'company' => $company
        ])->setPaper('a4', 'landscape');

        return $pdf->download('sites-export-' . now()->format('Y-m-d') . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = Site::where('company_id', $companyId)->with(['country', 'state']);

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('site_id', 'like', '%' . $request->search . '%');
            });
        }

        $sites = $query->get();

        // Create new Spreadsheet
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set headers
        $headers = [
            'ID',
            'Account Type',
            'Name',
            'Rate',
            'Site ID',
            'Language',
            'Contact Name',
            'Phone Main',
            'Phone Other',
            'Acc. Rep',
            'Sales Rep',
            'Fax',
            'Email',
            'Address 1',
            'Address 2',
            'Country',
            'State',
            'City',
            'Zip',
            'Lat',
            'Long',
            'Tags',
            'Website',
            'Active',
            'Created At'
        ];

        // Write headers
        $col = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($col . '1', $header);
            $col++;
        }

        // Style header row - Gray background and bold
        $headerStyle = [
            'font' => [
                'bold' => true,
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => [
                    'rgb' => 'CCCCCC', // Gray color
                ],
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                    'color' => ['rgb' => '000000'],
                ],
            ],
        ];

        $lastColumn = chr(64 + count($headers)); // Convert number to letter
        $sheet->getStyle('A1:' . $lastColumn . '1')->applyFromArray($headerStyle);

        // Write data rows
        $row = 2;
        foreach ($sites as $site) {
            $data = [
                $site->id,
                $site->account_type,
                $site->name,
                $site->rate,
                $site->site_id,
                $site->language,
                trim($site->contact_first_name . ' ' . $site->contact_last_name),
                $site->contact_phone_main,
                $site->contact_phone_other,
                $site->account_representative,
                $site->sales_representative,
                $site->fax,
                $site->contact_email,
                $site->address_line_1,
                $site->address_line_2,
                $site->country->name ?? '',
                $site->state->name ?? '',
                $site->city,
                $site->zip_code,
                $site->latitude,
                $site->longitude,
                $site->tags,
                $site->website,
                $site->active ? 'Yes' : 'No',
                $site->created_at->format('Y-m-d'),
            ];

            $col = 'A';
            foreach ($data as $value) {
                $sheet->setCellValue($col . $row, $value);
                $col++;
            }

            // Apply borders to data row
            $sheet->getStyle('A' . $row . ':' . $lastColumn . $row)->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        'color' => ['rgb' => '000000'],
                    ],
                ],
            ]);

            $row++;
        }

        // Auto-size columns
        foreach (range('A', $lastColumn) as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }

        // Create Excel file
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $filename = 'sites-export-' . now()->format('Y-m-d') . '.xlsx';

        // Set headers for download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    public function index(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = Site::where('company_id', $companyId);

        // Search
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('site_id', 'like', '%' . $request->search . '%');
            });
        }

        // Sort
        $allowedSort = ['name', 'site_id', 'account_type', 'rate'];
        $sort = $request->get('sort', 'name');
        if (!in_array($sort, $allowedSort)) {
            $sort = 'name';
        }
        $direction = $request->get('direction', 'asc');
        $query->orderBy($sort, $direction);

        $sites = $query->paginate(100)->withQueryString();

        $targetCompanies = auth()->user()->companies()
            ->where('companies.id', '!=', $companyId)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        if ($request->ajax()) {
            return view('sites._table', compact('sites', 'targetCompanies'))->render();
        }

        return view('sites.index', compact('sites', 'targetCompanies'));
    }

    public function create()
    {
        $companyId = session('selected_company_id');
        $countries = Country::where('active', true)->orderBy('name')->get();
        $states = State::where('active', true)->orderBy('name')->get();

        // Generate Site ID
        $setting = IdSetting::where('company_id', $companyId)->first();
        $prefix = $setting ? $setting->site_prefix : 'S';
        $nextNumber = $setting ? $setting->site_next_number : 1;
        $generatedId = $prefix . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);

        return view('sites.create', compact('countries', 'states', 'generatedId'));
    }

    public function validateForm(Request $request)
    {
        $rules = [
            'account_type' => 'required|string',
            'name' => 'required|string|max:255',
            'rate' => 'required|numeric',
            'address_line_1' => 'required|string|max:255',
            'country_id' => 'nullable|exists:countries,id',
            'state_id' => 'nullable|exists:states,id',
            'city' => 'nullable|string|max:255',
            'zip_code' => 'nullable|string|max:20',
            'latitude' => 'required|string',
            'longitude' => 'required|string',
        ];

        if ($request->has('step')) {
            if ($request->step === 'company') {
                $rules = array_intersect_key($rules, array_flip(['account_type', 'name', 'rate']));
            } elseif ($request->step === 'address') {
                $rules = array_intersect_key($rules, array_flip(['address_line_1', 'country_id', 'state_id', 'city', 'zip_code', 'latitude', 'longitude']));
            }
        }

        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        return response()->json(['success' => true]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'account_type' => 'required|string',
            'name' => 'required|string|max:255',
            'rate' => 'required|numeric',
            'address_line_1' => 'required|string|max:255',
            'country_id' => 'nullable|exists:countries,id',
            'state_id' => 'nullable|exists:states,id',
            'city' => 'nullable|string|max:255',
            'zip_code' => 'nullable|string|max:20',
            'latitude' => 'required|string',
            'longitude' => 'required|string',
        ]);

        DB::transaction(function () use ($request) {
            $companyId = session('selected_company_id');

            // Handle logo
            $logoPath = null;
            if ($request->hasFile('logo')) {
                $logoPath = $request->file('logo')->store('sites/logos', 'public');
            }

            // Create Site
            Site::create(array_merge($request->all(), [
                'company_id' => $companyId,
                'logo' => $logoPath,
                'site_id' => $request->site_id // We use the ID passed from the hidden/readonly field
            ]));

            // Increment Site Number
            $setting = IdSetting::where('company_id', $companyId)->first();
            if ($setting) {
                $setting->increment('site_next_number');
            }
        });

        return redirect()->route('sites.index')->with('success', 'Site created successfully!');
    }

    public function show(Site $site)
    {
        if ($site->company_id != session('selected_company_id')) {
            abort(403);
        }

        $site->load(['bannedEmployees']);

        $jobRoles = \App\Models\JobRole::where('active', true)->get();
        $employees = \App\Models\Employee::where('company_id', session('selected_company_id'))->where('active', true)->orderBy('first_name')->get();
        return view('sites.show', compact('site', 'jobRoles', 'employees'));
    }

    public function edit(Site $site)
    {
        if ($site->company_id != session('selected_company_id')) {
            abort(403);
        }

        $countries = Country::where('active', true)->orderBy('name')->get();
        $states = State::where('active', true)->orderBy('name')->get();

        return view('sites.edit', compact('site', 'countries', 'states'));
    }

    public function update(Request $request, Site $site)
    {
        if ($site->company_id != session('selected_company_id')) {
            abort(403);
        }

        $request->validate([
            'account_type' => 'required|string',
            'name' => 'required|string|max:255',
            'rate' => 'required|numeric',
            'address_line_1' => 'required|string|max:255',
            'country_id' => 'nullable|exists:countries,id',
            'state_id' => 'nullable|exists:states,id',
            'city' => 'nullable|string|max:255',
            'zip_code' => 'nullable|string|max:20',
            'latitude' => 'required|string',
            'longitude' => 'required|string',
        ]);

        if ($request->hasFile('logo')) {
            $site->logo = $request->file('logo')->store('sites/logos', 'public');
        }

        $site->update($request->except('logo'));

        return redirect()->route('sites.index')->with('success', 'Site updated successfully!');
    }

    public function destroy(Site $site)
    {
        if ($site->company_id != session('selected_company_id')) {
            abort(403);
        }

        $site->delete();
        return redirect()->route('sites.index')->with('success', 'Site deleted successfully!');
    }

    public function toggleStatus(Site $site)
    {
        if ($site->company_id != session('selected_company_id')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $site->active = !$site->active;
        $site->save();

        return response()->json([
            'success' => true,
            'active' => $site->active,
            'message' => 'Site status updated successfully!'
        ]);
    }

    public function banEmployee(Request $request, Site $site, \App\Models\Employee $employee)
    {
        if ($site->company_id != session('selected_company_id')) {
            abort(403);
        }

        // Attach employee to banned_employees
        $site->bannedEmployees()->syncWithoutDetaching([
            $employee->id => [
                'reason' => $request->input('reason', 'Banned via Schedule Interface'),
                'banned_at' => now()
            ]
        ]);

        return back()->with('success', "Employee {$employee->first_name} has been banned from this site.");
    }

    public function unbanEmployee(Site $site, \App\Models\Employee $employee)
    {
        if ($site->company_id != session('selected_company_id')) {
            abort(403);
        }

        $site->bannedEmployees()->detach($employee->id);

        return back()->with('success', "Ban removed for employee {$employee->first_name}.");
    }

    public function bulkUpdate(Request $request)
    {
        $request->validate([
            'sites' => 'required|array',
            'sites.*.rate' => 'nullable|numeric|min:0',
            'sites.*.address_1' => 'nullable|string|max:255',
            'sites.*.country_id' => 'nullable|exists:countries,id',
            'sites.*.state_id' => 'nullable|exists:states,id',
            'sites.*.city' => 'nullable|string|max:255',
            'sites.*.zipcode' => 'nullable|string|max:20',
            'sites.*.latitude' => 'nullable|string|max:255',
            'sites.*.longitude' => 'nullable|string|max:255',
        ]);

        $siteData = $request->sites;
        $count = count($siteData);
        $companyId = session('selected_company_id');

        DB::transaction(function () use ($siteData, $companyId) {
            foreach ($siteData as $id => $data) {
                // Ensure site belongs to current company
                $site = Site::where('company_id', $companyId)->find($id);
                if (!$site)
                    continue;

                $updateData = [];
                // Only update fields that are present/filled
                if (isset($data['rate']) && $data['rate'] !== '')
                    $updateData['rate'] = $data['rate'];
                if (isset($data['address_1']))
                    $updateData['address_1'] = $data['address_1'];
                if (isset($data['country_id']))
                    $updateData['country_id'] = $data['country_id'];
                if (isset($data['state_id']))
                    $updateData['state_id'] = $data['state_id'];
                if (isset($data['city']))
                    $updateData['city'] = $data['city'];
                if (isset($data['zipcode']))
                    $updateData['zipcode'] = $data['zipcode'];
                if (isset($data['latitude']))
                    $updateData['latitude'] = $data['latitude'];
                if (isset($data['longitude']))
                    $updateData['longitude'] = $data['longitude'];

                if (!empty($updateData)) {
                    $site->update($updateData);
                }
            }
        });

        return redirect()->back()->with('success', "$count sites updated successfully.");
    }

    public function copy(Request $request)
    {
        $request->validate([
            'site_ids' => 'required|array',
            'site_ids.*' => 'exists:sites,id',
            'target_company_id' => 'required|exists:companies,id',
        ]);

        $targetCompanyId = $request->target_company_id;
        // Verify user has access to target company
        if (!auth()->user()->companies()->where('companies.id', $targetCompanyId)->exists()) {
            return redirect()->back()->with('error', 'You do not have access to the selected company.');
        }

        $sites = Site::whereIn('id', $request->site_ids)->get();
        $count = 0;

        foreach ($sites as $site) {
            $newSite = $site->replicate();
            $newSite->company_id = $targetCompanyId;
            $newSite->site_id = $this->generateSiteId($targetCompanyId);
            $newSite->created_at = now();
            $newSite->updated_at = now();
            $newSite->save();

            // Increment Site Number for target company
            $setting = IdSetting::where('company_id', $targetCompanyId)->first();
            if ($setting) {
                $setting->increment('site_next_number');
            }

            $count++;
        }

        return redirect()->back()->with('success', "$count sites copied successfully.");
    }

    private function generateSiteId($companyId)
    {
        $setting = IdSetting::where('company_id', $companyId)->first();
        $prefix = $setting ? $setting->site_prefix : 'S';
        $nextNumber = $setting ? $setting->site_next_number : 1;

        return $prefix . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
    }
}
