<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class QuotationController extends Controller
{
    public function index(Request $request)
    {
        $companyId = $request->get('company_id');
        $creatorId = $request->get('user_id');
        $tab = $request->get('tab', 'all');

        // Get the currently selected company from session
        $selectedCompanyId = session('selected_company_id');

        $query = \App\Models\Quotation::with(['user', 'company']);

        // IMPORTANT: Filter by selected company to ensure quotations are company-specific
        if ($selectedCompanyId) {
            $query->where('company_id', $selectedCompanyId);
        }

        // Email History tab filter - show only quotations that have been emailed
        if ($tab === 'email_history') {
            $query->where('email_sent_count', '>', 0)->with('emailLogs.sender');
        }

        // Additional filters from request
        if ($companyId) {
            $query->where('company_id', $companyId);
        }
        if ($creatorId) {
            $query->where('user_id', $creatorId);
        }

        $quotations = $query->latest()->paginate(100)->appends($request->except('page'));

        $companies = \App\Models\Company::orderBy('name')->pluck('name', 'id');
        $creators = \App\Models\User::orderBy('name')->pluck('name', 'id');

        return view('quotations.index', compact('quotations', 'companies', 'creators'));
    }

    public function create()
    {
        // Pre-fill next quote number
        $nextNumber = $this->generateQuotationNumber();
        return view('quotations.create', compact('nextNumber'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'client_name' => 'required|string|max:255',
            'client_email' => 'nullable|email',
            'client_address' => 'nullable|string',
            'issue_date' => 'required|date',
            'valid_until' => 'nullable|date|after_or_equal:issue_date',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|string',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.hours' => 'nullable|numeric|min:0',
            'tax_rate' => 'nullable|numeric|min:0|max:100',
            'notes' => 'nullable|string',
        ]);

        // Calculate totals logic here or trust frontend submission but verify? 
        // Best practice: Recalculate on backend to prevent tampering.

        $subtotal = 0;
        $itemsData = [];

        foreach ($request->items as $item) {
            $qty = $item['quantity'] ?? 1;
            $hours = $item['hours'] ?? 0;
            $rate = $item['unit_price'] ?? 0;
            $type = $item['type'] ?? 'service';

            // If service has hours, it's Qty * Hours * Rate. If Product, Qty * Price.
            // Wait, standard might mean Rate is per hour.
            // If type is product, hours should be ignored or 1.

            $lineTotal = 0;
            if ($type === 'product' || empty($hours) || $hours == 0) {
                $lineTotal = $qty * $rate;
            } else {
                $lineTotal = $qty * $hours * $rate;
            }

            $subtotal += $lineTotal;

            $itemsData[] = [
                'type' => $type,
                'description' => $item['description'],
                'quantity' => $qty,
                'hours' => $hours,
                'unit_price' => $rate,
                'total_price' => $lineTotal,
            ];
        }

        $taxRate = $request->tax_rate ?? 0;
        $taxAmount = $subtotal * ($taxRate / 100);
        $totalAmount = $subtotal + $taxAmount;

        $quotation = \App\Models\Quotation::create([
            'user_id' => auth()->id() ?? 1, // Fallback for dev if needed
            'company_id' => session('selected_company_id'),
            'quotation_number' => $this->generateQuotationNumber(),
            'client_name' => $request->client_name,
            'client_email' => $request->client_email,
            'client_phone' => $request->client_phone,
            'client_address' => $request->client_address,
            'issue_date' => $request->issue_date,
            'valid_until' => $request->valid_until,
            'subtotal' => $subtotal,
            'tax_rate' => $taxRate,
            'tax_amount' => $taxAmount,
            'total_amount' => $totalAmount,
            'status' => 'draft',
            'notes' => $request->notes,
        ]);

        $quotation->items()->createMany($itemsData);

        return redirect()->route('quotations.index')->with('success', 'Quotation created successfully.');
    }

    private function generateQuotationNumber()
    {
        $latest = \App\Models\Quotation::latest()->first();
        if (!$latest) {
            return 'Q-' . date('Y') . '-001';
        }

        // simple increment logic
        $parts = explode('-', $latest->quotation_number);
        if (count($parts) === 3) {
            $num = intval($parts[2]) + 1;
            return 'Q-' . date('Y') . '-' . str_pad($num, 3, '0', STR_PAD_LEFT);
        }

        return 'Q-' . date('Y') . '-' . rand(100, 999);
    }

    public function show($id)
    {
        $selectedCompanyId = session('selected_company_id');

        // Scope query to selected company if set
        $query = \App\Models\Quotation::with(['items', 'user', 'company']);
        if ($selectedCompanyId) {
            $query->where('company_id', $selectedCompanyId);
        }

        $quotation = $query->findOrFail($id);
        return view('quotations.show', compact('quotation'));
    }

    public function downloadPdf($id)
    {
        $selectedCompanyId = session('selected_company_id');

        // Scope query to selected company if set
        $query = \App\Models\Quotation::with(['items', 'user', 'company']);
        if ($selectedCompanyId) {
            $query->where('company_id', $selectedCompanyId);
        }

        $quotation = $query->findOrFail($id);
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('quotations.pdf', compact('quotation'));
        return $pdf->download('quotation-' . $quotation->quotation_number . '.pdf');
    }

    public function sendEmail($id)
    {
        $selectedCompanyId = session('selected_company_id');

        // Scope query to selected company if set
        $query = \App\Models\Quotation::with(['items', 'user', 'company']);
        if ($selectedCompanyId) {
            $query->where('company_id', $selectedCompanyId);
        }

        $quotation = $query->findOrFail($id);
        // Routes are already protected by auth middleware - no additional checks needed

        if (!$quotation->client_email) {
            return back()->withErrors(['email' => 'Client email is not set for this quotation.']);
        }

        // Generate PDF
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('quotations.pdf', compact('quotation'));

        // Send Email
        try {
            \Illuminate\Support\Facades\Mail::send('emails.quotation_sent', ['quotation' => $quotation], function ($message) use ($quotation, $pdf) {
                $message->to($quotation->client_email, $quotation->client_name)
                    ->subject('Quotation ' . $quotation->quotation_number . ' from ' . $quotation->company->name)
                    ->attachData($pdf->output(), 'quotation-' . $quotation->quotation_number . '.pdf');
            });

            // Update quotation status and email tracking
            $quotation->update([
                'status' => $quotation->status === 'draft' ? 'sent' : $quotation->status,
                'email_sent_at' => $quotation->email_sent_at ?? now(),
                'last_email_sent_at' => now(),
                'email_sent_count' => $quotation->email_sent_count + 1,
            ]);

            // Create detailed email log
            \App\Models\QuotationEmailLog::create([
                'quotation_id' => $quotation->id,
                'company_id' => $quotation->company_id,
                'sent_by' => auth()->id(),
                'sent_to' => $quotation->client_email,
                'subject' => 'Quotation ' . $quotation->quotation_number . ' from ' . $quotation->company->name,
                'sent_at' => now(),
            ]);

            return back()->with('success', 'Quotation emailed successfully to ' . $quotation->client_email);
        } catch (\Exception $e) {
            return back()->withErrors(['email' => 'Failed to send email: ' . $e->getMessage()]);
        }
    }

    public function resendEmail($id)
    {
        return $this->sendEmail($id);
    }
}
