<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\Meeting;
use App\Models\MeetingParticipant;
use App\Models\MeetingMessage;
use App\Events\MeetingJoinRequested;
use App\Events\MeetingJoinApproved;
use App\Events\MeetingMessageSent;
use App\Events\MeetingEnded;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;

class MeetingController extends Controller
{
    public function index()
    {
        $meetings = Meeting::where('creator_id', Auth::id())
            ->orWhereHas('participants', function ($q) {
                $q->where('user_id', Auth::id())->where('status', 'approved');
            })
            ->orderBy('created_at', 'desc')
            ->get();

        return view('meetings.index', compact('meetings'));
    }

    public function create()
    {
        return view('meetings.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
        ]);

        $meeting = Meeting::create([
            'uuid' => (string) Str::uuid(),
            'title' => $request->title,
            'creator_id' => Auth::id(),
            'status' => 'active',
            'starts_at' => now(),
        ]);

        // Creator automatically approved
        MeetingParticipant::create([
            'meeting_id' => $meeting->id,
            'user_id' => Auth::id(),
            'status' => 'approved',
            'joined_at' => now(),
        ]);

        return redirect()->route('meetings.room', $meeting->uuid);
    }

    public function join($uuid)
    {
        $meeting = Meeting::where('uuid', $uuid)->firstOrFail();

        if ($meeting->status === 'finished') {
            return redirect()->route('meetings.index')->with('error', 'This meeting has ended.');
        }

        if ($meeting->creator_id === Auth::id()) {
            return redirect()->route('meetings.room', $uuid);
        }

        $participant = MeetingParticipant::firstOrCreate(
            ['meeting_id' => $meeting->id, 'user_id' => Auth::id()],
            ['status' => 'pending']
        );

        if ($participant->status === 'approved') {
            return redirect()->route('meetings.room', $uuid);
        }

        // Trigger Join Requested Event for the Host
        broadcast(new MeetingJoinRequested($meeting, Auth::user()));

        return view('meetings.waiting-room', compact('meeting', 'participant'));
    }

    public function requestJoin($uuid)
    {
        $meeting = Meeting::where('uuid', $uuid)->firstOrFail();
        broadcast(new MeetingJoinRequested($meeting, Auth::user()));
        return response()->json(['success' => true]);
    }

    public function approve(Request $request, $uuid)
    {
        $meeting = Meeting::where('uuid', $uuid)->firstOrFail();

        if ($meeting->creator_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $participant = MeetingParticipant::where('meeting_id', $meeting->id)
            ->where('user_id', $request->user_id)
            ->firstOrFail();

        $participant->update([
            'status' => 'approved',
            'joined_at' => now()
        ]);

        broadcast(new MeetingJoinApproved($meeting, $participant->user))->toOthers();

        return response()->json(['success' => true]);
    }

    public function meetingRoom($uuid)
    {
        $meeting = Meeting::where('uuid', $uuid)->with(['creator', 'messages.user'])->firstOrFail();

        if ($meeting->status === 'finished') {
            return redirect()->route('meetings.index')->with('error', 'This meeting has ended.');
        }

        $participant = MeetingParticipant::where('meeting_id', $meeting->id)
            ->where('user_id', Auth::id())
            ->firstOrFail();

        if ($participant->status !== 'approved') {
            return redirect()->route('meetings.join', $uuid);
        }

        $pendingParticipants = [];
        if ($meeting->creator_id === Auth::id()) {
            $pendingParticipants = MeetingParticipant::where('meeting_id', $meeting->id)
                ->where('status', 'pending')
                ->with('user')
                ->get()
                ->map(function ($p) {
                    return [
                        'user' => $p->user
                    ];
                });
        }

        return view('meetings.room', compact('meeting', 'participant', 'pendingParticipants'));
    }

    public function sendMessage(Request $request, $uuid)
    {
        $meeting = Meeting::where('uuid', $uuid)->firstOrFail();

        $request->validate([
            'message' => 'required_without:file|string|nullable',
            'file' => 'nullable|file|max:10240', // 10MB
        ]);

        $filePath = null;
        $type = 'text';

        if ($request->hasFile('file')) {
            $filePath = $request->file('file')->store('meeting_files/' . $uuid, 'public');
            $type = Str::contains($request->file('file')->getMimeType(), 'image') ? 'image' : 'file';
        }

        $message = MeetingMessage::create([
            'meeting_id' => $meeting->id,
            'user_id' => Auth::id(),
            'message' => $request->message,
            'file_path' => $filePath,
            'type' => $type,
        ]);

        broadcast(new MeetingMessageSent($message))->toOthers();

        return response()->json(['message' => $message->load('user')]);
    }

    public function endMeeting($uuid)
    {
        $meeting = Meeting::where('uuid', $uuid)->firstOrFail();

        if ($meeting->creator_id !== Auth::id()) {
            return redirect()->back()->with('error', 'Only the host can end the meeting.');
        }

        $meeting->update([
            'status' => 'finished',
            'ends_at' => now()
        ]);

        broadcast(new MeetingEnded($meeting))->toOthers();

        return redirect()->route('meetings.index')->with('success', 'Meeting ended successfully.');
    }
}
