<?php

namespace App\Http\Controllers;
use App\Models\CheckpointScan;
use App\Models\Incident;
use App\Models\JobNote;
use Illuminate\Http\Request;

class LiveActivityController extends Controller
{
    public function feed()
    {
        $companyId = session('selected_company_id');

        // Fetch Recent Notes
        $notes = JobNote::with(['employee'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->latest()
            ->limit(10)
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'note',
                    'title' => 'New Mission Note',
                    'content' => $item->note,
                    'employee' => $item->employee->first_name . ' ' . $item->employee->last_name,
                    'timestamp' => $item->created_at,
                    'image' => $item->image ? asset('storage/' . $item->image) : null,
                ];
            });

        // Fetch Recent Incidents
        $incidents = Incident::with(['employee', 'incidentCategory'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->latest()
            ->limit(10)
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'incident',
                    'title' => 'Incident Reported: ' . ($item->incidentCategory->name ?? 'General'),
                    'content' => $item->description,
                    'employee' => $item->employee->first_name . ' ' . $item->employee->last_name,
                    'timestamp' => $item->created_at,
                    'image' => count($item->images ?? []) > 0 ? asset('storage/' . $item->images[0]) : null,
                ];
            });

        // Fetch Recent Checkpoint Scans
        $scans = CheckpointScan::with(['employee', 'checkpoint'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->latest()
            ->limit(10)
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'scan',
                    'title' => 'Checkpoint Scanned',
                    'content' => 'Checkpoint: ' . ($item->checkpoint->name ?? 'Unknown'),
                    'employee' => $item->employee->first_name . ' ' . $item->employee->last_name,
                    'timestamp' => $item->created_at,
                    'image' => $item->evidence_image ? asset('storage/' . $item->evidence_image) : null,
                ];
            });

        // Fetch Recent Check-ins (Guard Started Job)
        $checkins = \Illuminate\Support\Facades\DB::table('employee_schedule')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->where('employees.company_id', $companyId)
            ->whereNotNull('employee_schedule.actual_start_at')
            ->orderBy('employee_schedule.actual_start_at', 'desc')
            ->limit(10)
            ->select(
                'employees.first_name',
                'employees.last_name',
                'sites.name as site_name',
                'employee_schedule.actual_start_at',
                'employee_schedule.checkin_images'
            )
            ->get()
            ->map(function ($item) {
                // Decode JSON images if present
                $images = json_decode($item->checkin_images);
                $image = (is_array($images) && count($images) > 0) ? $images[0] : null;

                return [
                    'type' => 'checkin',
                    'title' => 'Guard Checked In',
                    'content' => 'Started shift at ' . $item->site_name,
                    'employee' => $item->first_name . ' ' . $item->last_name,
                    'timestamp' => \Carbon\Carbon::parse($item->actual_start_at),
                    'image' => $image ? asset('storage/' . $image) : null,
                ];
            });

        // Fetch Recent Check-outs (Guard Ended Job)
        $checkouts = \Illuminate\Support\Facades\DB::table('employee_schedule')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->where('employees.company_id', $companyId)
            ->whereNotNull('employee_schedule.actual_end_at')
            ->orderBy('employee_schedule.actual_end_at', 'desc')
            ->limit(10)
            ->select(
                'employees.first_name',
                'employees.last_name',
                'sites.name as site_name',
                'employee_schedule.actual_end_at',
                'employee_schedule.checkout_images'
            )
            ->get()
            ->map(function ($item) {
                // Decode JSON images if present
                $images = json_decode($item->checkout_images);
                $image = (is_array($images) && count($images) > 0) ? $images[0] : null;

                return [
                    'type' => 'checkout',
                    'title' => 'Guard Checked Out',
                    'content' => 'Ended shift at ' . $item->site_name,
                    'employee' => $item->first_name . ' ' . $item->last_name,
                    'timestamp' => \Carbon\Carbon::parse($item->actual_end_at),
                    'image' => $image ? asset('storage/' . $image) : null,
                ];
            });

        // Fetch Recent Patroller Job Starts
        $patrollerStarts = \Illuminate\Support\Facades\DB::table('employee_patroller_schedule')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('employees.company_id', $companyId)
            ->whereNotNull('employee_patroller_schedule.actual_start_at')
            ->orderBy('employee_patroller_schedule.actual_start_at', 'desc')
            ->limit(10)
            ->select(
                'employees.first_name',
                'employees.last_name',
                'routes.name as route_name',
                'employee_patroller_schedule.actual_start_at',
                'employee_patroller_schedule.start_evidence_path'
            )
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'patrol_start',
                    'title' => 'Patroller Started Duty',
                    'content' => 'Started patrol on: ' . $item->route_name,
                    'employee' => $item->first_name . ' ' . $item->last_name,
                    'timestamp' => \Carbon\Carbon::parse($item->actual_start_at),
                    'image' => $item->start_evidence_path ? asset('storage/' . $item->start_evidence_path) : null,
                ];
            });

        // Fetch Recent Patroller Job Ends
        $patrollerEnds = \Illuminate\Support\Facades\DB::table('employee_patroller_schedule')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('employees.company_id', $companyId)
            ->whereNotNull('employee_patroller_schedule.actual_end_at')
            ->orderBy('employee_patroller_schedule.actual_end_at', 'desc')
            ->limit(10)
            ->select(
                'employees.first_name',
                'employees.last_name',
                'routes.name as route_name',
                'employee_patroller_schedule.actual_end_at',
                'employee_patroller_schedule.end_evidence_path'
            )
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'patrol_end',
                    'title' => 'Patroller Ended Duty',
                    'content' => 'Ended patrol on: ' . $item->route_name,
                    'employee' => $item->first_name . ' ' . $item->last_name,
                    'timestamp' => \Carbon\Carbon::parse($item->actual_end_at),
                    'image' => $item->end_evidence_path ? asset('storage/' . $item->end_evidence_path) : null,
                ];
            });

        // Fetch Recent Patroller Tickets
        $patrollerTickets = \App\Models\PatrollerIssueTicket::with(['employee', 'site'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->latest()
            ->limit(10)
            ->get()
            ->map(function ($item) {
                $image = (is_array($item->images) && count($item->images) > 0) ? $item->images[0] : null;
                return [
                    'type' => 'patrol_ticket',
                    'title' => 'Patroller Issued Ticket',
                    'content' => 'Ticket #' . $item->ticket_number . ' at ' . ($item->site->name ?? 'Unknown Site'),
                    'employee' => $item->employee->first_name . ' ' . $item->employee->last_name,
                    'timestamp' => $item->created_at,
                    'image' => $image ? asset('storage/' . $image) : null,
                ];
            });

        // Fetch Patroller Site Check-ins
        $patrollerSiteCheckins = \App\Models\PatrollerJobSite::with(['employee', 'site'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->whereNotNull('checked_in_at')
            ->orderBy('checked_in_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($item) {
                $image = (is_array($item->checkin_images) && count($item->checkin_images) > 0) ? $item->checkin_images[0] : null;
                return [
                    'type' => 'patrol_site_checkin',
                    'title' => 'Patroller Arrived at Site',
                    'content' => 'Arrived at ' . ($item->site->name ?? 'Unknown Site'),
                    'employee' => $item->employee->first_name . ' ' . $item->employee->last_name,
                    'timestamp' => $item->checked_in_at,
                    'image' => $image ? asset('storage/' . $image) : null,
                ];
            });

        // Fetch Patroller Site Check-outs
        $patrollerSiteCheckouts = \App\Models\PatrollerJobSite::with(['employee', 'site'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
            ->whereNotNull('checked_out_at')
            ->orderBy('checked_out_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'patrol_site_checkout',
                    'title' => 'Patroller Left Site',
                    'content' => 'Completed visit at ' . ($item->site->name ?? 'Unknown Site'),
                    'employee' => $item->employee->first_name . ' ' . $item->employee->last_name,
                    'timestamp' => $item->checked_out_at,
                    'image' => null,
                ];
            });

        $activities = $notes
            ->concat($incidents)
            ->concat($scans)
            ->concat($checkins)
            ->concat($checkouts)
            ->concat($patrollerStarts)
            ->concat($patrollerEnds)
            ->concat($patrollerTickets)
            ->concat($patrollerSiteCheckins)
            ->concat($patrollerSiteCheckouts)
            ->sortByDesc('timestamp')
            ->take(15);

        return view('partials.activity-feed', compact('activities'));
    }
}
