<?php

namespace App\Http\Controllers;

use App\Models\State;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Log;

class EmployeeController extends Controller
{
    public function index(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = \App\Models\Employee::where('company_id', $companyId)
            ->with(['jobRole', 'department', 'wageTypes']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('employee_id', 'like', "%{$search}%")
                    ->orWhere('phone_number', 'like', "%{$search}%");
            });
        }

        // Filter by Department
        if ($request->filled('department_id')) {
            $query->where('department_id', $request->department_id);
        }

        // Filter by Job Role
        if ($request->filled('job_role_id')) {
            $query->where('job_role_id', $request->job_role_id);
        }

        // Filter by Status
        if ($request->filled('status')) {
            $status = $request->status;
            if ($status === 'active') {
                $query->where('active', true);
            } elseif ($status === 'inactive') {
                $query->where('active', false);
            }
        }

        // Sorting
        $sortField = $request->get('sort', 'created_at');
        $sortDirection = $request->get('direction', 'desc');

        // Validate sort field
        $allowedSorts = ['first_name', 'last_name', 'employee_id', 'department_id', 'job_role_id', 'created_at'];
        if (!in_array($sortField, $allowedSorts)) {
            $sortField = 'created_at';
        }

        // Handle relationship sorting
        if ($sortField === 'department_id') {
            $query->leftJoin('departments', 'employees.department_id', '=', 'departments.id')
                ->select('employees.*')
                ->orderBy('departments.name', $sortDirection);
        } elseif ($sortField === 'job_role_id') {
            $query->leftJoin('job_roles', 'employees.job_role_id', '=', 'job_roles.id')
                ->select('employees.*')
                ->orderBy('job_roles.name', $sortDirection);
        } else {
            $query->orderBy($sortField, $sortDirection);
        }

        $employees = $query->paginate(100)
            ->withQueryString();

        $departments = \App\Models\Department::where('active', true)->orderBy('name')->get();
        $jobRoles = \App\Models\JobRole::where('active', true)->orderBy('name')->get();
        $zones = \App\Models\Zone::where('active', true)->orderBy('name')->get();
        $wageTypes = \App\Models\WageType::all();

        $targetCompanies = auth()->user()->companies()
            ->where('companies.id', '!=', $companyId)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        if ($request->ajax()) {
            return view('employees._table', compact('employees', 'departments', 'jobRoles', 'zones', 'wageTypes', 'targetCompanies'))->render();
        }

        return view('employees.index', compact('employees', 'departments', 'jobRoles', 'zones', 'wageTypes', 'targetCompanies'));
    }

    public function exportPDF(Request $request)
    {
        // Increase memory limit for PDF generation
        ini_set('memory_limit', '256M');

        $companyId = session('selected_company_id');
        $query = \App\Models\Employee::where('company_id', $companyId)
            ->with(['jobRole', 'department', 'wageTypes']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('employee_id', 'like', "%{$search}%")
                    ->orWhere('phone_number', 'like', "%{$search}%");
            });
        }

        // Filter by Department
        if ($request->filled('department_id')) {
            $query->where('department_id', $request->department_id);
        }

        // Filter by Job Role
        if ($request->filled('job_role_id')) {
            $query->where('job_role_id', $request->job_role_id);
        }

        // Filter by Status
        if ($request->filled('status')) {
            $status = $request->status;
            if ($status === 'active') {
                $query->where('active', true);
            } elseif ($status === 'inactive') {
                $query->where('active', false);
            }
        }

        $employees = $query->latest()->get();

        $headers = [
            'First Name',
            'Last Name',
            'Employee ID',
            'Email',
            'Phone',
            'Department',
            'Job Role',
            'SIN Rate',
            'Other Rate',
            'Hired Date'
        ];

        $data = $employees->map(function ($employee) {
            // Calculate SIN Rate and Other Rate (sum of wage type rates)
            $sinRate = 0;
            $otherRate = 0;
            if ($employee->wageTypes && $employee->wageTypes->count() > 0) {
                foreach ($employee->wageTypes as $wageType) {
                    $typeName = strtolower($wageType->name ?? '');
                    $rate = $wageType->pivot->rate ?? 0;
                    // If wage type name contains 'sin', add to sin rate
                    if (str_contains($typeName, 'sin')) {
                        $sinRate += $rate;
                    } else {
                        // Otherwise add to other rate
                        $otherRate += $rate;
                    }
                }
            }

            return [
                $employee->first_name,
                $employee->last_name,
                $employee->employee_id,
                $employee->email,
                $employee->phone_number,
                $employee->department->name ?? 'N/A',
                $employee->jobRole->name ?? 'N/A',
                number_format($sinRate, 2),
                number_format($otherRate, 2),
                $employee->employment_date,
            ];
        });

        // Get company information
        $company = \App\Models\Company::find($companyId);

        $pdf = Pdf::loadView('pdf.sites_all', [
            'title' => 'Employees List',
            'headers' => $headers,
            'data' => $data,
            'company' => $company
        ])->setPaper('a4', 'landscape');

        return $pdf->download('employees-export-' . now()->format('Y-m-d') . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = \App\Models\Employee::where('company_id', $companyId)
            ->with(['jobRole', 'department', 'wageTypes', 'permanentCountry', 'permanentState']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('employee_id', 'like', "%{$search}%")
                    ->orWhere('phone_number', 'like', "%{$search}%");
            });
        }

        // Filter by Department
        if ($request->filled('department_id')) {
            $query->where('department_id', $request->department_id);
        }

        // Filter by Job Role
        if ($request->filled('job_role_id')) {
            $query->where('job_role_id', $request->job_role_id);
        }

        // Filter by Status
        if ($request->filled('status')) {
            $status = $request->status;
            if ($status === 'active') {
                $query->where('active', true);
            } elseif ($status === 'inactive') {
                $query->where('active', false);
            }
        }

        $employees = $query->latest()->get();

        // Create new Spreadsheet
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set headers
        $headers = [
            'First Name',
            'Last Name',
            'Employee ID',
            'Gender',
            'Email',
            'Phone',
            'Department',
            'Job Role',
            'DOB',
            'Marital Status',
            'Perm. Address 1',
            'Perm. City',
            'License Number',
            'License Expiry',
            'SIN',
            'SIN Rate',
            'Other Rate',
            'Hired Date',
            'Reporting Manager',
            'Created At'
        ];

        // Write headers
        $col = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($col . '1', $header);
            $col++;
        }

        // Style header row - Gray background and bold
        $headerStyle = [
            'font' => [
                'bold' => true,
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => [
                    'rgb' => 'CCCCCC', // Gray color
                ],
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                    'color' => ['rgb' => '000000'],
                ],
            ],
        ];

        $lastColumn = chr(64 + count($headers)); // Convert number to letter (A, B, C, etc.)
        $sheet->getStyle('A1:' . $lastColumn . '1')->applyFromArray($headerStyle);

        // Write data rows
        $row = 2;
        foreach ($employees as $employee) {
            // Calculate SIN Rate and Other Rate
            $sinRate = 0;
            $otherRate = 0;
            if ($employee->wageTypes && $employee->wageTypes->count() > 0) {
                foreach ($employee->wageTypes as $wageType) {
                    $typeName = strtolower($wageType->name ?? '');
                    $rate = $wageType->pivot->rate ?? 0;
                    if (str_contains($typeName, 'sin')) {
                        $sinRate += $rate;
                    } else {
                        $otherRate += $rate;
                    }
                }
            }

            $data = [
                $employee->first_name,
                $employee->last_name,
                $employee->employee_id,
                $employee->gender,
                $employee->email,
                $employee->phone_number,
                $employee->department->name ?? 'N/A',
                $employee->jobRole->name ?? 'N/A',
                $employee->dob,
                $employee->marital_status,
                $employee->permanent_address_line_1,
                $employee->permanent_city,
                $employee->license_number,
                $employee->license_expiry,
                $employee->sin_number,
                number_format($sinRate, 2),
                number_format($otherRate, 2),
                $employee->employment_date,
                $employee->reporting_manager,
                $employee->created_at->format('Y-m-d'),
            ];

            $col = 'A';
            foreach ($data as $value) {
                $sheet->setCellValue($col . $row, $value);
                $col++;
            }

            // Apply borders to data row
            $sheet->getStyle('A' . $row . ':' . $lastColumn . $row)->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        'color' => ['rgb' => '000000'],
                    ],
                ],
            ]);

            $row++;
        }

        // Auto-size columns
        foreach (range('A', $lastColumn) as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }

        // Create Excel file
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $filename = 'employees-export-' . now()->format('Y-m-d') . '.xlsx';

        // Set headers for download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    public function show(\App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        $employee->load(['jobRole', 'department', 'wageTypes', 'identifications', 'permanentCountry', 'permanentState', 'licenceIssuingProvinceState', 'bannedSites', 'issuedUniforms.variant.uniform', 'fcmTokens']);

        $sites = \App\Models\Site::where('company_id', session('selected_company_id'))
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $availableUniforms = \App\Models\UniformVariant::whereHas('uniform', function ($q) {
            $q->where('company_id', session('selected_company_id'))
                ->where('active', true);
        })
            ->with('uniform')
            ->where('active', true)
            ->where('stock_quantity', '>', 0)
            ->get();

        return view('employees.show', compact('employee', 'sites', 'availableUniforms'));
    }

    public function banSites(Request $request, \App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        $request->validate([
            'site_ids' => 'required|array',
            'site_ids.*' => 'exists:sites,id',
            'reason' => 'nullable|string|max:255'
        ]);

        $syncData = [];
        foreach ($request->site_ids as $siteId) {
            $syncData[$siteId] = [
                'reason' => $request->input('reason', 'Manual Ban from Profile'),
                'banned_at' => now()
            ];
        }

        $employee->bannedSites()->syncWithoutDetaching($syncData);

        return back()->with('success', 'Employee has been banned from the selected sites.');
    }

    public function create()
    {
        $provinces = State::where('active', true)->orderBy('name')->get();
        $countries = \App\Models\Country::where('active', true)->orderBy('name')->get();
        $zones = \App\Models\Zone::where('active', true)->orderBy('name')->get();
        $departments = \App\Models\Department::where('active', true)->orderBy('name')->get();
        // Assuming these models exist or will be created, using dummy placeholders if empty
        $jobRoles = \App\Models\JobRole::where('active', true)->orderBy('name')->get();
        $wageTypes = \App\Models\WageType::where('active', true)->orderBy('name')->get();
        $skills = \App\Models\Skill::where('active', true)->orderBy('name')->get();

        // Generate unique Employee ID Logic
        // Generate unique Employee ID Logic
        $companyId = session('selected_company_id');
        $idSetting = \App\Models\IdSetting::firstOrCreate(
            ['company_id' => $companyId],
            ['emp_prefix' => 'EMP', 'emp_next_number' => 1]
        );
        $uniqueId = ($idSetting->emp_prefix ?? '') . $idSetting->emp_next_number;

        return view('employees.create', compact('provinces', 'countries', 'zones', 'departments', 'jobRoles', 'wageTypes', 'skills', 'uniqueId'));
    }

    public function validateForm(Request $request)
    {
        $rules = [
            'first_name' => 'required',
            'email' => 'required|email|unique:employees,email',
            'gender' => 'required|string',
            'job_role_id' => 'required|exists:job_roles,id',
            'phone_number' => 'required|string',
            'license_number' => 'required|string',
            'license_expiry' => 'required|date',
            'file_license' => 'accepted', // Check for true (file selected)
            'file_contract' => 'accepted', // Check for true (file selected)
            'file_id' => 'accepted', // Check for true (file selected)
        ];

        if ($request->has('step')) {
            if ($request->step === 'general') {
                $rules = array_intersect_key($rules, array_flip(['first_name', 'email', 'gender', 'job_role_id']));
            } elseif ($request->step === 'address') {
                $rules = array_intersect_key($rules, array_flip(['phone_number']));
            } elseif ($request->step === 'licence') {
                $rules = array_intersect_key($rules, array_flip(['license_number', 'license_expiry', 'file_license', 'file_contract', 'file_id']));
            } elseif ($request->step === 'other') {
                $rules = [];
            }
        }

        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        return response()->json(['success' => true]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'first_name' => 'required|string|max:255',
            'email' => 'required|email|unique:employees,email',
            'phone_number' => 'required|string|max:50',
            'gender' => 'required|string',
            'job_role_id' => 'required|exists:job_roles,id',
            'license_number' => 'required|string',
            'license_expiry' => 'required|date',
            'file_license' => 'required|file|mimes:pdf,jpg,png,jpeg|max:10240',
            'file_contract' => 'required|file|mimes:pdf,docx,doc|max:10240',
            'file_id.*' => 'required|file|mimes:pdf,jpg,png,jpeg|max:10240',
            'has_car' => 'nullable',
            // Add other validations as necessary
        ]);

        \Illuminate\Support\Facades\DB::transaction(function () use ($request) {
            $companyId = session('selected_company_id');

            // Handle Files
            $licensePath = $request->file('file_license')->store('employees/licenses', 'public');
            $contractPath = $request->file('file_contract')->store('employees/contracts', 'public');
            $directDepositPath = $request->hasFile('file_direct_deposit')
                ? $request->file('file_direct_deposit')->store('employees/deposits', 'public')
                : null;
            $profilePicPath = $request->hasFile('profile_picture')
                ? $request->file('profile_picture')->store('employees/profiles', 'public')
                : null;

            // Create Employee
            $employee = \App\Models\Employee::create([
                'company_id' => $companyId,
                'employee_id' => $request->employee_id,
                'first_name' => $request->first_name,
                'middle_name' => $request->middle_name,
                'last_name' => $request->last_name,
                'gender' => $request->gender,
                'email' => $request->email,
                'dob' => $request->dob,
                'marital_status' => $request->marital_status,
                'zone_id' => $request->zone_id,
                'job_role_id' => $request->job_role_id,
                'department_id' => $request->department_id,
                // sin_rate is now handled in wage_types

                'phone_number' => $request->phone_number,
                'emergency_number' => $request->emergency_number,
                'permanent_address_line_1' => $request->perm_address_1,
                'permanent_address_line_2' => $request->perm_address_2,
                'permanent_city' => $request->perm_city,
                'permanent_state_id' => $request->perm_state_id,
                'permanent_country_id' => $request->perm_country_id,
                'permanent_zip_code' => $request->perm_zip,

                'corresponding_address_line_1' => $request->cors_address_1,
                'corresponding_address_line_2' => $request->cors_address_2,
                'corresponding_city' => $request->cors_city,
                'corresponding_state_id' => $request->cors_state_id,
                'corresponding_country_id' => $request->cors_country_id,
                'corresponding_zip_code' => $request->cors_zip,

                'licence_issuing_province' => $request->licence_province_name ? trim($request->licence_province_name) : null,
                'license_number' => $request->license_number,
                'license_expiry' => $request->license_expiry,
                'file_license' => $licensePath,
                'file_contract' => $contractPath,

                'reporting_manager' => $request->reporting_manager,
                'sin_number' => $request->sin_number,
                'file_direct_deposit' => $directDepositPath,
                'profile_picture' => $profilePicPath,
                'additional_remarks' => $request->additional_remarks,
                'tags' => $request->tags,
                'fax' => $request->fax,
                'employment_date' => $request->employment_date,
                'birthday' => $request->birthday,
                'terminated_date' => $request->terminated_date,
                'has_car' => $request->has('has_car'),
                'password' => Hash::make($request->employee_id),
            ]);

            // Assign role if 'Guard' role exists, otherwise default is 'user'
            if (\Spatie\Permission\Models\Role::where('name', 'Guard')->exists()) {
                $employee->assignRole('Guard');
            }

            // Bind user to the current company (User model call removed, assuming employee->company_id handles association)
            // $user->companies()->syncWithoutDetaching([$companyId]); 
            // NOTE: Employee is strictly company-scoped via column, so no pivot for company usually. The earlier code synced user->companies.

            // Handle IDs (Multiple)
            if ($request->hasFile('file_id')) {
                foreach ($request->file('file_id') as $file) {
                    $path = $file->store('employees/ids', 'public');
                    $employee->identifications()->create(['file_path' => $path]);
                }
            }

            // Handle Wage Types and Rates
            // wage_rates is an array [wage_type_id => rate]
            if ($request->has('wage_rates')) {
                foreach ($request->wage_rates as $wageTypeId => $rate) {
                    $employee->wageTypes()->attach($wageTypeId, ['rate' => $rate]);
                }
            }

            // Handle Experiences
            // Handle Experiences
            if ($request->has('experience')) {
                foreach ($request->experience as $index => $exp) {
                    // Check if at least company name is provided to avoid empty rows
                    if (!empty($exp['company'])) {
                        $proofPath = null;
                        if ($request->hasFile("experience.{$index}.proof")) {
                            $proofPath = $request->file("experience.{$index}.proof")->store('employees/experience_proofs', 'public');
                        }

                        $employee->experiences()->create([
                            'company_name' => $exp['company'] ?? null,
                            'start_date' => $exp['start_date'] ?? null,
                            'end_date' => $exp['end_date'] ?? null,
                            'reason_for_leaving' => $exp['reason'] ?? null,
                            'proof_document' => $proofPath,
                        ]);
                    }
                }
            }

            // Handle Skills
            if ($request->filled('skills')) {
                $employee->skills()->sync($request->skills);
            }

            // Increment ID Setting
            $idSetting = \App\Models\IdSetting::where('company_id', $companyId)->first();
            if ($idSetting) {
                $idSetting->increment('emp_next_number');
            }
        });

        return redirect()->route('employees.index')->with('success', 'Employee created successfully!');
    }

    public function edit(\App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        $provinces = State::where('active', true)->orderBy('name')->get();
        $countries = \App\Models\Country::where('active', true)->orderBy('name')->get();
        $zones = \App\Models\Zone::where('active', true)->orderBy('name')->get();
        $departments = \App\Models\Department::where('active', true)->orderBy('name')->get();
        $jobRoles = \App\Models\JobRole::where('active', true)->orderBy('name')->get();
        $wageTypes = \App\Models\WageType::where('active', true)->orderBy('name')->get();
        $skills = \App\Models\Skill::where('active', true)->orderBy('name')->get();

        $employee->load(['wageTypes', 'skills', 'experiences']);

        $experiencesData = $employee->experiences->map(function ($exp) {
            $start = $exp->start_date;
            $end = $exp->end_date;

            // Check if instance of Carbon/DateTime to format, otherwise use as is
            if ($start instanceof \DateTimeInterface) {
                $start = $start->format('Y-m-d');
            }
            if ($end instanceof \DateTimeInterface) {
                $end = $end->format('Y-m-d');
            }

            return [
                'id' => $exp->id,
                'company' => $exp->company_name,
                'start_date' => $start,
                'end_date' => $end,
                'reason' => $exp->reason_for_leaving,
            ];
        });

        if ($experiencesData->isEmpty()) {
            $experiencesData = [['id' => 'new_' . time()]];
        }

        return view('employees.edit', compact('employee', 'provinces', 'countries', 'zones', 'departments', 'jobRoles', 'wageTypes', 'skills', 'experiencesData'));
    }

    public function update(Request $request, \App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        $request->validate([
            'first_name' => 'required|string|max:255',
            'email' => 'required|email|unique:employees,email,' . $employee->id,
            'phone_number' => 'required|string|max:50',
            'gender' => 'required|string',
            'job_role_id' => 'required|exists:job_roles,id',
            'license_number' => 'required|string',
            'license_expiry' => 'required|date',
            'file_license' => 'nullable|file|mimes:pdf,jpg,png,jpeg|max:10240',
            'file_contract' => 'nullable|file|mimes:pdf,docx,doc|max:10240',
            'file_id.*' => 'nullable|file|mimes:pdf,jpg,png,jpeg|max:10240',
            'has_car' => 'nullable',
        ]);

        \Illuminate\Support\Facades\DB::transaction(function () use ($request, $employee) {

            // Handle Files
            if ($request->hasFile('file_license')) {
                $employee->file_license = $request->file('file_license')->store('employees/licenses', 'public');
            }
            if ($request->hasFile('file_contract')) {
                $employee->file_contract = $request->file('file_contract')->store('employees/contracts', 'public');
            }
            if ($request->hasFile('file_direct_deposit')) {
                $employee->file_direct_deposit = $request->file('file_direct_deposit')->store('employees/deposits', 'public');
            }
            if ($request->hasFile('profile_picture')) {
                $employee->profile_picture = $request->file('profile_picture')->store('employees/profiles', 'public');
            }
            $employee->save();
            if ($request->hasFile('file_id')) {
                foreach ($request->file('file_id') as $file) {
                    $path = $file->store('employees/ids', 'public');
                    $employee->identifications()->create(['file_path' => $path]);
                }
            }

            // Update basic fields
            $employee->update([
                'first_name' => $request->first_name,
                'middle_name' => $request->middle_name,
                'last_name' => $request->last_name,
                'gender' => $request->gender,
                'email' => $request->email,
                'dob' => $request->dob,
                'marital_status' => $request->marital_status,
                'zone_id' => $request->zone_id,
                'job_role_id' => $request->job_role_id,
                'department_id' => $request->department_id,
                'phone_number' => $request->phone_number,
                'emergency_number' => $request->emergency_number,
                'permanent_address_line_1' => $request->perm_address_1,
                'permanent_address_line_2' => $request->perm_address_2,
                'permanent_country_id' => $request->perm_country_id,
                'permanent_city' => $request->perm_city,
                'permanent_state_id' => $request->perm_state_id,
                'permanent_zip_code' => $request->perm_zip,
                'corresponding_address_line_1' => $request->cors_address_1,
                'corresponding_address_line_2' => $request->cors_address_2,
                'corresponding_city' => $request->cors_city,
                'corresponding_state_id' => $request->cors_state_id,
                'corresponding_country_id' => $request->cors_country_id,
                'corresponding_zip_code' => $request->cors_zip,
                'licence_issuing_province' => $request->licence_province_name ? trim($request->licence_province_name) : null,
                'license_number' => $request->license_number,
                'license_expiry' => $request->license_expiry,
                'employment_date' => $request->hiring_date,
                'terminated_date' => $request->termination_date,
                'reporting_manager' => $request->reporting_manager,
                'sin_number' => $request->sin_number,
                'fax' => $request->fax,
                'tags' => $request->tags,
                'has_car' => $request->has('has_car'),
                'additional_remarks' => $request->additional_remarks,
            ]);

            // Sync Wage Types
            if ($request->has('wage_types')) {
                $syncData = [];
                foreach ($request->wage_types as $typeId) {
                    $rate = isset($request->wage_rates[$typeId]) ? $request->wage_rates[$typeId] : 0;
                    $syncData[$typeId] = ['rate' => $rate];
                }
                $employee->wageTypes()->sync($syncData);
            } else {
                $employee->wageTypes()->detach();
            }

            // Sync Skills
            if ($request->has('skills')) {
                $employee->skills()->sync($request->skills);
            } else {
                $employee->skills()->detach();
            }

            // Sync Experiences
            if ($request->has('experience')) {
                $submittedIds = [];
                foreach ($request->experience as $index => $exp) {
                    if (!empty($exp['company'])) {
                        $proofPath = null;
                        if ($request->hasFile("experience.{$index}.proof")) {
                            $proofPath = $request->file("experience.{$index}.proof")->store('employees/experience_proofs', 'public');
                        }

                        $expData = [
                            'company_name' => $exp['company'] ?? null,
                            'start_date' => $exp['start_date'] ?? null,
                            'end_date' => $exp['end_date'] ?? null,
                            'reason_for_leaving' => $exp['reason'] ?? null,
                        ];

                        if ($proofPath) {
                            $expData['proof_document'] = $proofPath;
                        }

                        // Determine if update or create
                        if (isset($exp['id']) && is_numeric($exp['id'])) {
                            $existingExp = $employee->experiences()->find($exp['id']);
                            if ($existingExp) {
                                // If new file uploaded, delete old one? Optional, keeping it simple for now.
                                $existingExp->update($expData);
                                $submittedIds[] = $existingExp->id;
                            }
                        } else {
                            $newExp = $employee->experiences()->create($expData);
                            $submittedIds[] = $newExp->id;
                        }
                    }
                }
                // Delete removed experiences
                $employee->experiences()->whereNotIn('id', $submittedIds)->delete();
            } else {
                // If no experiences sent, delete all
                $employee->experiences()->delete();
            }
        });

        return redirect()->route('employees.index')->with('success', 'Employee updated successfully');
    }

    public function updateAvailability(Request $request, \App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        // We receive the DAYS that represent AVAILABILITY (checked checkboxes).
        $data = $request->validate([
            'available_days' => 'nullable|array',
            'available_days.*' => 'string|in:Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday'
        ]);

        $allDays = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
        $availableDays = $data['available_days'] ?? [];

        // Unavailable days are those NOT in the available_days array
        $unavailableDays = array_values(array_diff($allDays, $availableDays));

        $employee->unavailable_days = $unavailableDays;
        $employee->save();

        return back()->with('success', 'Availability updated successfully.');
    }

    public function assignUniform(Request $request, \App\Models\Employee $employee)
    {
        $request->validate([
            'uniform_variant_id' => 'required|exists:uniform_variants,id',
            'quantity' => 'required|integer|min:1',
            'issued_at' => 'required|date',
            'notes' => 'nullable|string'
        ]);

        $variant = \App\Models\UniformVariant::with('uniform')->findOrFail($request->uniform_variant_id);

        if ($variant->stock_quantity < $request->quantity) {
            return back()->with('error', "Insufficient stock for {$variant->uniform->name} ({$variant->size}/{$variant->color}). Available: {$variant->stock_quantity}");
        }

        \DB::transaction(function () use ($request, $employee, $variant) {
            // Deduct stock
            $variant->decrement('stock_quantity', $request->quantity);

            // Assign to employee
            $employee->issuedUniforms()->create([
                'uniform_variant_id' => $request->uniform_variant_id,
                'quantity' => $request->quantity,
                'issued_at' => $request->issued_at,
                'notes' => $request->notes
            ]);
        });

        return back()->with('success', 'Uniform assigned successfully.');
    }
    public function changePassword(Request $request, \App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        $request->validate([
            'password' => 'required|string|min:8',
        ]);

        $employee->update([
            'password' => Hash::make($request->password),
        ]);

        return back()->with('success', 'Password changed successfully!');
    }

    public function uploadPhoto(Request $request, \App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        $request->validate([
            'profile_picture' => 'required|image|max:2048', // Max 2MB
        ]);

        if ($request->hasFile('profile_picture')) {
            // Delete old photo if exists
            if ($employee->profile_picture) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($employee->profile_picture);
            }

            $path = $request->file('profile_picture')->store('employees/profiles', 'public');
            $employee->update(['profile_picture' => $path]);

            return response()->json([
                'success' => true,
                'message' => 'Profile picture updated.',
                'url' => asset('storage/' . $path)
            ]);
        }

        return response()->json(['success' => false, 'message' => 'No file uploaded.'], 400);
    }

    public function idCard(\App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        $employee->load(['jobRole', 'company']);

        $pdf = Pdf::loadView('employees.id_card', compact('employee'))
            ->setPaper([0, 0, 153, 243], 'portrait') // Standard CR80 Size (2.125" x 3.375")
            ->setOption('isRemoteEnabled', true)
            ->setOption('isHtml5ParserEnabled', true);

        return $pdf->stream($employee->first_name . '_' . $employee->last_name . '_ID_Card.pdf');
    }

    public function schedules(\App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        // Get guard schedules
        $guardSchedules = $employee->schedules()->with('site')->where('schedules.active', true)->get();

        // Get patroller schedules
        $patrollerSchedules = $employee->patrollerSchedules()->with('route')->get();

        $events = [];

        // Map guard schedules
        foreach ($guardSchedules as $schedule) {
            $color = '#64748b'; // Default gray for completed/archived
            if ($schedule->status === 'upcoming')
                $color = '#22c55e'; // Green
            if ($schedule->status === 'missed')
                $color = '#ef4444'; // Red
            if ($schedule->status === 'today')
                $color = '#f59e0b'; // Orange
            if ($schedule->status === 'completed')
                $color = '#3b82f6'; // Blue

            $events[] = [
                'id' => $schedule->id,
                'title' => '[GUARD] ' . $schedule->site->name . " (#" . $schedule->duty_number . ")",
                'start' => $schedule->from_datetime->toIso8601String(),
                'end' => $schedule->to_datetime->toIso8601String(),
                'backgroundColor' => $color,
                'borderColor' => $color,
                'textColor' => '#ffffff',
                'url' => route('schedules.show', $schedule->id),
                'extendedProps' => [
                    'type' => 'guard',
                    'status' => $schedule->status,
                    'location' => $schedule->site->name,
                    'duty_number' => $schedule->duty_number,
                ]
            ];
        }

        // Map patroller schedules
        foreach ($patrollerSchedules as $schedule) {
            // Calculate dynamic status for patroller schedules
            $displayStatus = $schedule->status;
            if ($schedule->status !== 'completed') {
                $now = now();
                $fromTime = $schedule->from_time;
                $toTime = $schedule->to_time;

                if ($fromTime->lte($now) && $toTime->gte($now)) {
                    $displayStatus = 'active';
                } elseif ($toTime->lt($now)) {
                    $displayStatus = 'missed';
                } elseif ($fromTime->isToday() && $fromTime->gt($now)) {
                    $displayStatus = 'today';
                } elseif ($fromTime->isFuture()) {
                    $displayStatus = 'upcoming';
                }
            }

            $color = '#64748b';
            if ($displayStatus === 'upcoming')
                $color = '#8b5cf6'; // Purple for patroller
            if ($displayStatus === 'missed')
                $color = '#ef4444'; // Red
            if ($displayStatus === 'today' || $displayStatus === 'active')
                $color = '#f59e0b'; // Orange
            if ($displayStatus === 'completed')
                $color = '#06b6d4'; // Cyan for patroller

            $events[] = [
                'id' => 'p-' . $schedule->id,
                'title' => '[PATROLLER] ' . $schedule->route->name . " (#" . $schedule->duty_number . ")",
                'start' => $schedule->from_time->toIso8601String(),
                'end' => $schedule->to_time->toIso8601String(),
                'backgroundColor' => $color,
                'borderColor' => $color,
                'textColor' => '#ffffff',
                'url' => route('patroller-schedules.show', $schedule->id),
                'extendedProps' => [
                    'type' => 'patroller',
                    'status' => $displayStatus,
                    'location' => $schedule->route->name,
                    'duty_number' => $schedule->duty_number,
                ]
            ];
        }

        return view('employees.schedules', compact('employee', 'events'));
    }

    public function analytics(\App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            abort(403);
        }

        $rangeStart = now()->subDays(30);
        $lateTolerance = (int) \App\Models\Setting::where('key', 'late_checkin_tolerance')->value('value') ?: 15;

        // --- Guard Metrics ---
        $guardShifts = $employee->schedules()
            ->where('from_datetime', '>=', $rangeStart)
            ->withPivot('actual_start_at', 'actual_end_at')
            ->get();

        $guardStats = [
            'total' => $guardShifts->count(),
            'completed' => 0,
            'missed' => 0,
            'upcoming' => 0,
            'late' => 0,
            'on_time' => 0,
            'total_duration' => 0,
            'duration_count' => 0
        ];

        foreach ($guardShifts as $shift) {
            // Duration
            $guardStats['total_duration'] += $shift->from_datetime->floatDiffInHours($shift->to_datetime);
            $guardStats['duration_count']++;

            // Status & Attendance
            if ($shift->pivot->actual_start_at) {
                // Check Late
                $actual = \Carbon\Carbon::parse($shift->pivot->actual_start_at);
                $scheduled = $shift->from_datetime;
                if ($actual->gt($scheduled->copy()->addMinutes($lateTolerance))) {
                    $guardStats['late']++;
                } else {
                    $guardStats['on_time']++;
                }

                if ($shift->pivot->actual_end_at) {
                    $guardStats['completed']++;
                }
            } else {
                if ($shift->to_datetime->lt(now())) {
                    $guardStats['missed']++;
                } else {
                    $guardStats['upcoming']++;
                }
            }
        }

        $guardAvgDuration = $guardStats['duration_count'] > 0 ? round($guardStats['total_duration'] / $guardStats['duration_count'], 1) : 0;
        $guardAttendanceRate = $guardStats['total'] > 0 ? round(($guardStats['on_time'] / $guardStats['total']) * 100) : 0;


        // --- Patroller Metrics ---
        // Assuming relationship 'patrollerSchedules' exists on Employee
        $patrollerShifts = $employee->patrollerSchedules()
            ->where('patroller_schedules.scheduled_date', '>=', $rangeStart->toDateString())
            ->withPivot('actual_start_at', 'actual_end_at')
            ->get();

        $patrollerStats = [
            'total' => $patrollerShifts->count(),
            'completed' => 0, // Visits completed
            'missed' => 0,
            'late' => 0,
            'on_time' => 0,
            'total_duration' => 0,
            'duration_count' => 0
        ];

        foreach ($patrollerShifts as $shift) {
            // Calculate scheduled duration
            // Assuming from_time / to_time are properly cast or strings
            // We need to create Carbon objects
            try {
                // If from_time is carbon
                $startTime = $shift->from_time instanceof \Carbon\Carbon ? $shift->from_time : \Carbon\Carbon::parse($shift->scheduled_date . ' ' . $shift->from_time);
                $endTime = $shift->to_time instanceof \Carbon\Carbon ? $shift->to_time : \Carbon\Carbon::parse($shift->scheduled_date . ' ' . $shift->to_time);
            } catch (\Exception $e) {
                $startTime = now();
                $endTime = now();
            }

            if ($endTime->lt($startTime))
                $endTime->addDay();

            $patrollerStats['total_duration'] += $startTime->floatDiffInHours($endTime);
            $patrollerStats['duration_count']++;

            // Attendance - check pivot first, then job-level
            $actualStart = $shift->pivot->actual_start_at ?? $shift->job_started_at;
            $actualEnd = $shift->pivot->actual_end_at ?? $shift->job_ended_at;

            if ($actualStart) {
                $actual = \Carbon\Carbon::parse($actualStart);
                $scheduledWithTolerance = $startTime->copy()->addMinutes($lateTolerance);

                if ($actual->gt($scheduledWithTolerance)) {
                    $patrollerStats['late']++;
                } else {
                    $patrollerStats['on_time']++;
                }

                if ($actualEnd) {
                    $patrollerStats['completed']++;
                }
            } else {
                if ($endTime->lt(now())) {
                    $patrollerStats['missed']++;
                }
            }
        }

        $patrollerAvgDuration = $patrollerStats['duration_count'] > 0 ? round($patrollerStats['total_duration'] / $patrollerStats['duration_count'], 1) : 0;
        $patrollerAttendanceRate = $patrollerStats['total'] > 0 ? round(($patrollerStats['on_time'] / $patrollerStats['total']) * 100) : 0;


        // --- Tickets Data (Site Wise) ---
        $siteTickets = \App\Models\PatrollerIssueTicket::where('employee_id', $employee->id)
            ->join('sites', 'patroller_issue_tickets.site_id', '=', 'sites.id')
            ->select('sites.name', \Illuminate\Support\Facades\DB::raw('count(*) as count'))
            ->groupBy('sites.name')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        $siteTicketsLabels = $siteTickets->pluck('name');
        $siteTicketsData = $siteTickets->pluck('count');

        // --- Tickets Data (Monthly) ---
        $monthlyTickets = \App\Models\PatrollerIssueTicket::where('employee_id', $employee->id)
            ->selectRaw("DATE_FORMAT(created_at, '%Y-%m') as month, count(*) as count")
            ->where('created_at', '>=', now()->subMonths(6))
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        $monthlyTicketsLabels = $monthlyTickets->pluck('month');
        $monthlyTicketsData = $monthlyTickets->pluck('count');


        // Comparison Charts
        $assignedComparison = [
            'labels' => ['Guard Jobs', 'Patroller Jobs'],
            'data' => [$guardStats['total'], $patrollerStats['total']]
        ];

        return view('employees.analytics', compact(
            'employee',
            'guardStats',
            'patrollerStats',
            'guardAvgDuration',
            'patrollerAvgDuration',
            'guardAttendanceRate',
            'patrollerAttendanceRate',
            'siteTicketsLabels',
            'siteTicketsData',
            'monthlyTicketsLabels',
            'monthlyTicketsData',
            'assignedComparison'
        ));
    }

    public function toggleStatus(\App\Models\Employee $employee)
    {
        if ($employee->company_id != session('selected_company_id')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $employee->active = !$employee->active;
        $employee->save();

        return response()->json([
            'success' => true,
            'active' => $employee->active,
            'message' => 'Status updated successfully.'
        ]);
    }

    public function copy(Request $request)
    {
        $request->validate([
            'employee_ids' => 'required|array',
            'employee_ids.*' => 'exists:employees,id',
            'target_company_id' => 'required|exists:companies,id',
        ]);

        $targetCompanyId = $request->target_company_id;
        // Verify user has access to target company
        if (!auth()->user()->companies()->where('companies.id', $targetCompanyId)->exists()) {
            return redirect()->back()->with('error', 'You do not have access to the selected company.');
        }

        $employees = \App\Models\Employee::with(['skills', 'wageTypes', 'experiences', 'identifications'])->whereIn('id', $request->employee_ids)->get();
        $count = 0;

        foreach ($employees as $employee) {
            // Check if employee already exists in target company (by email)?
            // If email must be unique globally, we must modify it.
            // But let's check if the existing email is already in use by ANYONE (including in other companies).
            // Logic: Append timestamp if exists.

            $newEmail = $employee->email;
            while (\App\Models\Employee::where('email', $newEmail)->exists()) {
                // If collision, append timestamp
                // Using regex to check if already has timestamp suffix to avoid stacking
                $newEmail = preg_replace('/(\+\d+)?@/', '+' . time() . '@', $employee->email);
                // If still same (fast loop), add random
                if (\App\Models\Employee::where('email', $newEmail)->exists()) {
                    $newEmail = preg_replace('/(\+\d+)?@/', '+' . time() . rand(10, 99) . '@', $employee->email);
                }
            }

            // Replicate
            $newEmployee = $employee->replicate();
            $newEmployee->company_id = $targetCompanyId;
            $newEmployee->employee_id = $this->generateEmployeeId($targetCompanyId);
            $newEmployee->email = $newEmail;
            $newEmployee->created_at = now();
            $newEmployee->updated_at = now();
            $newEmployee->save();

            // Replicate Relations
            // Wage Types
            foreach ($employee->wageTypes as $wt) {
                $newEmployee->wageTypes()->attach($wt->id, ['rate' => $wt->pivot->rate]);
            }

            // Skills
            $newEmployee->skills()->sync($employee->skills->pluck('id'));

            // Experiences
            foreach ($employee->experiences as $exp) {
                $newExp = $exp->replicate();
                $newExp->employee_id = $newEmployee->id;
                $newExp->save();
            }

            // Identifications (Data only)
            foreach ($employee->identifications as $iden) {
                $newIden = $iden->replicate();
                $newIden->employee_id = $newEmployee->id;
                $newIden->save();
            }

            $count++;
        }

        return redirect()->back()->with('success', "$count employees copied successfully.");
    }

    private function generateEmployeeId($companyId)
    {
        $company = \App\Models\Company::find($companyId);
        $companyName = $company ? $company->name : 'Global';

        $prefix = strtoupper(substr($companyName, 0, 1) . 'E');
        // Ensure uniqueness
        do {
            $uniqueId = $prefix . mt_rand(10000, 99999);
        } while (\App\Models\Employee::where('employee_id', $uniqueId)->exists());

        return $uniqueId;
    }

    public function importCSV(Request $request)
    {
        set_time_limit(0); // Prevent timeout for large imports

        $request->validate([
            'csv_file' => 'required|file|mimes:csv,txt|max:5120',
        ]);

        $file = $request->file('csv_file');
        $handle = fopen($file->getRealPath(), 'r');

        // Read header
        $header = fgetcsv($handle);
        if (!$header) {
            return redirect()->back()->with('error', 'Invalid CSV file format.');
        }

        // Clean headers: handle BOM, spaces, and normalize names
        $normalizedHeaders = [];
        foreach ($header as $value) {
            // Remove non-printable characters and handle BOM
            $clean = strtolower(trim(preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $value)));
            // Replace spaces/hyphens with underscores for consistent matching
            $clean = str_replace([' ', '-'], '_', $clean);

            // Map specific user-requested headers to expected keys
            if (strpos($clean, 'employment') !== false)
                $clean = 'employment_date';
            if (strpos($clean, 'alternate') !== false)
                $clean = 'alternate_number';
            if (strpos($clean, 'permanent_address') !== false)
                $clean = 'permanent_address_line';
            if (strpos($clean, 'license_expiry') !== false)
                $clean = 'license_expiry_date';
            if (strpos($clean, 'licence_i') !== false || strpos($clean, 'issuing') !== false)
                $clean = 'licence_issuing';
            if ($clean === 'sin_no')
                $clean = 'sin_no'; // Ensure it's what we expect

            $normalizedHeaders[] = $clean;
        }

        $companyId = session('selected_company_id');
        $importedCount = 0;
        $errors = [];

        // Pre-fetch lookups to avoid N+1 queries in the loop
        $departments = \App\Models\Department::where('company_id', $companyId)->pluck('id', 'name')->mapWithKeys(fn($id, $name) => [strtolower(trim($name)) => $id])->toArray();
        $jobRoles = \App\Models\JobRole::pluck('id', 'name')->mapWithKeys(fn($id, $name) => [strtolower(trim($name)) => $id])->toArray();
        $states = \App\Models\State::select('id', 'name', 'iso2')->get()->flatMap(function ($s) {
            return [
                strtolower(trim($s->name)) => $s->id,
                strtolower(trim($s->iso2)) => $s->id
            ];
        })->toArray();

        \Illuminate\Support\Facades\DB::beginTransaction();
        try {
            while (($row = fgetcsv($handle)) !== false) {
                if (count($normalizedHeaders) !== count($row))
                    continue;

                $data = array_combine($normalizedHeaders, $row);

                if (empty($data['email'])) {
                    continue;
                }

                try {
                    // Determine Department and Job Role IDs (pre-fetched)
                    $deptName = strtolower(trim($data['department'] ?? ''));
                    $departmentId = $departments[$deptName] ?? null;

                    $roleName = strtolower(trim($data['job_role'] ?? ''));
                    $jobRoleId = $jobRoles[$roleName] ?? null;

                    $employeeId = $data['employee_id'] ?? null;
                    if (!$employeeId) {
                        $employeeId = 'EMP-' . strtoupper(\Illuminate\Support\Str::random(6));
                    }

                    // Handle Date formats safely
                    $formatDate = function ($val) {
                        if (empty($val))
                            return null;
                        try {
                            return date('Y-m-d', strtotime($val));
                        } catch (\Exception $e) {
                            return null;
                        }
                    };

                    // Handle State/Province lookup (pre-fetched)
                    $provinceSearch = strtolower(trim($data['licence_issuing'] ?? ''));
                    $provinceId = $states[$provinceSearch] ?? null;

                    $employee = \App\Models\Employee::where('email', trim($data['email']))
                        ->where('company_id', $companyId)
                        ->first();

                    $updateData = [
                        'employee_id' => $employeeId,
                        'first_name' => $data['first_name'] ?? 'Incomplete',
                        'middle_name' => $data['middle_name'] ?? null,
                        'last_name' => $data['last_name'] ?? null,
                        'phone_number' => $data['phone_number'] ?? '',
                        'emergency_number' => $data['alternate_number'] ?? null,
                        'gender' => !empty($data['gender']) ? ucfirst(strtolower(trim($data['gender']))) : 'Other',
                        'dob' => $formatDate($data['dob'] ?? null),
                        'employment_date' => $formatDate($data['employment_date'] ?? null),
                        'permanent_address_line_1' => $data['permanent_address_line'] ?? null,
                        'sin_number' => $data['sin_no'] ?? null,
                        'active' => isset($data['active']) ? (strtolower(trim($data['active'])) === 'yes' || $data['active'] == 1) : true,
                        'license_number' => $data['license_number'] ?? null,
                        'license_expiry' => $formatDate($data['license_expiry_date'] ?? null),
                        'licence_issuing_province_id' => $provinceId,
                        'has_car' => isset($data['car_available']) ? (strtolower(trim($data['car_available'])) === 'yes' || $data['car_available'] == 1) : false,
                        'department_id' => $departmentId,
                        'job_role_id' => $jobRoleId,
                    ];

                    if (!$employee) {
                        $updateData['password'] = \Illuminate\Support\Facades\Hash::make(\Illuminate\Support\Str::random(12));
                        $updateData['email'] = trim($data['email']);
                        $updateData['company_id'] = $companyId;
                        \App\Models\Employee::create($updateData);
                    } else {
                        $employee->update($updateData);
                    }
                    $importedCount++;
                } catch (\Exception $e) {
                    $errors[] = "Error at row " . ($importedCount + 1) . ": " . $e->getMessage();
                }
            }
            \Illuminate\Support\Facades\DB::commit();
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\DB::rollBack();
            $errors[] = "Critical Import Error: " . $e->getMessage();
        }

        fclose($handle);

        $msg = "$importedCount employees processed.";
        if (count($errors) > 0) {
            return redirect()->route('employees.index')->with('success', $msg)->with('error', implode('<br>', array_slice($errors, 0, 5)));
        }

        return redirect()->route('employees.index')->with('success', $msg);
    }

    public function downloadSample()
    {
        $headers = [
            'employee_id',
            'first_name',
            'middle_name',
            'last_name',
            'email',
            'phone_number',
            'alternate_number',
            'gender',
            'dob',
            'employment_date',
            'department',
            'job_role',
            'permanent_address_line',
            'sin_no',
            'active',
            'license_number',
            'license_expiry_date',
            'licence_issuing',
            'car_available'
        ];

        $sampleData = [
            [
                'EMP-1001',
                'John',
                '',
                'Doe',
                'john.doe@example.com',
                '1234567890',
                '0987654321',
                'Male',
                '1990-01-01',
                '2024-01-01',
                'Security',
                'Guard',
                '123 Main St, Toronto, ON',
                '123-456-789',
                'yes',
                'LIC-998877',
                '2026-12-31',
                'Ontario',
                'yes'
            ]
        ];

        $callback = function () use ($headers, $sampleData) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $headers);
            foreach ($sampleData as $row) {
                fputcsv($file, $row);
            }
            fclose($file);
        };

        $fileName = "employee_import_sample.csv";
        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$fileName",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        return response()->stream($callback, 200, $headers);
    }

    public function bulkUpdate(Request $request)
    {
        // Filter out empty wage rows before validation to avoid "required" failures
        $input = $request->all();
        if (isset($input['employees']) && is_array($input['employees'])) {
            foreach ($input['employees'] as $id => &$emp) {
                if (isset($emp['wages']) && is_array($emp['wages'])) {
                    $emp['wages'] = array_filter($emp['wages'], function ($w) {
                        return !empty($w['type_id']) || (isset($w['rate']) && $w['rate'] !== '');
                    });
                    // Re-index array to avoid numeric gaps
                    $emp['wages'] = array_values($emp['wages']);

                    if (empty($emp['wages'])) {
                        unset($emp['wages']);
                    }
                }
            }
        }
        $request->replace($input);

        Log::info('Bulk Update Request Received', ['input' => $request->all()]);

        try {
            $validated = $request->validate([
                'employees' => 'required|array',
                'employees.*.job_role_id' => 'nullable|exists:job_roles,id',
                'employees.*.wages' => 'nullable|array',
                'employees.*.wages.*.type_id' => 'required|exists:wage_types,id',
                'employees.*.wages.*.rate' => 'required|numeric|min:0',
                'employees.*.license_number' => 'nullable|string|max:255',
                'employees.*.license_expiry' => 'nullable|date',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Bulk Update Validation Failed', ['errors' => $e->errors()]);
            throw $e;
        }

        $employeeData = $request->employees;
        $count = count($employeeData);
        $updatedCount = 0;

        \Illuminate\Support\Facades\DB::transaction(function () use ($employeeData, &$updatedCount) {
            foreach ($employeeData as $id => $data) {
                $employee = \App\Models\Employee::find($id);
                if (!$employee) {
                    Log::warning("Bulk Update: Employee not found with ID $id");
                    continue;
                }

                $updateData = [];

                // Use array_key_exists to allow clearing values (if sent as null/empty)
                if (array_key_exists('job_role_id', $data)) {
                    $updateData['job_role_id'] = $data['job_role_id'] ?: null;
                }

                if (array_key_exists('license_number', $data)) {
                    $updateData['license_number'] = $data['license_number'];
                }
                if (array_key_exists('license_expiry', $data)) {
                    $updateData['license_expiry'] = $data['license_expiry'] ?: null;
                }

                if (!empty($updateData)) {
                    Log::info("Updating Employee $id", ['updateData' => $updateData]);
                    $employee->update($updateData);
                } else {
                    Log::info("No update data for Employee $id");
                }

                if (!empty($data['wages'])) {
                    $syncData = [];
                    foreach ($data['wages'] as $wage) {
                        if (!empty($wage['type_id']) && isset($wage['rate']) && $wage['rate'] !== '') {
                            $syncData[$wage['type_id']] = ['rate' => $wage['rate']];
                        }
                    }
                    if (!empty($syncData)) {
                        $employee->wageTypes()->syncWithoutDetaching($syncData);
                    }
                }
                $updatedCount++;
            }
        });

        return redirect()->back()->with('success', "$updatedCount employees processed for update.");
    }
}
