<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\PatrollerSchedule;
use App\Models\PatrollerJobSite;
use App\Models\PatrollerIssueTicket;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use ZipArchive;
use App\Services\ImageService;

class PatrollerJobController extends Controller
{
    /**
     * List all patroller jobs
     */
    public function index(Request $request)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            Auth::guard('employee')->logout();
            return redirect()->route('employee.login')->withErrors(['email' => 'Session invalid.']);
        }
        $type = $request->query('type', 'all');
        $today = Carbon::today();

        $query = $employee->patrollerSchedules()->with(['company', 'route', 'employees', 'jobSites', 'issueTickets']);

        switch ($type) {
            case 'active':
                $query->active()->orderBy('patroller_schedules.job_started_at', 'desc');
                break;
            case 'today':
                $query->todayUpcoming();
                break;
            case 'upcoming':
                $query->upcomingFuture();
                break;
            case 'completed':
                $query->completed();
                break;
            case 'missed':
                $query->missed();
                break;
            case 'pending':
                $query->pendingApproval();
                break;
            case 'cancelled':
                $query->cancelled();
                break;
        }

        $jobs = $query->orderBy('patroller_schedules.from_time')->paginate(100);

        // Counts using standardized scopes
        $counts = [
            'all' => $employee->patrollerSchedules()->count(),
            'active' => $employee->patrollerSchedules()->active()->count(),
            'today' => $employee->patrollerSchedules()->todayUpcoming()->count(),
            'upcoming' => $employee->patrollerSchedules()->upcomingFuture()->count(),
            'completed' => $employee->patrollerSchedules()->completed()->count(),
            'missed' => $employee->patrollerSchedules()->missed()->count(),
            'pending' => $employee->patrollerSchedules()->pendingApproval()->count(),
            'cancelled' => $employee->patrollerSchedules()->cancelled()->count(),
        ];

        return view('employee.patroller-jobs.index', compact('jobs', 'type', 'counts', 'employee'));
    }

    /**
     * Show patroller job history
     */
    public function history()
    {
        $employee = Auth::guard('employee')->user();
        $completedJobs = $employee->patrollerSchedules()
            ->with(['route', 'jobSites'])
            ->whereIn('job_status', ['completed', 'auto_ended'])
            ->orderBy('from_time', 'desc')
            ->paginate(10);

        return view('employee.patroller-jobs.history', compact('completedJobs'));
    }

    /**
     * Show job details with site list
     */
    public function show(PatrollerSchedule $schedule)
    {
        $employee = Auth::guard('employee')->user();

        // Validate employee is assigned to this schedule
        if (!$schedule->employees->contains($employee->id)) {
            abort(403, 'You are not assigned to this schedule');
        }

        $schedule->load(['route.sites', 'jobSites.site', 'jobSites.issueTickets']);

        // Check if employee is currently checked in anywhere
        $activeCheckIn = PatrollerJobSite::where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->with('site')
            ->first();

        // Get visibility settings
        $visibilityMinutes = 0;
        if ($employee->job_role_id) {
            $visibilitySetting = \App\Models\CompanyJobRoleVisibility::where('company_id', $schedule->company_id)
                ->where('job_role_id', $employee->job_role_id)
                ->first();
            if ($visibilitySetting) {
                $visibilityMinutes = $visibilitySetting->visibility_minutes;
            }
        }

        return view('employee.patroller-jobs.show', compact('schedule', 'activeCheckIn', 'visibilityMinutes'));
    }

    /**
     * Start a patroller job
     */
    public function start(Request $request, PatrollerSchedule $schedule)
    {
        $employee = Auth::guard('employee')->user();

        // Validate employee is assigned to this schedule
        if (!$schedule->employees->contains($employee->id)) {
            return response()->json(['error' => 'You are not assigned to this schedule'], 403);
        }

        // Check if job already started
        if ($schedule->isJobStarted()) {
            return response()->json(['error' => 'Job already started'], 400);
        }

        // Validate location is provided
        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);

        $now = now();
        $actualStartAt = $now->gt($schedule->from_time) ? $now : $schedule->from_time;

        // Start the job
        $schedule->update([
            'job_started_at' => $actualStartAt,
            'job_status' => 'in_progress',
        ]);

        // Also update the individual deployment start for this employee
        $schedule->employees()->updateExistingPivot($employee->id, [
            'actual_start_at' => $actualStartAt,
        ]);

        // Create job site records for all sites in the route
        foreach ($schedule->route->sites as $site) {
            PatrollerJobSite::create([
                'patroller_schedule_id' => $schedule->id,
                'site_id' => $site->id,
                'employee_id' => $employee->id,
                'status' => 'pending',
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Job started successfully',
            'redirect' => route('employee.patroller-jobs.show', $schedule->id),
        ]);
    }

    public function requestCancellation(Request $request, PatrollerSchedule $schedule)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        // Validate employee is assigned to this schedule
        if (!$schedule->employees->contains($employee->id)) {
            abort(403);
        }

        // Check if schedule is truly upcoming
        if ($schedule->from_time->isPast()) {
            return back()->with('error', 'Cancellation requests are only allowed for future schedules that have not started.');
        }

        $schedule->employees()->updateExistingPivot($employee->id, [
            'cancellation_requested_at' => now(),
            'cancellation_reason' => $request->reason,
            'cancellation_status' => 'pending'
        ]);

        // Notify Company Users
        foreach ($employee->company->users as $user) {
            \App\Models\Notification::create([
                'company_id' => $employee->company_id,
                'user_id' => $user->id,
                'title' => 'New Patroller Cancellation Request',
                'message' => "Employee {$employee->first_name} {$employee->last_name} has requested to cancel Patroller Duty #{$schedule->duty_number}.",
                'url' => route('patroller-schedules.index', ['type' => 'all', 'search' => $schedule->duty_number]),
            ]);
        }

        return back()->with('success', 'Cancellation request submitted for review.');
    }

    /**
     * Check in to a site
     */
    public function checkin(Request $request, PatrollerSchedule $schedule, $siteId)
    {
        $employee = Auth::guard('employee')->user();

        // Prevent concurrent check-ins
        $activeCheckin = PatrollerJobSite::where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->where('site_id', '!=', $siteId) // Allow re-checkin logic to handle same-site (though it usually errors)
            ->with(['site'])
            ->first();

        if ($activeCheckin) {
            $siteName = $activeCheckin->site->name ?? 'another site';
            return response()->json([
                'error' => "You are currently checked in at {$siteName}. Please checkout there first."
            ], 400);
        }

        $rules = [
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ];

        $request->validate($rules);

        // Images are no longer required for patroller check-ins
        $imagePaths = null;

        // Find the latest job site record
        $latestRecord = PatrollerJobSite::where('patroller_schedule_id', $schedule->id)
            ->where('site_id', $siteId)
            ->where('employee_id', $employee->id)
            ->latest()
            ->first();

        $now = now();
        $actualStartAt = $now->gt($schedule->from_time) ? $now : $schedule->from_time;

        // Auto-start job if needed
        if ($schedule->job_status === 'pending') {
            $schedule->update([
                'job_started_at' => $actualStartAt,
                'job_status' => 'in_progress'
            ]);
        }

        // Ensure this individual employee's deployment is marked as started
        $pivot = $schedule->employees()->where('employees.id', $employee->id)->first()->pivot;
        if (!$pivot->actual_start_at) {
            $schedule->employees()->updateExistingPivot($employee->id, [
                'actual_start_at' => $actualStartAt,
            ]);
        }

        if (!$latestRecord) {
            // Should not happen if start() was called correctly, but fallback: create new
            $jobSite = PatrollerJobSite::create([
                'patroller_schedule_id' => $schedule->id,
                'site_id' => $siteId,
                'employee_id' => $employee->id,
                'status' => 'checked_in',
                'checked_in_at' => now(),
                'checkin_latitude' => $request->latitude,
                'checkin_longitude' => $request->longitude,
                'checkin_images' => $imagePaths,
            ]);
        } elseif ($latestRecord->status === 'pending') {
            // First visit (using the pending record created at start)
            $latestRecord->update([
                'status' => 'checked_in',
                'checked_in_at' => now(),
                'checkin_latitude' => $request->latitude,
                'checkin_longitude' => $request->longitude,
                'checkin_images' => $imagePaths,
            ]);
            $jobSite = $latestRecord;
        } elseif ($latestRecord->status === 'checked_out') {
            // Subsequent visit: Create NEW record
            $jobSite = PatrollerJobSite::create([
                'patroller_schedule_id' => $schedule->id,
                'site_id' => $siteId,
                'employee_id' => $employee->id,
                'status' => 'checked_in',
                'checked_in_at' => now(),
                'checkin_latitude' => $request->latitude,
                'checkin_longitude' => $request->longitude,
                'checkin_images' => $imagePaths,
            ]);
        } else {
            // Already checked in or other state
            return response()->json(['error' => 'Already checked in to this site'], 400);
        }

        return response()->json([
            'success' => true,
            'message' => 'Checked in successfully',
            'job_site' => $jobSite,
        ]);
    }

    /**
     * Create an issue ticket
     */
    public function createTicket(Request $request, PatrollerSchedule $schedule, $siteId)
    {
        $employee = Auth::guard('employee')->user();

        $request->validate([
            'ticket_number' => 'required|string|max:50',
            'description' => 'required|string|max:1000',
            'status' => 'required|in:good,bad',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            'images' => 'nullable|array',
            'images.*' => 'image|max:5120',
        ]);

        // Find the active job site record (must be checked_in)
        $jobSite = PatrollerJobSite::where('patroller_schedule_id', $schedule->id)
            ->where('site_id', $siteId)
            ->where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->with('site')
            ->latest()
            ->firstOrFail();

        // Upload images
        $imagePaths = [];
        if ($request->hasFile('images')) {
            $folderPath = ImageService::getPatrollerFolderPath($schedule->duty_number, $jobSite->site->name);
            foreach ($request->file('images') as $image) {
                $path = ImageService::processAndStore($image, $folderPath);
                if ($path) {
                    $imagePaths[] = $path;
                }
            }
        }

        // Create ticket
        $ticket = PatrollerIssueTicket::create([
            'patroller_job_site_id' => $jobSite->id,
            'patroller_schedule_id' => $schedule->id,
            'site_id' => $siteId,
            'employee_id' => $employee->id,
            'ticket_number' => $request->ticket_number,
            'description' => $request->description,
            'status' => $request->status,
            'images' => $imagePaths,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Ticket created successfully',
            'ticket' => $ticket,
        ]);
    }

    /**
     * Check out from a site
     */
    public function checkout(Request $request, PatrollerSchedule $schedule, $siteId)
    {
        $employee = Auth::guard('employee')->user();

        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);

        // Find the active job site record
        $jobSite = PatrollerJobSite::where('patroller_schedule_id', $schedule->id)
            ->where('site_id', $siteId)
            ->where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->latest()
            ->first();

        // Validate site is checked in
        if (!$jobSite) {
            return response()->json(['error' => 'Site not checked in'], 400);
        }

        // Update job site
        $jobSite->update([
            'checked_out_at' => now(),
            'checkout_latitude' => $request->latitude,
            'checkout_longitude' => $request->longitude,
            'status' => 'checked_out',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Checked out successfully',
            'job_site' => $jobSite,
        ]);
    }

    /**
     * End the job
     */
    public function endJob(Request $request, PatrollerSchedule $schedule)
    {
        $employee = Auth::guard('employee')->user();

        // Validate employee is assigned
        if (!$schedule->employees->contains($employee->id)) {
            return response()->json(['error' => 'You are not assigned to this schedule'], 403);
        }

        // Check if job is in progress
        if (!$schedule->isJobInProgress()) {
            return response()->json(['error' => 'Job is not in progress'], 400);
        }

        // Update the pivot table for the current employee
        $schedule->employees()->updateExistingPivot($employee->id, [
            'actual_end_at' => now(),
        ]);

        // Check if ALL employees have finished
        $allFinished = true;
        foreach ($schedule->employees as $emp) {
            // Reload pivot data to get fresh values, or check logic carefully
            // We need to check the pivot 'actual_end_at' for all employees
            // Since we just updated the current one, we can trust the DB result for others.
            // A more efficient query:
            $pivot = $emp->pivot; // This might be stale if loaded earlier. 
        }

        // Better check: Count employees where actual_end_at is NULL
        $activeEmployeesCount = $schedule->employees()
            ->wherePivot('actual_end_at', null)
            ->count();

        // If count is 0, everyone has finished (since we just updated the current user, they won't be null anymore ideally, 
        // strictly speaking we should check if the update worked first, but updateExistingPivot is direct).
        // Wait, 'employees()' relationship returns a query builder.

        if ($activeEmployeesCount === 0) {
            // All finished
            $schedule->update([
                'job_ended_at' => now(),
                'job_status' => 'completed',
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Job ended successfully',
            'redirect' => route('employee.dashboard'),
        ]);
    }

    /**
     * Show job report
     */
    public function report(PatrollerSchedule $schedule)
    {
        $employee = Auth::guard('employee')->user();

        // Load all data
        $schedule->load([
            'route.sites',
            'jobSites.site',
            'issueTickets.site',
        ]);

        // Statistics
        $totalTickets = $schedule->issueTickets->count();
        $goodTickets = $schedule->issueTickets->where('status', 'good')->count();
        $badTickets = $schedule->issueTickets->where('status', 'bad')->count();

        // Group tickets by site
        $ticketsBySite = $schedule->issueTickets->groupBy('site_id');

        return view('employee.patroller-jobs.report', compact(
            'schedule',
            'totalTickets',
            'goodTickets',
            'badTickets',
            'ticketsBySite'
        ));
    }

    /**
     * Export tickets as ZIP
     */
    public function exportTickets(PatrollerSchedule $schedule)
    {
        $employee = Auth::guard('employee')->user();

        $schedule->load(['route.sites', 'issueTickets.site']);

        $dutyId = $schedule->duty_number ?? $schedule->id;
        $zipFileName = "Patroller_Evidence_Duty_{$dutyId}.zip";
        $zipPath = storage_path("app/public/temp/{$zipFileName}");

        // Ensure temp directory exists
        if (!file_exists(dirname($zipPath))) {
            mkdir(dirname($zipPath), 0755, true);
        }

        // Create ZIP
        $zip = new ZipArchive();
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
            return response()->json(['error' => 'Could not create ZIP file'], 500);
        }

        // Grouping logic for the ZIP structure
        foreach ($schedule->issueTickets as $ticket) {
            $siteName = $ticket->site->name ?? 'Unknown_Site';
            $cleanSiteName = str_replace([' ', '/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $siteName);

            // Structure: patroller_images/tickets_issued_{dutyId}/{dutyId}_{siteName}/
            $ticketBasePath = "patroller_images/tickets_issued_{$dutyId}/{$dutyId}_{$cleanSiteName}";

            // Add ticket description as a text file
            $ticketInfo = "Ticket Number: {$ticket->ticket_number}\n";
            $ticketInfo .= "Status: {$ticket->status}\n";
            $ticketInfo .= "Description: {$ticket->description}\n";
            $ticketInfo .= "Location: {$ticket->latitude}, {$ticket->longitude}\n";
            $ticketInfo .= "Reported At: {$ticket->created_at}\n";

            $zip->addFromString("{$ticketBasePath}/ticket_{$ticket->ticket_number}.txt", $ticketInfo);

            // Add images
            if ($ticket->images && is_array($ticket->images)) {
                foreach ($ticket->images as $index => $imagePath) {
                    $fullPath = storage_path("app/public/{$imagePath}");
                    if (file_exists($fullPath)) {
                        $extension = pathinfo($imagePath, PATHINFO_EXTENSION);
                        $zip->addFile($fullPath, "{$ticketBasePath}/img_{$index}.{$extension}");
                    }
                }
            }
        }

        $zip->close();

        return response()->download($zipPath, $zipFileName)->deleteFileAfterSend(true);
    }

    /**
     * Export report as PDF
     */
    public function exportPdf(PatrollerSchedule $schedule)
    {
        $employee = Auth::guard('employee')->user();

        // Load all data
        $schedule->load([
            'route.sites',
            'jobSites.site',
            'issueTickets.site',
        ]);

        // Statistics
        $totalTickets = $schedule->issueTickets->count();
        $goodTickets = $schedule->issueTickets->where('status', 'good')->count();
        $badTickets = $schedule->issueTickets->where('status', 'bad')->count();

        // Group tickets by site
        $ticketsBySite = $schedule->issueTickets->groupBy('site_id');

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('employee.patroller-jobs.pdf', compact(
            'schedule',
            'totalTickets',
            'goodTickets',
            'badTickets',
            'ticketsBySite'
        ))->setOptions([
                    'tempDir' => public_path(),
                    'chroot' => public_path(),
                    'isHtml5ParserEnabled' => true,
                    'isRemoteEnabled' => true,
                    'defaultFont' => 'DejaVu Sans',
                    'dpi' => 72, // Reduced DPI to decrease image file size
                    'defaultCompression' => true // Enable internal PDF compression
                ]);

        return $pdf->download("Patrol_Report_{$schedule->duty_number}.pdf");
    }
    public function getRelieverInfo(PatrollerSchedule $schedule)
    {
        $nextSchedule = PatrollerSchedule::where('route_id', $schedule->route_id)
            ->where('from_time', '>=', $schedule->to_time->subMinutes(30))
            ->where('id', '!=', $schedule->id)
            ->whereNotIn('status', ['cancelled']) // Assuming 'status' helps filter valid schedules
            // ->whereNotIn('job_status', ['cancelled']) // Or job_status?
            ->with(['employees'])
            ->orderBy('from_time', 'asc')
            ->first();

        if (!$nextSchedule) {
            return response()->json(null);
        }

        $currentEmployeeId = Auth::guard('employee')->id();
        $reliever = $nextSchedule->employees->where('id', '!=', $currentEmployeeId)->first();

        if (!$reliever) {
            $reliever = $nextSchedule->employees->first();
            if ($reliever && $reliever->id == $currentEmployeeId) {
                return response()->json(['self_relief' => true]);
            }
            return response()->json([
                'site_name' => $schedule->route->name,
                'start_time' => $nextSchedule->from_time->format('H:i'),
                'employee' => null
            ]);
        }

        return response()->json([
            'site_name' => $schedule->route->name,
            'start_time' => $nextSchedule->from_time->format('H:i, M d'),
            'employee' => [
                'name' => $reliever->first_name . ' ' . $reliever->last_name,
                'photo' => $reliever->profile_picture_url,
                'phone' => $reliever->phone_number,
                'id' => $reliever->employee_id
            ]
        ]);
    }

    public function storeNote(Request $request, $id)
    {
        $request->validate([
            'note' => 'required|string',
            'job_type' => 'required|in:guard,patroller',
            'image' => 'nullable|image|max:5120',
        ]);

        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return response()->json(['error' => 'Unauthorized'], 401);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $schedule = \App\Models\PatrollerSchedule::findOrFail($id);
            $folderPath = \App\Services\ImageService::getFolderPath('patroller-job-notes', $schedule->duty_number, 'Patrol Route', now());
            $imagePath = \App\Services\ImageService::processAndStore($request->file('image'), $folderPath);
        }

        \App\Models\JobNote::create([
            'employee_id' => $employee->id,
            'job_type' => $request->job_type,
            'job_id' => $id,
            'note' => $request->note,
            'image' => $imagePath,
        ]);

        return response()->json(['success' => true, 'message' => 'Note saved successfully.']);
    }

    public function getNotes($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return response()->json(['error' => 'Unauthorized'], 401);

        $notes = \App\Models\JobNote::where('job_id', $id)
            ->where('job_type', 'patroller')
            ->latest()
            ->get();

        return response()->json(['success' => true, 'notes' => $notes]);
    }
}
