<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\Uniform;
use App\Models\UniformVariant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EmployeeUniformController extends Controller
{
    public function index()
    {
        if (Auth::guard('web')->check()) {
            return $this->adminIndex();
        }

        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            abort(403, 'Unauthorized');
        }

        $inventory = UniformVariant::with('uniform')
            ->whereHas('uniform', function ($q) use ($employee) {
                $q->where('company_id', $employee->company_id)
                    ->where('active', true);
            })
            ->where('active', true)
            ->orderBy('uniform_id')
            ->get();

        $myUniforms = $employee->issuedUniforms()->with('variant.uniform')->latest('issued_at')->get();

        return view('employee.uniforms.index', compact('inventory', 'myUniforms'));
    }

    public function exportStockPdf()
    {
        $companyId = session('selected_company_id');

        $inventory = UniformVariant::with('uniform')
            ->whereHas('uniform', function ($q) use ($companyId) {
                $q->where('company_id', $companyId)
                    ->where('active', true);
            })
            ->where('active', true)
            ->orderBy('uniform_id')
            ->get();

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('reports.uniforms.stock_pdf', compact('inventory'));
        return $pdf->download('Uniform_Stock_Levels_' . now()->format('Ymd_Hi') . '.pdf');
    }

    private function adminIndex()
    {
        $companyId = session('selected_company_id');

        $inventory = UniformVariant::with('uniform')
            ->whereHas('uniform', function ($q) use ($companyId) {
                $q->where('company_id', $companyId)
                    ->where('active', true);
            })
            ->where('active', true)
            ->orderBy('uniform_id')
            ->get();

        // Fetch all issued uniforms for the company
        $allIssuedUniforms = \App\Models\EmployeeUniform::whereHas('employee', function ($q) use ($companyId) {
            $q->where('company_id', $companyId);
        })
            ->with(['employee', 'variant.uniform'])
            ->latest('issued_at')
            ->get();

        // We can reuse the view if we adapt it or create a new one. 
        // Given the request "show ... which uniform is assigned to which employee", a new view is better.
        // Let's create 'user.uniforms.index' view.
        return view('user.uniforms.index', compact('inventory', 'allIssuedUniforms'));
    }
    public function assign(Request $request)
    {
        $employee = Auth::guard('employee')->user();

        $request->validate([
            'uniform_variant_id' => 'required|exists:uniform_variants,id',
            'quantity' => 'required|integer|min:1',
            'issued_at' => 'required|date',
            'notes' => 'nullable|string'
        ]);

        $variant = UniformVariant::with('uniform')->findOrFail($request->uniform_variant_id);

        // Security check: ensure variant belongs to employee's company
        if ($variant->uniform->company_id != $employee->company_id) {
            abort(403, 'Unauthorized access to uniform item.');
        }

        if ($variant->stock_quantity < $request->quantity) {
            return back()->with('error', "Insufficient stock for {$variant->uniform->name}. Available: {$variant->stock_quantity}");
        }

        \Illuminate\Support\Facades\DB::transaction(function () use ($request, $employee, $variant) {
            // Deduct stock
            $variant->decrement('stock_quantity', $request->quantity);

            // Assign to employee
            $employee->issuedUniforms()->create([
                'uniform_variant_id' => $request->uniform_variant_id,
                'quantity' => $request->quantity,
                'issued_at' => $request->issued_at,
                'notes' => $request->notes
            ]);
        });

        return back()->with('success', 'Uniform assigned successfully.');
    }
}
