<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;

class EmployeeReportController extends Controller
{
    public function index(Request $request)
    {
        $employee = Auth::guard('employee')->user();
        $reportType = $request->query('type', 'checkin');

        $query = $this->buildQuery($employee, $request);
        $reports = $query->paginate($reportType === 'scans_history' ? 40 : 20);

        // Get unique sites this employee has worked at for the filter
        $sites = $employee->schedules()
            ->with('site')
            ->get()
            ->pluck('site')
            ->whereNotNull()
            ->unique('id')
            ->sortBy('name');

        return view('employee.reports.index', compact('reports', 'reportType', 'sites'));
    }

    public function exportPdf(Request $request)
    {
        $employee = Auth::guard('employee')->user();
        $reportType = $request->query('type', 'checkin');

        $query = $this->buildQuery($employee, $request);
        $reports = $query->get();

        $view = $reportType === 'scans_history' ? 'employee.reports.scans_pdf' : 'employee.reports.pdf';
        $pdf = Pdf::loadView($view, compact('reports', 'reportType', 'employee'));
        return $pdf->download("Operational_Report_{$reportType}_" . now()->format('Ymd_Hi') . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $employee = Auth::guard('employee')->user();
        $reportType = $request->query('type', 'checkin');

        $query = $this->buildQuery($employee, $request);
        $reports = $query->get();

        $filename = "Operational_Report_{$reportType}_" . now()->format('Ymd_Hi') . ".csv";

        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        if ($reportType === 'scans_history') {
            $columns = ['Date/Time', 'Site Name', 'Duty #', 'Checkpoint', 'Type', 'Message', 'Manual Scan', 'Photos Count'];
        } else {
            $columns = $reportType === 'scans'
                ? ['Duty Number', 'Site Name', 'Scheduled Start', 'Scheduled End', 'Total Scans', 'Actual Start', 'Actual End', 'Status']
                : ['Duty Number', 'Site Name', 'Scheduled Start', 'Scheduled End', 'Actual Start', 'Actual End', 'Status'];
        }

        $callback = function () use ($reports, $columns, $reportType) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($reports as $item) {
                if ($reportType === 'scans_history') {
                    fputcsv($file, [
                        $item->scanned_at->format('Y-m-d H:i'),
                        $item->schedule->site->name ?? 'N/A',
                        $item->schedule->duty_number ?? 'N/A',
                        $item->checkpoint->name ?? 'N/A',
                        ucfirst(str_replace('_', ' ', $item->checkpoint->extra_scan_options['type'] ?? 'Log Only')),
                        $item->employee_message ?? 'N/A',
                        $item->is_manual ? 'Yes' : 'No',
                        count($item->additional_photos ?? []) + ($item->evidence_image ? 1 : 0)
                    ]);
                } elseif ($reportType === 'scans') {
                    fputcsv($file, [
                        $item->duty_number,
                        $item->site->name ?? 'Unknown',
                        $item->from_datetime->format('Y-m-d H:i'),
                        $item->to_datetime->format('Y-m-d H:i'),
                        $item->checkpointScans->count(),
                        $item->pivot->actual_start_at ? Carbon::parse($item->pivot->actual_start_at)->format('Y-m-d H:i') : 'N/A',
                        $item->pivot->actual_end_at ? Carbon::parse($item->pivot->actual_end_at)->format('Y-m-d H:i') : 'N/A',
                        ucfirst($item->status)
                    ]);
                } else {
                    fputcsv($file, [
                        $item->duty_number,
                        $item->site->name ?? 'Unknown',
                        $item->from_datetime->format('Y-m-d H:i'),
                        $item->to_datetime->format('Y-m-d H:i'),
                        $item->pivot->actual_start_at ? Carbon::parse($item->pivot->actual_start_at)->format('Y-m-d H:i') : 'N/A',
                        $item->pivot->actual_end_at ? Carbon::parse($item->pivot->actual_end_at)->format('Y-m-d H:i') : 'N/A',
                        ucfirst($item->status)
                    ]);
                }
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function buildQuery($employee, Request $request)
    {
        $reportType = $request->query('type', 'checkin');

        if ($reportType === 'scans_history') {
            $query = $employee->checkpointScans()
                ->with(['checkpoint', 'schedule.site', 'tourRoute'])
                ->orderBy('scanned_at', 'desc');

            if ($request->filled('start_date')) {
                $query->whereDate('scanned_at', '>=', $request->start_date);
            }
            if ($request->filled('end_date')) {
                $query->whereDate('scanned_at', '<=', $request->end_date);
            }
            if ($request->filled('site_id')) {
                $query->whereHas('schedule', function ($q) use ($request) {
                    $q->where('site_id', $request->site_id);
                });
            }
            if ($request->filled('search')) {
                $search = $request->search;
                $query->whereHas('schedule', function ($q) use ($search) {
                    $q->where('duty_number', 'like', "%{$search}%");
                });
            }
            if ($request->filled('schedule_id')) {
                $query->where('schedule_id', $request->schedule_id);
            }

            return $query;
        }

        $query = $employee->schedules()
            ->with([
                'site',
                'checkpointScans' => function ($q) use ($employee) {
                    $q->where('employee_id', $employee->id);
                }
            ])
            ->whereNotNull('employee_schedule.actual_start_at')
            ->orderBy('schedule_date', 'desc');

        if ($reportType === 'checkout') {
            $query->whereNotNull('employee_schedule.actual_end_at');
        }

        if ($reportType === 'scans') {
            $query->whereHas('checkpointScans', function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            });
        }

        // Filters
        if ($request->filled('start_date')) {
            $query->whereDate('schedule_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('schedule_date', '<=', $request->end_date);
        }
        if ($request->filled('site_id')) {
            $query->where('site_id', $request->site_id);
        }
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('duty_number', 'like', "%{$search}%");
        }
        if ($request->filled('schedule_id')) {
            $query->where('schedules.id', $request->schedule_id);
        }

        return $query;
    }
}
