<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class EmployeePasswordResetController extends Controller
{
    public function showLinkRequestForm()
    {
        return view('employee.auth.passwords.email');
    }

    public function sendResetLinkEmail(Request $request)
    {
        $request->validate(['email' => 'required|email']);

        $employee = \App\Models\Employee::where('email', $request->email)->first();

        if (!$employee) {
            return back()->withErrors(['email' => 'We can\'t find a guard with that email address.']);
        }

        $token = \Illuminate\Support\Str::random(60);

        \Illuminate\Support\Facades\DB::table('password_reset_tokens')->updateOrInsert(
            ['email' => $request->email],
            ['token' => \Illuminate\Support\Facades\Hash::make($token), 'created_at' => now()]
        );

        // Send Email
        $resetLink = route('employee.password.reset', ['token' => $token, 'email' => $request->email]);

        try {
            \Illuminate\Support\Facades\Mail::send('emails.employee.password_reset', ['resetLink' => $resetLink, 'employee' => $employee], function ($message) use ($request) {
                $message->to($request->email);
                $message->subject('Reset Password Notification');
            });
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Employee Password Reset Email Error: ' . $e->getMessage());
            return back()->withErrors(['email' => 'Failed to send reset link. Please try again later.']);
        }

        return back()->with('status', 'We have e-mailed your password reset link!');
    }

    public function showResetForm(Request $request, $token = null)
    {
        return view('employee.auth.passwords.reset')->with(
            ['token' => $token, 'email' => $request->email]
        );
    }

    public function reset(Request $request)
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|confirmed|min:8',
        ]);

        $record = \Illuminate\Support\Facades\DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->first();

        if (!$record || !\Illuminate\Support\Facades\Hash::check($request->token, $record->token)) {
            return back()->withErrors(['email' => 'Invalid or expired password reset token.']);
        }

        $employee = \App\Models\Employee::where('email', $request->email)->first();

        if (!$employee) {
            return back()->withErrors(['email' => 'We can\'t find a guard with that email address.']);
        }

        $employee->password = \Illuminate\Support\Facades\Hash::make($request->password);
        $employee->save();

        // Delete token
        \Illuminate\Support\Facades\DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        return redirect()->route('employee.login')->with('status', 'Your password has been reset successfully! You can now login with your new password.');
    }
}
