<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class EmployeeAnalyticsController extends Controller
{
    use \App\Traits\CalculatesEarnings;

    public function index(Request $request)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            abort(403);
        }

        $rangeStart = now()->subDays(30);
        $lateTolerance = (int) \App\Models\Setting::where('key', 'late_checkin_tolerance')->value('value') ?: 15;

        // --- Guard Job Stats (30 days) ---
        $guardSchedules = $employee->schedules()
            ->where('from_datetime', '>=', $rangeStart)
            ->get();

        $guardShiftCount = $guardSchedules->count();
        $guardCompletedCount = $guardSchedules->where('status', 'completed')->count();
        $guardCancelledCount = $guardSchedules->filter(fn($j) => $j->status === 'cancelled' || $j->pivot->cancellation_status === 'approved')->count();
        $guardMissedCount = $guardSchedules->where('status', 'missed')->count();
        $guardUpcomingCount = $guardSchedules->whereIn('status', ['published', 'scheduled', 'upcoming', 'today'])
            ->where('from_datetime', '>', now())
            ->where('pivot.cancellation_status', '!=', 'approved')
            ->count();

        // --- Patroller Job Stats (30 days) ---
        $patrollerSchedules = $employee->patrollerSchedules()
            ->where('patroller_schedules.from_time', '>=', $rangeStart)
            ->get();

        $patrollerShiftCount = $patrollerSchedules->count();
        $patrollerCompletedCount = $patrollerSchedules->whereIn('job_status', ['completed', 'auto_ended'])->count();
        $patrollerMissedCount = $patrollerSchedules->filter(fn($j) => $j->to_time->isPast() && $j->job_status === 'pending')->count();
        $patrollerUpcomingCount = $patrollerSchedules->filter(fn($j) => $j->from_time->isFuture() && $j->job_status === 'pending')->count();

        // --- Attendance & Duration (Aggregated) ---
        $totalPastShifts = 0;
        $onTimeCount = 0;
        $lateCount = 0;
        $totalDurationHours = 0;
        $shiftsWithDuration = 0;

        // Process Guard Jobs
        foreach ($guardSchedules->where('from_datetime', '<=', now())->filter(fn($j) => $j->status !== 'cancelled' && $j->pivot->cancellation_status !== 'approved') as $shift) {
            $totalPastShifts++;
            if ($shift->pivot && $shift->pivot->actual_start_at) {
                $scheduledStart = $shift->from_datetime;
                $actualStart = Carbon::parse($shift->pivot->actual_start_at);
                if ($actualStart->lte($scheduledStart->copy()->addMinutes($lateTolerance))) {
                    $onTimeCount++;
                } else {
                    $lateCount++;
                }
                if ($shift->pivot->actual_end_at) {
                    $totalDurationHours += $actualStart->floatDiffInHours(Carbon::parse($shift->pivot->actual_end_at));
                    $shiftsWithDuration++;
                }
            }
        }

        // Process Patroller Jobs
        foreach ($patrollerSchedules->where('from_time', '<=', now())->filter(fn($j) => $j->status !== 'cancelled' && $j->pivot->cancellation_status !== 'approved') as $shift) {
            $totalPastShifts++;
            if ($shift->pivot && $shift->pivot->actual_start_at) {
                $scheduledStart = $shift->from_time;
                $actualStart = Carbon::parse($shift->pivot->actual_start_at);
                if ($actualStart->lte($scheduledStart->copy()->addMinutes($lateTolerance))) {
                    $onTimeCount++;
                } else {
                    $lateCount++;
                }
                if ($shift->pivot->actual_end_at) {
                    $totalDurationHours += $actualStart->floatDiffInHours(Carbon::parse($shift->pivot->actual_end_at));
                    $shiftsWithDuration++;
                }
            }
        }

        $attendanceRate = $totalPastShifts > 0 ? round(($onTimeCount / $totalPastShifts) * 100) : 0;
        $avgShiftDuration = $shiftsWithDuration > 0 ? round($totalDurationHours / $shiftsWithDuration, 1) : 0;

        // --- Incidents (30 days) ---
        $incidentCount = \App\Models\Incident::where('employee_id', $employee->id)
            ->where('created_at', '>=', $rangeStart)
            ->count();

        $patrollerTickets = \App\Models\PatrollerIssueTicket::where('employee_id', $employee->id)
            ->where('created_at', '>=', $rangeStart)
            ->count();

        // --- 12-Month Trends ---
        $months = [];
        for ($i = 11; $i >= 0; $i--) {
            $months[] = now()->subMonths($i)->format('Y-m');
        }

        $incomeData = [];
        $trendIncidentData = [];
        foreach ($months as $month) {
            $startOfMonth = Carbon::parse($month . '-01')->startOfMonth();
            $endOfMonth = Carbon::parse($month . '-01')->endOfMonth();

            // Guard Income
            $guardIncome = $employee->schedules()
                ->where('schedules.status', 'completed')
                ->whereBetween('from_datetime', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($j) {
                    return $this->calculateGuardEarnings($j);
                });

            // Patroller Income
            $patrollerIncome = $employee->patrollerSchedules()
                ->whereIn('job_status', ['completed', 'auto_ended'])
                ->whereBetween('patroller_schedules.from_time', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($j) {
                    return $this->calculatePatrollerEarnings($j);
                });

            $incomeData[] = round($guardIncome + $patrollerIncome, 2);

            // Incidents (Guard)
            $gInc = \App\Models\Incident::where('employee_id', $employee->id)
                ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->count();

            // Tickets (Patroller)
            $pInc = \App\Models\PatrollerIssueTicket::where('employee_id', $employee->id)
                ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->count();

            $trendIncidentData[] = $gInc + $pInc;
        }

        $monthLabels = array_map(fn($m) => Carbon::parse($m . '-01')->format('M'), $months);

        return view('employee.analytics.index', compact(
            'employee',
            'attendanceRate',
            'avgShiftDuration',
            'incidentCount',
            'patrollerTickets',
            'guardShiftCount',
            'guardCompletedCount',
            'guardMissedCount',
            'guardUpcomingCount',
            'patrollerShiftCount',
            'patrollerCompletedCount',
            'patrollerMissedCount',
            'patrollerUpcomingCount',
            'guardCancelledCount',
            'monthLabels',
            'incomeData',
            'trendIncidentData'
        ));
    }
}
