<?php

namespace App\Http\Controllers;

use App\Models\Checkpoint;
use App\Models\Site;
use Illuminate\Http\Request;

class CheckpointController extends Controller
{
    /**
     * Display a listing of checkpoints for a site.
     */
    public function index(Site $site)
    {
        $checkpoints = $site->checkpoints()->orderBy('created_at', 'desc')->get();

        if (request()->wantsJson()) {
            return response()->json(['checkpoints' => $checkpoints]);
        }

        return view('checkpoints.index', compact('site', 'checkpoints'));
    }

    /**
     * Store a newly created checkpoint.
     */
    public function store(Request $request, Site $site)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'special_instructions' => 'nullable|string',
            'scan_type' => 'required|in:all,qr,nfc,manual',
            'monitoring_option' => 'required|in:regular,scan_regular,scan_interval',
            'scan_interval_minutes' => 'nullable|integer|min:1',
            'checkpoint_type' => 'nullable|string|max:255',
            'checkpoint_id_code' => [
                'nullable',
                'string',
                'max:255',
                \Illuminate\Validation\Rule::unique('checkpoints')->where('site_id', $site->id)
            ],
            'checkpoint_id_image' => 'nullable|image|max:2048',
            'allow_manual_scan' => 'nullable',
            'grace_period_minutes' => 'nullable|integer|min:0',
            'extra_scan_options' => 'nullable|string', // Received as JSON string from FormData
        ]);

        $validated['site_id'] = $site->id;
        $validated['allow_manual_scan'] = filter_var($request->allow_manual_scan, FILTER_VALIDATE_BOOLEAN);

        if ($request->has('extra_scan_options')) {
            $validated['extra_scan_options'] = json_decode($request->extra_scan_options, true);
        }

        if ($request->hasFile('checkpoint_id_image')) {
            $path = $request->file('checkpoint_id_image')->store('checkpoints', 'public');
            $validated['checkpoint_id_image'] = $path;
        }

        $checkpoint = Checkpoint::create($validated);

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Checkpoint created successfully',
                'checkpoint' => $checkpoint
            ]);
        }

        return redirect()->route('sites.checkpoints.index', $site)
            ->with('success', 'Checkpoint created successfully');
    }

    /**
     * Update the specified checkpoint.
     */
    public function update(Request $request, Site $site, Checkpoint $checkpoint)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'special_instructions' => 'nullable|string',
            'scan_type' => 'required|in:all,qr,nfc,manual',
            'monitoring_option' => 'required|in:regular,scan_regular,scan_interval',
            'scan_interval_minutes' => 'nullable|integer|min:1',
            'checkpoint_type' => 'nullable|string|max:255',
            'checkpoint_id_code' => [
                'nullable',
                'string',
                'max:255',
                \Illuminate\Validation\Rule::unique('checkpoints')
                    ->where('site_id', $site->id)
                    ->ignore($checkpoint->id)
            ],
            'checkpoint_id_image' => 'nullable|image|max:2048',
            'allow_manual_scan' => 'nullable',
            'grace_period_minutes' => 'nullable|integer|min:0',
            'extra_scan_options' => 'nullable|string', // Received as JSON string from FormData
        ]);

        $validated['allow_manual_scan'] = filter_var($request->allow_manual_scan, FILTER_VALIDATE_BOOLEAN);

        if ($request->has('extra_scan_options')) {
            $validated['extra_scan_options'] = json_decode($request->extra_scan_options, true);
        }

        if ($request->hasFile('checkpoint_id_image')) {
            // Delete old image if exists
            if ($checkpoint->checkpoint_id_image) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($checkpoint->checkpoint_id_image);
            }
            $path = $request->file('checkpoint_id_image')->store('checkpoints', 'public');
            $validated['checkpoint_id_image'] = $path;
        }

        $checkpoint->update($validated);

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Checkpoint updated successfully',
                'checkpoint' => $checkpoint
            ]);
        }

        return redirect()->route('sites.checkpoints.index', $site)
            ->with('success', 'Checkpoint updated successfully');
    }

    /**
     * Remove the specified checkpoint.
     */
    public function destroy(Request $request, Site $site, Checkpoint $checkpoint)
    {
        $checkpoint->delete();

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Checkpoint deleted successfully'
            ]);
        }

        return redirect()->route('sites.checkpoints.index', $site)
            ->with('success', 'Checkpoint deleted successfully');
    }
}
