<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CompanySelectionController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        $companies = $user->companies()->where('active', true)->with('timeZone')->get();

        // 1. If only one company, auto-select regardless of manual flag
        if ($companies->count() === 1) {
            $company = $companies->first();
            $this->applySelection($user, $company);
            return redirect()->intended('/dashboard');
        }

        // 2. If no companies, logout
        if ($companies->isEmpty()) {
            Auth::logout();
            return redirect()->route('login')->withErrors(['email' => 'No active companies assigned to your account.']);
        }

        // 3. Persistent selection: If not manually switching and has a last preference, auto-select it
        if (!$request->has('manual') && $user->last_selected_company_id) {
            $lastCompany = $companies->firstWhere('id', $user->last_selected_company_id);
            if ($lastCompany) {
                $this->applySelection($user, $lastCompany);
                return redirect()->intended('/dashboard');
            }
        }

        return view('auth.select-company', compact('companies'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'company_id' => 'required|exists:companies,id',
        ]);

        $user = Auth::user();

        // Ensure the company actually belongs to the user
        $company = $user->companies()->with('timeZone')
            ->where('companies.id', $request->company_id)
            ->where('companies.active', true)
            ->first();

        if (!$company) {
            return back()->withErrors(['company_id' => 'Invalid company selection.']);
        }

        $this->applySelection($user, $company);

        return redirect()->intended('/dashboard');
    }

    /**
     * Helper to set session and save user preference.
     */
    protected function applySelection($user, $company)
    {
        session([
            'selected_company_id' => $company->id,
            'selected_company_timezone' => $company->timeZone->name ?? config('app.timezone')
        ]);

        // Save for next time
        if ($user->last_selected_company_id !== $company->id) {
            $user->update(['last_selected_company_id' => $company->id]);
        }
    }
}
