<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Zone;
use App\Models\Country;
use App\Models\State;
use Illuminate\Http\Request;

class ZoneController extends Controller
{
    public function index()
    {
        $zones = Zone::with(['country', 'state'])->latest()->paginate(100);
        return view('admin.zones.index', compact('zones'));
    }

    public function create()
    {
        $countries = Country::where('active', true)->orderBy('name')->get();
        // Static list of languages for now, as no master exists.
        $languages = ['English', 'Spanish', 'French', 'German', 'Chinese', 'Japanese', 'Arabic', 'Hindi', 'Portuguese', 'Russian'];
        return view('admin.zones.create', compact('countries', 'languages'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'details' => 'nullable|string',
            'external_id' => 'nullable|string|max:255',
            'preferred_language' => 'nullable|string|max:255',
            'zone_template' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:255',
            'country_id' => 'nullable|exists:countries,id',
            'state_id' => 'nullable|exists:states,id',
            'zip_code' => 'nullable|string|max:20',
        ]);

        Zone::create($validated);

        return redirect()->route('admin.zones.index')->with('success', 'Zone created successfully.');
    }

    public function edit(Zone $zone)
    {
        $countries = Country::where('active', true)->orderBy('name')->get();
        $languages = ['English', 'Spanish', 'French', 'German', 'Chinese', 'Japanese', 'Arabic', 'Hindi', 'Portuguese', 'Russian'];
        return view('admin.zones.edit', compact('zone', 'countries', 'languages'));
    }

    public function update(Request $request, Zone $zone)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'details' => 'nullable|string',
            'external_id' => 'nullable|string|max:255',
            'preferred_language' => 'nullable|string|max:255',
            'zone_template' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:255',
            'country_id' => 'nullable|exists:countries,id',
            'state_id' => 'nullable|exists:states,id',
            'zip_code' => 'nullable|string|max:20',
        ]);

        $zone->update($validated);

        return redirect()->route('admin.zones.index')->with('success', 'Zone updated successfully.');
    }

    public function destroy(Zone $zone)
    {
        $zone->delete();
        return redirect()->route('admin.zones.index')->with('success', 'Zone deleted successfully.');
    }

    public function toggleStatus(Zone $zone)
    {
        $zone->active = !$zone->active;
        $zone->save();
        return response()->json(['message' => 'Status updated successfully']);
    }

    public function getStates(Country $country)
    {
        return response()->json($country->states()->where('active', true)->orderBy('name')->get());
    }
}
