<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\EmailTemplate;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class EmailTemplateController extends Controller
{
    public function index(Request $request)
    {
        $query = EmailTemplate::with('company');

        if ($request->company_id) {
            $query->where('company_id', $request->company_id);
        }

        if ($request->search) {
            $query->where('name', 'like', '%' . $request->search . '%')
                ->orWhere('subject', 'like', '%' . $request->search . '%');
        }

        $templates = $query->paginate(100);
        $companies = Company::where('active', true)->orderBy('name')->get();
        // Fetch strictly active employees and sites to avoid sending to inactive ones
        $employees = \App\Models\Employee::where('active', 1)->select('id', 'first_name', 'last_name', 'email')->get();
        $sites = \App\Models\Site::where('active', true)->select('id', 'name', 'contact_email')->get();

        return view('admin.email-templates.index', compact('templates', 'companies', 'employees', 'sites'));
    }

    public function create()
    {
        $companies = Company::where('active', true)->orderBy('name')->get();
        // Get signatures
        $emailSignatures = \App\Models\EmailSignature::all()->groupBy('company_id');
        return view('admin.email-templates.create', compact('companies', 'emailSignatures'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'company_id' => 'required|exists:companies,id',
            'name' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'description' => 'nullable|string',
            'content' => 'required|string',
            'attachment' => 'nullable|file|max:10240', // 10MB
            'is_active' => 'boolean'
        ]);

        $data = $request->all();

        if ($request->hasFile('attachment')) {
            $data['attachment'] = $request->file('attachment')->store('email-templates/attachments', 'public');
        }

        EmailTemplate::create($data);

        return redirect()->route('admin.email-templates.index')->with('success', 'Email template created successfully.');
    }

    public function edit(EmailTemplate $emailTemplate)
    {
        $companies = Company::where('active', true)->orderBy('name')->get();
        return view('admin.email-templates.edit', compact('emailTemplate', 'companies'));
    }

    public function update(Request $request, EmailTemplate $emailTemplate)
    {
        $request->validate([
            'company_id' => 'required|exists:companies,id',
            'name' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'description' => 'nullable|string',
            'content' => 'required|string',
            'attachment' => 'nullable|file|max:10240',
            'is_active' => 'boolean'
        ]);

        $data = $request->all();

        if ($request->hasFile('attachment')) {
            if ($emailTemplate->attachment) {
                Storage::disk('public')->delete($emailTemplate->attachment);
            }
            $data['attachment'] = $request->file('attachment')->store('email-templates/attachments', 'public');
        }

        $emailTemplate->update($data);

        return redirect()->route('admin.email-templates.index')->with('success', 'Email template updated successfully.');
    }

    public function destroy(EmailTemplate $emailTemplate)
    {
        if ($emailTemplate->attachment) {
            Storage::disk('public')->delete($emailTemplate->attachment);
        }
        $emailTemplate->delete();
        return redirect()->route('admin.email-templates.index')->with('success', 'Email template deleted successfully.');
    }

    public function send(Request $request, EmailTemplate $emailTemplate)
    {
        $request->validate([
            'target_audience' => 'required|in:employees,sites,custom',
            'employee_ids' => 'nullable|required_if:target_audience,employees|array',
            'site_ids' => 'nullable|required_if:target_audience,sites|array',
            'excel_file' => 'nullable|required_if:target_audience,custom|file|mimes:xlsx,csv,xls',
        ]);

        $emails = [];

        if ($request->target_audience === 'employees') {
            $emails = \App\Models\Employee::whereIn('id', $request->employee_ids)
                ->whereNotNull('email')
                ->pluck('email')
                ->toArray();
        } elseif ($request->target_audience === 'sites') {
            $emails = \App\Models\Site::whereIn('id', $request->site_ids)
                ->whereNotNull('contact_email')
                ->pluck('contact_email')
                ->toArray();
        } elseif ($request->target_audience === 'custom' && $request->hasFile('excel_file')) {
            try {
                $rows = \Maatwebsite\Excel\Facades\Excel::toCollection(new \stdClass, $request->file('excel_file'))->first();
                // Assuming email is in the first column or named 'email'
                foreach ($rows as $row) {
                    $email = null;
                    // Check for 'email' key or first column
                    if (isset($row['email'])) {
                        $email = $row['email'];
                    } elseif (isset($row[0]) && filter_var($row[0], FILTER_VALIDATE_EMAIL)) {
                        $email = $row[0];
                    }

                    if ($email && filter_var($email, FILTER_VALIDATE_EMAIL)) {
                        $emails[] = $email;
                    }
                }
            } catch (\Exception $e) {
                return back()->with('error', 'Error parsing Excel file: ' . $e->getMessage());
            }
        }

        $emails = array_unique(array_filter($emails));

        if (empty($emails)) {
            return back()->with('error', 'No valid email addresses found to send to.');
        }

        // Send Emails (Queueing recommended for production)
        foreach ($emails as $email) {
            try {
                \Illuminate\Support\Facades\Mail::html($emailTemplate->content, function ($message) use ($email, $emailTemplate) {
                    $message->to($email)
                        ->subject($emailTemplate->subject);

                    if ($emailTemplate->attachment) {
                        $message->attach(Storage::disk('public')->path($emailTemplate->attachment));
                    }
                });
            } catch (\Exception $e) {
                // Log error but continue sending to others
                \Illuminate\Support\Facades\Log::error("Failed to send email template to $email: " . $e->getMessage());
            }
        }

        return back()->with('success', 'Email sending initiated for ' . count($emails) . ' recipients.');
    }
}
