<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Department;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class DepartmentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');
        $sort = $request->input('sort', 'latest');

        $query = Department::query();

        if ($search) {
            $query->where('name', 'like', "%{$search}%");
        }

        switch ($sort) {
            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            case 'oldest':
                $query->orderBy('created_at', 'asc');
                break;
            default:
                $query->latest();
                break;
        }

        $departments = $query->paginate(100)->withQueryString();
        return view('admin.departments.index', compact('departments', 'search'));
    }

    public function exportPDF()
    {
        $departments = Department::latest()->get();
        $data = [];
        foreach ($departments as $department) {
            $data[] = [
                $department->name,
                $department->active ? 'Active' : 'Inactive',
                $department->created_at->format('M d, Y')
            ];
        }

        $pdf = Pdf::loadView('admin.exports.pdf_template', [
            'title' => 'Departments List',
            'headers' => ['Name', 'Status', 'Created At'],
            'data' => $data
        ]);

        return $pdf->download('departments-' . date('Y-m-d') . '.pdf');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.departments.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'display_id' => 'nullable|string|max:255',
            'details' => 'nullable|string',
        ]);

        Department::create($validated);

        return redirect()->route('admin.departments.index')->with('success', 'Department created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Department $department)
    {
        return view('admin.departments.show', compact('department'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Department $department)
    {
        return view('admin.departments.edit', compact('department'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Department $department)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'display_id' => 'nullable|string|max:255',
            'details' => 'nullable|string',
            'active' => 'sometimes|boolean',
        ]);

        $department->update($validated);

        return redirect()->route('admin.departments.index')->with('success', 'Department updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Department $department)
    {
        $department->delete();

        return redirect()->route('admin.departments.index')->with('success', 'Department deleted successfully.');
    }

    public function toggleStatus(Department $department)
    {
        $department->update(['active' => !$department->active]);
        return response()->json(['success' => true, 'active' => (bool)$department->active]);
    }
}
