<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Models\Company;
use App\Models\JobRole;
use App\Models\WageType;
use App\Models\CompanyJobRoleVisibility;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

use Illuminate\Support\Facades\Storage;

class AdminSettingController extends Controller
{
    public function appearance()
    {
        $keys = [
            'default_user_male_image',
            'default_user_female_image',
            'default_site_image',
            'default_employee_male_image',
            'default_employee_female_image',
            'default_license_image'
        ];
        $settings = Setting::whereIn('key', $keys)->pluck('value', 'key');
        return view('admin.settings.appearance', compact('settings'));
    }

    public function updateAppearance(Request $request)
    {
        $request->validate([
            'default_user_male_image' => 'nullable|image|max:2048',
            'default_user_female_image' => 'nullable|image|max:2048',
            'default_site_image' => 'nullable|image|max:2048',
            'default_employee_male_image' => 'nullable|image|max:2048',
            'default_employee_female_image' => 'nullable|image|max:2048',
            'default_license_image' => 'nullable|image|max:2048',
        ]);

        $keys = [
            'default_user_male_image',
            'default_user_female_image',
            'default_site_image',
            'default_employee_male_image',
            'default_employee_female_image',
            'default_license_image'
        ];

        foreach ($keys as $key) {
            if ($request->hasFile($key)) {
                // Delete old file if exists
                $oldValue = Setting::where('key', $key)->value('value');
                if ($oldValue && Storage::disk('public')->exists($oldValue)) {
                    Storage::disk('public')->delete($oldValue);
                }

                // Store new file
                $path = $request->file($key)->store('settings/defaults', 'public');
                Setting::updateOrCreate(['key' => $key], ['value' => $path]);
            }
        }

        return redirect()->back()->with('success', 'Appearance settings updated successfully.');
    }
    public function cssSettings()
    {
        $settings = Setting::whereIn('key', ['font_size', 'font_family', 'font_weight'])->pluck('value', 'key');

        $googleFonts = [
            'Inter',
            'Roboto',
            'Open Sans',
            'Lato',
            'Montserrat',
            'Oswald',
            'Source Sans Pro',
            'Slabo 27px',
            'Raleway',
            'PT Sans',
            'Merriweather',
            'Noto Sans',
            'Nunito',
            'Playfair Display',
            'Lora',
            'Muli',
            'Rubik',
            'Poppins',
            'Arvo',
            'Work Sans',
            'Quicksand',
            'Karla',
            'Ubuntu',
            'PT Serif',
            'Crimson Text',
            'Fira Sans',
            'Varela Round',
            'Libre Baskerville',
            'Titillium Web',
            'Assistant',
            'Josefin Sans',
            'Dosis',
            'Signika',
            'Nanum Gothic',
            'Oxygen',
            'Anton',
            'Cabin',
            'Archivo Narrow',
            'Tajawal',
            'Overpass',
            'Questrial',
            'Exo 2',
            'Asap',
            'Ubuntu Condensed',
            'Maven Pro',
            'Inconsolata',
            'Pacifico',
            'Bitter',
            'Dancing Script',
            'EB Garamond',
            'Kanit',
            'Shadows Into Light',
            'Abel',
            'Comfortaa',
            'Lobster',
            'Vollkorn',
            'Crete Round',
            'Hind',
            'Francois One',
            'Pathway Gothic One',
            'Barlow',
            'Cuprum',
            'Philosopher',
            'Prompt',
            'Antic Slab',
            'Russo One',
            'Righteous',
            'Vidaloka',
            'Play',
            'Old Standard TT',
            'Orbitron',
            'Cardo',
            'Teko',
            'Rajdhani',
            'Alegreya',
            'Exo',
            'Yanone Kaffeesatz',
            'Architects Daughter',
            'Fredoka One',
            'Kaushan Script',
            'Alegreya Sans',
            'Bree Serif',
            'Alfa Slab One',
            'Bebas Neue',
            'DM Sans',
            'Manrope',
            'Public Sans',
            'Space Grotesk',
            'Syne',
            'Fraunces',
            'Lexend',
            'Outfit',
            'Plus Jakarta Sans',
            'Be Vietnam Pro',
            'Sora',
            'Figtree',
            'Urbanist',
            'Bricolage Grotesque',
            'Instrument Sans',
            'Hanken Grotesk'
        ];

        return view('admin.settings.css', compact('settings', 'googleFonts'));
    }

    public function updateCssSettings(Request $request)
    {
        $data = $request->validate([
            'font_size' => 'required|string',
            'font_family' => 'required|string',
            'font_weight' => 'required|string',
        ]);

        foreach ($data as $key => $value) {
            Setting::updateOrCreate(['key' => $key], ['value' => $value]);
        }

        return redirect()->back()->with('success', 'CSS settings updated successfully.');
    }

    public function changePassword()
    {
        return view('admin.settings.password');
    }

    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => [
                'required',
                'confirmed',
                Password::min(8)
                    ->letters()
                    ->mixedCase()
                    ->numbers()
                    ->symbols()
                    ->uncompromised()
            ],
        ]);

        $request->user()->update([
            'password' => Hash::make($request->password),
        ]);

        return redirect()->back()->with('success', 'Password updated successfully.');
    }

    public function integrations()
    {
        $keys = [
            'firebase_api_key',
            'firebase_auth_domain',
            'firebase_project_id',
            'firebase_storage_bucket',
            'firebase_messaging_sender_id',
            'firebase_app_id',
            'firebase_service_account_json',
            'mailgun_domain',
            'mailgun_secret',
            'mailgun_endpoint',
            'mail_from_address',
            'mail_from_name',
            'mail_host',
            'mail_port',
            'mail_username',
            'mail_password',
            'mail_encryption',
            'recaptcha_site_key',
            'recaptcha_secret_key',
            'recaptcha_enabled',
            'google_authenticator_enabled',
        ];
        $settings = Setting::whereIn('key', $keys)->pluck('value', 'key');

        return view('admin.settings.integrations', compact('settings'));
    }

    public function updateIntegrations(Request $request)
    {
        $data = $request->validate([
            'firebase_api_key' => 'nullable|string',
            'firebase_auth_domain' => 'nullable|string',
            'firebase_project_id' => 'nullable|string',
            'firebase_storage_bucket' => 'nullable|string',
            'firebase_messaging_sender_id' => 'nullable|string',
            'firebase_app_id' => 'nullable|string',
            'firebase_service_account_json' => 'nullable|string',
            'mailgun_domain' => 'nullable|string',
            'mailgun_secret' => 'nullable|string',
            'mailgun_endpoint' => 'nullable|string',
            'mail_from_address' => 'nullable|email',
            'mail_from_name' => 'nullable|string',
            'mail_host' => 'nullable|string',
            'mail_port' => 'nullable|string',
            'mail_username' => 'nullable|string',
            'mail_password' => 'nullable|string',
            'mail_encryption' => 'nullable|string',
            'recaptcha_site_key' => 'nullable|string',
            'recaptcha_secret_key' => 'nullable|string',
            'recaptcha_enabled' => 'nullable|string',
            'google_authenticator_enabled' => 'nullable|string',
        ]);

        if (!$request->has('recaptcha_enabled')) {
            $data['recaptcha_enabled'] = '0';
        }

        if (!$request->has('google_authenticator_enabled')) {
            $data['google_authenticator_enabled'] = '0';
        }

        foreach ($data as $key => $value) {
            Setting::updateOrCreate(['key' => $key], ['value' => $value ?? '']);
        }

        $redirect = redirect()->back()->with('success', 'Integrations updated successfully.');

        if ($request->tab) {
            $redirect->with('active_tab', $request->tab);
        }

        return $redirect;
    }

    public function operational()
    {
        $companies = Company::with(['defaultWageType', 'visibilitySettings'])->orderBy('name')->get();
        $jobRoles = JobRole::where('active', true)->orderBy('name')->get();
        $wageTypes = WageType::where('active', true)->orderBy('name')->get();

        $lateCheckinTolerance = \App\Models\Setting::where('key', 'late_checkin_tolerance')->value('value');
        $notesEnabled = \App\Models\Setting::where('key', 'notes_enabled')->value('value') ?? 0;
        $notesInterval = \App\Models\Setting::where('key', 'notes_interval')->value('value') ?? 30;

        $maxShiftHours = \App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12;
        $overtimeThreshold = \App\Models\Setting::where('key', 'overtime_threshold')->value('value') ?? 8;

        return view('admin.settings.operational', compact(
            'companies',
            'jobRoles',
            'wageTypes',
            'lateCheckinTolerance',
            'notesEnabled',
            'notesInterval',
            'maxShiftHours',
            'overtimeThreshold'
        ));
    }

    public function updateOperational(Request $request)
    {
        $request->validate([
            'defaults' => 'array',
            'defaults.*.wage_type_id' => 'nullable|exists:wage_types,id',
            'visibility' => 'array',
            'visibility.*.*' => 'nullable|integer|min:0',
            'late_checkin_tolerance' => 'nullable|integer|min:0',
            'notes_enabled' => 'nullable|boolean',
            'notes_interval' => 'nullable|integer|min:1',
            'max_shift_hours' => 'nullable|numeric|min:0',
            'overtime_threshold' => 'nullable|numeric|min:0',
        ]);

        // Update individual settings
        $settings = [
            'late_checkin_tolerance' => $request->late_checkin_tolerance,
            'notes_enabled' => $request->has('notes_enabled') ? 1 : 0,
            'notes_interval' => $request->notes_interval,
            'max_shift_hours' => $request->max_shift_hours,
            'overtime_threshold' => $request->overtime_threshold,
        ];

        foreach ($settings as $key => $value) {
            if ($value !== null) {
                \App\Models\Setting::updateOrCreate(['key' => $key], ['value' => $value]);
            }
        }

        // Update Default Wage Types for Companies
        if ($request->has('defaults')) {
            foreach ($request->defaults as $companyId => $data) {
                Company::where('id', $companyId)->update([
                    'default_wage_type_id' => $data['wage_type_id'] ?: null,
                ]);
            }
        }

        // Update Job Role Visibility Levels
        if ($request->has('visibility')) {
            foreach ($request->visibility as $companyId => $roles) {
                foreach ($roles as $roleId => $minutes) {
                    if ($minutes !== null && $minutes !== '') {
                        CompanyJobRoleVisibility::updateOrCreate(
                            ['company_id' => $companyId, 'job_role_id' => $roleId],
                            ['visibility_minutes' => $minutes]
                        );
                    }
                }
            }
        }

        return redirect()->back()->with('success', 'Operational settings updated successfully.');
    }

    public function security()
    {
        $keys = ['login_max_attempts', 'login_decay_minutes', 'dashboard_refresh_enabled', 'dashboard_refresh_interval'];
        $settings = Setting::whereIn('key', $keys)->pluck('value', 'key');
        return view('admin.settings.security', compact('settings'));
    }

    public function updateSecurity(Request $request)
    {
        $data = $request->validate([
            'login_max_attempts' => 'required|integer|min:1|max:50',
            'login_decay_minutes' => 'required|integer|min:1|max:1440',
            'dashboard_refresh_enabled' => 'nullable|boolean',
            'dashboard_refresh_interval' => 'nullable|required_if:dashboard_refresh_enabled,1|integer|min:1|max:60',
        ]);

        // Handle boolean checkbox which might not be in request if unchecked
        if (!$request->has('dashboard_refresh_enabled')) {
            $data['dashboard_refresh_enabled'] = 0;
        }

        foreach ($data as $key => $value) {
            Setting::updateOrCreate(['key' => $key], ['value' => $value]);
        }

        return redirect()->back()->with('success', 'Security settings updated successfully.');
    }
    public function system()
    {
        return view('admin.settings.system');
    }

    public function runMigration()
    {
        try {
            \Illuminate\Support\Facades\Artisan::call('migrate', ['--force' => true]);
            $output = \Illuminate\Support\Facades\Artisan::output();
            return redirect()->back()->with('success', 'Migration executed successfully. Output: ' . $output);
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Migration failed: ' . $e->getMessage()]);
        }
    }

    public function clearCache()
    {
        try {
            \Illuminate\Support\Facades\Artisan::call('optimize:clear');
            return redirect()->back()->with('success', 'System cache cleared successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Cache clearing failed: ' . $e->getMessage()]);
        }
    }
}
