<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use PragmaRX\Google2FA\Google2FA;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class TwoFactorController extends Controller
{
    public function setup()
    {
        $user = auth()->user();

        if ($user->google2fa_secret) {
            return redirect()->route('dashboard');
        }

        $google2fa = new Google2FA();
        $secret = session('2fa_secret') ?: $google2fa->generateSecretKey();

        session(['2fa_secret' => $secret]);

        $qrCodeUrl = $google2fa->getQRCodeUrl(
            config('app.name'),
            $user->email,
            $secret
        );

        $qrCode = QrCode::size(200)->generate($qrCodeUrl);

        return view('auth.2fa_setup', compact('qrCode', 'secret'));
    }

    public function storeSetup(Request $request)
    {
        $request->validate([
            'code' => 'required',
        ]);

        $user = auth()->user();
        $secret = session('2fa_secret');

        if (!$secret) {
            return redirect()->route('2fa.setup')->withErrors(['code' => 'Session expired. Please try again.']);
        }

        $google2fa = new Google2FA();
        $valid = $google2fa->verifyKey($secret, $request->code);

        if ($valid) {
            $user->google2fa_secret = $secret;
            $user->save();

            session()->forget('2fa_secret');
            session(['google2fa_verified' => true]);

            return redirect()->route('dashboard')->with('success', 'Google Authenticator set up successfully.');
        }

        return redirect()->back()->withErrors(['code' => 'Invalid verification code.']);
    }

    public function verify()
    {
        return view('auth.2fa_verify');
    }

    public function storeVerify(Request $request)
    {
        $request->validate([
            'code' => 'required',
        ]);

        $user = auth()->user();
        $google2fa = new Google2FA();
        $valid = $google2fa->verifyKey($user->google2fa_secret, $request->code);

        if ($valid) {
            session(['google2fa_verified' => true]);
            return redirect()->intended(route('dashboard'));
        }

        return redirect()->back()->withErrors(['code' => 'Invalid verification code.']);
    }
}
