<?php

namespace App\Http\Controllers;

use App\Models\TourRoute;
use App\Models\Site;
use Illuminate\Http\Request;

class TourRouteController extends Controller
{
    /**
     * Display a listing of tour routes for a site.
     */
    public function index(Site $site)
    {
        $tourRoutes = $site->tourRoutes()->with('checkpoints')->withCount('checkpoints')->orderBy('created_at', 'desc')->get();

        if (request()->wantsJson()) {
            return response()->json(['tourRoutes' => $tourRoutes]);
        }

        return view('tour-routes.index', compact('site', 'tourRoutes'));
    }

    /**
     * Store a newly created tour route.
     */
    public function store(Request $request, Site $site)
    {
        $validated = $request->validate([
            'description' => 'required|string|max:255',
            'assigned_to' => 'required|in:all,specific',
            'special_instructions' => 'nullable|string',
            'estimated_duration_minutes' => 'nullable|integer|min:0',
            'grace_period_minutes' => 'nullable|integer|min:0',
            'recurrence_type' => 'required|in:weekly,monthly',
            'schedule_days' => 'nullable|array',
            'checkpoint_ids' => 'nullable|array',
            'checkpoint_ids.*' => 'exists:checkpoints,id'
        ]);

        $validated['site_id'] = $site->id;
        $tourRoute = TourRoute::create($validated);

        if (!empty($request->checkpoint_ids)) {
            $syncData = [];
            foreach ($request->checkpoint_ids as $index => $checkpointId) {
                $syncData[$checkpointId] = ['order' => $index];
            }
            $tourRoute->checkpoints()->sync($syncData);
        }

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Tour route created successfully',
                'tourRoute' => $tourRoute->load('checkpoints')
            ]);
        }

        return redirect()->route('sites.tour-routes.index', $site)
            ->with('success', 'Tour route created successfully');
    }

    /**
     * Update the specified tour route.
     */
    public function update(Request $request, Site $site, TourRoute $tourRoute)
    {
        $validated = $request->validate([
            'description' => 'required|string|max:255',
            'assigned_to' => 'required|in:all,specific',
            'special_instructions' => 'nullable|string',
            'estimated_duration_minutes' => 'nullable|integer|min:0',
            'grace_period_minutes' => 'nullable|integer|min:0',
            'recurrence_type' => 'required|in:weekly,monthly',
            'schedule_days' => 'nullable|array',
            'checkpoint_ids' => 'nullable|array',
            'checkpoint_ids.*' => 'exists:checkpoints,id'
        ]);

        $tourRoute->update($validated);

        if (isset($request->checkpoint_ids)) {
            $syncData = [];
            foreach ($request->checkpoint_ids as $index => $checkpointId) {
                $syncData[$checkpointId] = ['order' => $index];
            }
            $tourRoute->checkpoints()->sync($syncData);
        }

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Tour route updated successfully',
                'tourRoute' => $tourRoute->load('checkpoints')
            ]);
        }

        return redirect()->route('sites.tour-routes.index', $site)
            ->with('success', 'Tour route updated successfully');
    }

    /**
     * Remove the specified tour route.
     */
    public function destroy(Request $request, Site $site, TourRoute $tourRoute)
    {
        $tourRoute->delete();

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Tour route deleted successfully'
            ]);
        }

        return redirect()->route('sites.tour-routes.index', $site)
            ->with('success', 'Tour route deleted successfully');
    }
}
