<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\Country;
use App\Models\State;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EmployeeProfileController extends Controller
{
    public function index()
    {
        $employee = Auth::guard('employee')->user();

        if (! $employee) {
            abort(403, 'Unauthorized');
        }

        $employee->load([
            'jobRole',
            'department',
            'permanentCountry',
            'permanentState',
            'correspondingCountry',
            'correspondingState',
            'licenceIssuingProvinceState',
            'skills',
            'identifications',
        ]);

        $states = State::where('active', true)->orderBy('name')->get();
        $countries = Country::where('active', true)->orderBy('name')->get();

        return view('employee.profile', compact('employee', 'states', 'countries'));
    }

    public function update(Request $request)
    {
        $employee = Auth::guard('employee')->user();
        \Log::info('Employee profile update attempt', ['id' => $employee->id, 'request' => $request->all()]);

        if (! $employee) {
            abort(403, 'Unauthorized');
        }

        $validated = $request->validate([
            // Personal Info
            'first_name' => 'required|string|max:255',
            'middle_name' => 'nullable|string|max:255',
            'last_name' => 'required|string|max:255',
            'gender' => 'required|in:Male,Female,Other',
            'dob' => 'required|date',
            'marital_status' => 'required|string',
            
            // Contact & Credential
            'phone_number' => 'required|string|max:20',
            'cell_number' => 'nullable|string|max:20',
            'emergency_number' => 'nullable|string|max:20',
            'license_number' => 'nullable|string|max:50',
            'license_expiry' => 'nullable|date',
            'file_license' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',

            // Permanent Address
            'permanent_address_line_1' => 'required|string|max:255',
            'permanent_address_line_2' => 'nullable|string|max:255',
            'permanent_city' => 'required|string|max:100',
            'permanent_state_id' => 'required|exists:states,id',
            'permanent_country_id' => 'required|exists:countries,id',
            'permanent_zip_code' => 'required|string|max:20',

            // Corresponding Address
            'corresponding_address_line_1' => 'nullable|string|max:255',
            'corresponding_address_line_2' => 'nullable|string|max:255',
            'corresponding_city' => 'nullable|string|max:100',
            'corresponding_state_id' => 'nullable|exists:states,id',
            'corresponding_country_id' => 'nullable|exists:countries,id',
            'corresponding_zip_code' => 'nullable|string|max:20',
        ]);

        $updateData = [
            'first_name' => $validated['first_name'],
            'middle_name' => $validated['middle_name'],
            'last_name' => $validated['last_name'],
            'gender' => $validated['gender'],
            'dob' => $validated['dob'],
            'birthday' => $validated['dob'], // Sync legacy column
            'marital_status' => $validated['marital_status'],
            'phone_number' => $validated['phone_number'],
            'cell_number' => $validated['cell_number'],
            'emergency_number' => $validated['emergency_number'],
            'permanent_address_line_1' => $validated['permanent_address_line_1'],
            'permanent_address_line_2' => $validated['permanent_address_line_2'],
            'permanent_city' => $validated['permanent_city'],
            'permanent_state_id' => $validated['permanent_state_id'],
            'permanent_country_id' => $validated['permanent_country_id'],
            'permanent_zip_code' => $validated['permanent_zip_code'],
            'corresponding_address_line_1' => $validated['corresponding_address_line_1'],
            'corresponding_address_line_2' => $validated['corresponding_address_line_2'],
            'corresponding_city' => $validated['corresponding_city'],
            'corresponding_state_id' => $validated['corresponding_state_id'],
            'corresponding_country_id' => $validated['corresponding_country_id'],
            'corresponding_zip_code' => $validated['corresponding_zip_code'],
            'license_number' => $validated['license_number'],
            'license_expiry' => $validated['license_expiry'],
        ];

        if ($request->hasFile('file_license')) {
            $path = $request->file('file_license')->store('employees/licenses', 'public');
            $updateData['file_license'] = $path;
        }

        try {
            $employee->update($updateData);
            \Log::info('Employee profile updated successfully', ['id' => $employee->id]);
            return redirect()->back()->with('success', 'Profile updated successfully.');
        } catch (\Exception $e) {
            \Log::error('Employee profile update failed', ['id' => $employee->id, 'error' => $e->getMessage()]);
            return redirect()->back()->withInput()->with('error', 'Critical operational failure: ' . $e->getMessage());
        }
    }

    public function downloadIdCard()
    {
        $employee = Auth::guard('employee')->user();
        if (! $employee) {
            abort(403);
        }
        $employee->load(['jobRole', 'company']);
        $pdf = Pdf::loadView('employees.id_card', compact('employee'));

        return $pdf->stream('My_ID_Card.pdf');
    }
}
