<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function showLogin()
    {
        if (Auth::check()) {
            return redirect()->route('dashboard'); // Assuming a dashboard route for users exists or will be created
        }
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => ['required', 'email'],
            'password' => ['required'],
        ]);

        // Rate Limiting Logic
        $throttleKey = Str::transliterate(Str::lower($request->input('email')) . '|' . $request->ip());

        // Fetch settings or defaults
        $maxAttempts = (int) \App\Models\Setting::where('key', 'login_max_attempts')->value('value') ?: 5;
        $decayMinutes = (int) \App\Models\Setting::where('key', 'login_decay_minutes')->value('value') ?: 1;

        if (RateLimiter::tooManyAttempts($throttleKey, $maxAttempts)) {
            $seconds = RateLimiter::availableIn($throttleKey);
            $minutes = ceil($seconds / 60);

            throw ValidationException::withMessages([
                'email' => trans('auth.throttle', [
                    'seconds' => $seconds,
                    'minutes' => $minutes,
                ]),
            ]);
        }

        if (Auth::attempt($credentials)) {
            RateLimiter::clear($throttleKey);

            $user = Auth::user();
            // Optional: Check if user is active or has specific role if needed
            if (!$user->active) {
                Auth::logout();
                return back()->withErrors([
                    'email' => 'Your account is inactive.',
                ])->onlyInput('email');
            }

            $request->session()->regenerate();

            // Ensure single session for 'user' role
            if ($user->role === 'user') {
                Auth::logoutOtherDevices($request->post('password'));
            }

            return redirect()->intended(route('dashboard'));
        }

        // Try Employee Login
        if (Auth::guard('employee')->attempt($credentials)) {
            RateLimiter::clear($throttleKey);

            $employee = Auth::guard('employee')->user();

            // Check for termination
            if ($employee->terminated_date && \Carbon\Carbon::parse($employee->terminated_date)->isPast()) {
                Auth::guard('employee')->logout();
                return back()->withErrors([
                    'email' => 'Your account is terminated.',
                ])->onlyInput('email');
            }

            $request->session()->regenerate();

            // Set company context for the employee
            if ($employee->company_id) {
                session(['selected_company_id' => $employee->company_id]);
            }

            return redirect()->intended(route('employee.dashboard'));
        }

        RateLimiter::hit($throttleKey, $decayMinutes * 60);

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ])->onlyInput('email');
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('login');
    }
}
