<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Route;
use App\Models\Company;
use Illuminate\Http\Request;

use Barryvdh\DomPDF\Facade\Pdf;

class RouteController extends Controller
{
    public function exportPDF(Request $request)
    {
        $query = Route::with(['company', 'sites']);

        // Search
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('route_code', 'like', '%' . $request->search . '%');
            });
        }

        // Company Filter
        if ($request->company_id) {
            $query->where('company_id', $request->company_id);
        }

        $routes = $query->get();

        $headers = [
            'ID',
            'Company',
            'Route Code',
            'Name',
            'Sites',
            'Rate',
            'Description',
            'Active',
            'Created At',
            'Updated At'
        ];

        $data = $routes->map(function ($route) {
            return [
                $route->id,
                $route->company->name ?? 'N/A',
                $route->route_code,
                $route->name,
                $route->sites->pluck('name')->implode(', ') ?: 'N/A',
                $route->rate,
                $route->description,
                $route->active ? 'Yes' : 'No',
                $route->created_at->format('Y-m-d H:i'),
                $route->updated_at->format('Y-m-d H:i'),
            ];
        });

        $pdf = Pdf::loadView('pdf.sites_all', [
            'title' => 'Admin - Route Masters List',
            'headers' => $headers,
            'data' => $data
        ])->setPaper('a4', 'landscape');

        return $pdf->download('admin-routes-export-' . now()->format('Y-m-d') . '.pdf');
    }
    public function index(Request $request)
    {
        $query = Route::with(['company', 'sites']);

        // Search
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('route_code', 'like', '%' . $request->search . '%');
            });
        }

        // Company Filter
        if ($request->company_id) {
            $query->where('company_id', $request->company_id);
        }

        // Sort
        $sort = $request->get('sort', 'name');
        $direction = $request->get('direction', 'asc');
        $query->orderBy($sort, $direction);

        $routes = $query->paginate(100);
        $companies = Company::orderBy('name')->get();

        return view('admin.routes.index', compact('routes', 'companies'));
    }

    public function toggleStatus(Route $route)
    {
        $route->active = !$route->active;
        $route->save();

        return response()->json([
            'success' => true,
            'active' => $route->active,
            'message' => 'Route status updated successfully!'
        ]);
    }

    public function destroy(Route $route)
    {
        $route->delete();
        return redirect()->route('admin.routes.index')->with('success', 'Route deleted successfully!');
    }
}
