<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Position;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class PositionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Position::query();

        // Search
        if ($request->has('search') && $request->search) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Sorting
        $sort = $request->get('sort', 'created_at_desc');
        switch ($sort) {
            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            default:
                $query->orderBy('created_at', 'desc');
                break;
        }

        $positions = $query->paginate(100);

        return view('admin.positions.index', compact('positions'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.positions.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:positions,name',
        ]);

        Position::create([
            'name' => $request->name,
            'active' => true
        ]);

        return redirect()->route('admin.positions.index')->with('success', 'Duty Position created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Position $position)
    {
        return view('admin.positions.show', compact('position'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Position $position)
    {
        return view('admin.positions.edit', compact('position'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Position $position)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:positions,name,' . $position->id,
            'active' => 'boolean'
        ]);

        $position->update([
            'name' => $request->name,
            'active' => $request->has('active') ? $request->active : $position->active
        ]);

        return redirect()->route('admin.positions.index')->with('success', 'Duty Position updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Position $position)
    {
        $position->delete();
        return redirect()->route('admin.positions.index')->with('success', 'Duty Position deleted successfully.');
    }

    public function toggleStatus(Position $position)
    {
        $position->active = !$position->active;
        $position->save();

        return response()->json([
            'message' => 'Duty Position status updated successfully.',
            'active' => $position->active
        ]);
    }

    public function exportPDF()
    {
        $positions = Position::orderBy('name')->get();
        $pdf = Pdf::loadView('pdf.positions', compact('positions'));
        return $pdf->download('duty_positions.pdf');
    }
}
