<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\JobRole;
use Illuminate\Http\Request;

use Barryvdh\DomPDF\Facade\Pdf;

class JobRoleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');
        $sort = $request->input('sort', 'latest');

        $query = \App\Models\JobRole::query();

        if ($search) {
            $query->where('name', 'like', "%{$search}%");
        }

        switch ($sort) {
            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            case 'oldest':
                $query->orderBy('created_at', 'asc');
                break;
            default:
                $query->latest();
                break;
        }

        $jobRoles = $query->paginate(100)->withQueryString();
        return view('admin.job-roles.index', compact('jobRoles', 'search'));
    }

    public function exportPDF()
    {
        $jobRoles = \App\Models\JobRole::latest()->get();
        $data = [];
        foreach ($jobRoles as $role) {
            $data[] = [
                $role->name,
                $role->active ? 'Active' : 'Inactive',
                $role->created_at->format('M d, Y')
            ];
        }

        $pdf = Pdf::loadView('admin.exports.pdf_template', [
            'title' => 'Job Roles List',
            'headers' => ['Name', 'Status', 'Created At'],
            'data' => $data
        ]);

        return $pdf->download('job-roles-' . date('Y-m-d') . '.pdf');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.job-roles.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:job_roles,name',
            'active' => 'boolean',
        ]);

        JobRole::create($validated);

        return redirect()->route('admin.job-roles.index')->with('success', 'Job Role created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(JobRole $jobRole)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(JobRole $jobRole)
    {
        return view('admin.job-roles.edit', compact('jobRole'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, JobRole $jobRole)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:job_roles,name,' . $jobRole->id,
            'active' => 'boolean',
        ]);

        $jobRole->update($validated);

        return redirect()->route('admin.job-roles.index')->with('success', 'Job Role updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(JobRole $jobRole)
    {
        $jobRole->delete();

        return redirect()->route('admin.job-roles.index')->with('success', 'Job Role deleted successfully.');
    }

    public function toggleStatus(JobRole $jobRole)
    {
        $jobRole->update(['active' => !$jobRole->active]);
        return response()->json(['success' => true, 'active' => (bool)$jobRole->active]);
    }
}
