<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\IncidentCategory;
use App\Models\IncidentParentCategory;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\IncidentParentCategoryImport;
use App\Imports\IncidentCategoryImport;

class IncidentCategoryController extends Controller
{
    public function index()
    {
        $parents = IncidentParentCategory::with('incident_categories')->get();
        // Determine active tab or separate lists
        return view('admin.incident_categories.index', compact('parents'));
    }

    // Parent Categories
    public function storeParent(Request $request)
    {
        $request->validate([
            'code' => 'required|string|unique:incident_parent_categories,code',
            'description' => 'nullable|string',
        ]);

        IncidentParentCategory::create([
            'code' => $request->code,
            'description' => $request->description,
            'status' => true,
        ]);

        $redirect = back()->with('success', 'Parent Category created successfully');

        if ($request->has('add_another')) {
            $redirect->with('show_parent_form', true);
        }

        return $redirect;
    }

    public function updateParent(Request $request, IncidentParentCategory $parent)
    {
        $request->validate([
            'code' => 'required|string|unique:incident_parent_categories,code,' . $parent->id,
            'description' => 'nullable|string',
        ]);

        $parent->update([
            'code' => $request->code,
            'description' => $request->description,
        ]);

        return back()->with('success', 'Parent Category updated successfully');
    }

    public function toggleStatusParent(IncidentParentCategory $parent)
    {
        $parent->update(['status' => !$parent->status]);
        return response()->json(['success' => true]);
    }

    public function destroyParent(IncidentParentCategory $parent)
    {
        $parent->delete();
        return back()->with('success', 'Parent Category deleted successfully');
    }

    public function importParents(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,csv,xls',
        ]);

        try {
            Excel::import(new IncidentParentCategoryImport, $request->file('file'));
            return back()->with('success', 'Parent Categories imported successfully');
        } catch (\Exception $e) {
            return back()->with('error', 'Error importing file: ' . $e->getMessage());
        }
    }

    // Child Categories
    public function storeCategory(Request $request)
    {
        $request->validate([
            'incident_parent_category_id' => 'required|exists:incident_parent_categories,id',
            'description' => 'required|string',
            'code' => 'nullable|string',
            'priority_level' => 'required|integer|min:1|max:5',
        ]);

        IncidentCategory::create([
            'incident_parent_category_id' => $request->incident_parent_category_id,
            'description' => $request->description,
            'code' => $request->code,
            'priority_level' => $request->priority_level,
            'status' => true,
        ]);

        $redirect = back()->with('success', 'Incident Category created successfully');

        if ($request->has('add_another')) {
            $redirect->with('show_child_form', true);
        }

        return $redirect;
    }

    public function updateCategory(Request $request, IncidentCategory $category)
    {
        $request->validate([
            'incident_parent_category_id' => 'required|exists:incident_parent_categories,id',
            'description' => 'required|string',
            'code' => 'nullable|string',
            'priority_level' => 'required|integer|min:1|max:5',
        ]);

        $category->update([
            'incident_parent_category_id' => $request->incident_parent_category_id,
            'description' => $request->description,
            'code' => $request->code,
            'priority_level' => $request->priority_level,
        ]);

        return back()->with('success', 'Incident Category updated successfully');
    }

    public function toggleStatusCategory(IncidentCategory $category)
    {
        $category->update(['status' => !$category->status]);
        return response()->json(['success' => true]);
    }

    public function destroyCategory(IncidentCategory $category)
    {
        $category->delete();
        return back()->with('success', 'Incident Category deleted successfully');
    }

    public function importCategories(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,csv,xls',
        ]);

        try {
            Excel::import(new IncidentCategoryImport, $request->file('file'));
            return back()->with('success', 'Incident Categories imported successfully');
        } catch (\Exception $e) {
            return back()->with('error', 'Error importing file: ' . $e->getMessage());
        }
    }
}
