<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use Barryvdh\DomPDF\Facade\Pdf;

class CountryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');
        $sort = $request->input('sort', 'latest');

        $query = \App\Models\Country::query();

        if ($search) {
            $query->where('name', 'like', "%{$search}%")
                  ->orWhere('iso2', 'like', "%{$search}%")
                  ->orWhere('iso3', 'like', "%{$search}%");
        }

        switch ($sort) {
            case 'name_asc': $query->orderBy('name', 'asc'); break;
            case 'name_desc': $query->orderBy('name', 'desc'); break;
            case 'oldest': $query->orderBy('created_at', 'asc'); break;
            default: $query->latest(); break;
        }

        $countries = $query->paginate(100)->withQueryString();
        return view('admin.countries.index', compact('countries', 'search'));
    }

    public function exportPDF()
    {
        $countries = \App\Models\Country::latest()->get();
        $data = [];
        foreach ($countries as $country) {
            $data[] = [
                $country->name,
                $country->iso2,
                $country->iso3,
                $country->phone_code,
                $country->active ? 'Active' : 'Inactive'
            ];
        }

        $pdf = Pdf::loadView('admin.exports.pdf_template', [
            'title' => 'Countries List',
            'headers' => ['Name', 'ISO2', 'ISO3', 'Phone Code', 'Status'],
            'data' => $data
        ]);

        return $pdf->download('countries-' . date('Y-m-d') . '.pdf');
    }

    public function create()
    {
        return view('admin.countries.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:countries',
            'iso2' => 'nullable|string|max:2',
            'iso3' => 'nullable|string|max:3',
            'phone_code' => 'nullable|string|max:10',
            'active' => 'boolean',
        ]);

        \App\Models\Country::create($validated);

        return redirect()->route('admin.countries.index')
            ->with('success', 'Country created successfully.');
    }

    public function edit(\App\Models\Country $country)
    {
        return view('admin.countries.edit', compact('country'));
    }

    public function update(Request $request, \App\Models\Country $country)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:countries,name,' . $country->id,
            'iso2' => 'nullable|string|max:2',
            'iso3' => 'nullable|string|max:3',
            'phone_code' => 'nullable|string|max:10',
            'active' => 'boolean',
        ]);

        $country->update($validated);

        return redirect()->route('admin.countries.index')
            ->with('success', 'Country updated successfully.');
    }

    public function destroy(\App\Models\Country $country)
    {
        $country->delete();
        return redirect()->route('admin.countries.index')
            ->with('success', 'Country deleted successfully.');
    }

    public function toggleStatus(\App\Models\Country $country)
    {
        $country->update(['active' => !$country->active]);
        return response()->json(['success' => true, 'active' => (bool)$country->active]);
    }
}
