<?php

use App\Http\Controllers\Admin\AdminAuthController;
use App\Http\Controllers\Admin\CompanyController;
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\TimeZoneController;
use App\Http\Controllers\Admin\UserController;
use App\Http\Controllers\AuthController;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\PublicSiteController;

Route::get('/', [PublicSiteController::class, 'index'])->name('landing');
Route::get('/about-us', [PublicSiteController::class, 'about'])->name('about');
Route::get('/contact-us', [PublicSiteController::class, 'contact'])->name('contact');
Route::get('/become-a-guard', [PublicSiteController::class, 'becomeGuard'])->name('become-guard');
Route::post('/become-a-guard', [PublicSiteController::class, 'storeGuardApplication'])->name('become-guard.store');

Route::group(['middleware' => 'guest'], function () {
    Route::get('/login', [AuthController::class, 'showLogin'])->name('login');
    Route::post('/login', [AuthController::class, 'login']);

    // Password Reset Routes
    Route::get('/forgot-password', [\App\Http\Controllers\Auth\PasswordResetController::class, 'showLinkRequestForm'])->name('password.request');
    Route::post('/forgot-password', [\App\Http\Controllers\Auth\PasswordResetController::class, 'sendResetLinkEmail'])->name('password.email');
    Route::get('/reset-password/{token}', [\App\Http\Controllers\Auth\PasswordResetController::class, 'showResetForm'])->name('password.reset');
    Route::post('/reset-password', [\App\Http\Controllers\Auth\PasswordResetController::class, 'reset'])->name('password.update');
});

use App\Http\Controllers\Auth\CompanySelectionController;
use App\Http\Controllers\Admin\DepartmentController;

Route::group(['middleware' => ['auth', '2fa']], function () {
    Route::post('/logout', [AuthController::class, 'logout'])->name('logout');

    // Google Authenticator (2FA) Routes
    Route::controller(\App\Http\Controllers\TwoFactorController::class)->prefix('2fa')->name('2fa.')->group(function () {
        Route::get('/setup', 'setup')->name('setup');
        Route::post('/setup', 'storeSetup')->name('storeSetup');
        Route::get('/verify', 'verify')->name('verify');
        Route::post('/verify', 'storeVerify')->name('storeVerify');
    });

    // Uniforms Inventory (User View)
    Route::get('/uniforms', [\App\Http\Controllers\UniformController::class, 'index'])->name('user.uniforms.index');

    // Real-time Meetings & Discussion Forum
    Route::prefix('meetings')->name('meetings.')->group(function () {
        Route::get('/', [\App\Http\Controllers\MeetingController::class, 'index'])->name('index');
        Route::get('/create', [\App\Http\Controllers\MeetingController::class, 'create'])->name('create');
        Route::post('/', [\App\Http\Controllers\MeetingController::class, 'store'])->name('store');
        Route::get('/join/{uuid}', [\App\Http\Controllers\MeetingController::class, 'join'])->name('join');
        Route::get('/room/{uuid}', [\App\Http\Controllers\MeetingController::class, 'meetingRoom'])->name('room');
        Route::post('/room/{uuid}/approve', [\App\Http\Controllers\MeetingController::class, 'approve'])->name('approve');
        Route::post('/room/{uuid}/request-join', [\App\Http\Controllers\MeetingController::class, 'requestJoin'])->name('request-join');
        Route::post('/room/{uuid}/message', [\App\Http\Controllers\MeetingController::class, 'sendMessage'])->name('message');
        Route::post('/room/{uuid}/end', [\App\Http\Controllers\MeetingController::class, 'endMeeting'])->name('end');
    });

    // API Routes
    Route::get('/api/countries/{country}/states', function ($countryId) {
        return \App\Models\State::where('country_id', $countryId)->orderBy('name')->get();
    });

    Route::get('/select-company', [CompanySelectionController::class, 'index'])->name('select.company');
    Route::post('/select-company', [CompanySelectionController::class, 'store'])->name('select.company.store');

    Route::middleware([\App\Http\Middleware\EnsureCompanySelected::class])->group(function () {
        Route::view('/tutorials', 'tutorials.index')->name('tutorials.index');
        Route::get('/activity-feed', [\App\Http\Controllers\LiveActivityController::class, 'feed'])->name('activity-feed');
        Route::get('/test-mailgun', function () {
            try {
                $config = [
                    'Mailer' => config('mail.default'),
                    'Domain' => config('services.mailgun.domain'),
                    'Endpoint' => config('services.mailgun.endpoint'),
                    'From' => config('mail.from.address'),
                ];

                $user = auth()->user();
                $to = $user ? $user->email : config('mail.from.address');

                \Illuminate\Support\Facades\Mail::raw("Test email from server.\n\nConfig:\n" . print_r($config, true), function ($message) use ($to) {
                    $message->to($to)
                        ->subject('Test Email Server - ' . now());
                });

                return response()->json([
                    'status' => 'success',
                    'message' => 'Email sent successfully to ' . $to,
                    'config' => $config
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'status' => 'error',
                    'message' => $e->getMessage(),
                    'config' => $config ?? 'Unable to fetch',
                    'trace' => $e->getTraceAsString()
                ], 500);
            }
        });
        Route::get('/dashboard', function () {
            $company_id = session('selected_company_id');
            $employeeCount = \App\Models\Employee::where('company_id', $company_id)->count();
            $siteCount = \App\Models\Site::where('company_id', $company_id)->count();
            $routeCount = \App\Models\Route::where('company_id', $company_id)->count();

            // Today's Schedules: Count of schedules covering today
            $today = now()->format('Y-m-d');
            $todaysScheduleCount = \App\Models\Schedule::where('company_id', $company_id)
                ->whereDate('schedule_date', $today)
                ->count();

            // Upcoming Schedules: Count of schedules starting after today
            $upcomingScheduleCount = \App\Models\Schedule::where('company_id', $company_id)
                ->whereDate('schedule_date', '>', $today)
                ->where('status', '!=', 'active')
                ->where('status', '!=', 'completed')
                ->count();


            // Patroller Stats
            $todaysPatrollerCount = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->whereDate('from_time', $today)
                ->count();

            $upcomingPatrollerCount = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->whereDate('from_time', '>', $today)
                ->whereNotIn('job_status', ['in_progress', 'completed', 'auto_ended'])
                ->count();

            $activePatrollerCount = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->where('job_status', 'in_progress')
                ->count();

            $pendingPatrollerCount = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->whereDate('from_time', $today)
                ->where('job_status', 'pending')
                ->count();

            $completedPatrollerCount = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->whereDate('from_time', $today)
                ->whereIn('job_status', ['completed', 'auto_ended'])
                ->count();

            // Cancellation Requests (Guard)
            $cancellationRequests = \App\Models\Schedule::where('company_id', $company_id)
                ->whereHas('employees', function ($q) {
                    $q->where('employee_schedule.cancellation_status', 'pending');
                })
                ->with([
                    'site',
                    'employees' => function ($q) {
                        $q->where('employee_schedule.cancellation_status', 'pending');
                    }
                ])
                ->get();

            // Cancellation Requests (Patroller)
            $patrollerCancellationRequests = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->whereHas('employees', function ($q) {
                    $q->where('employee_patroller_schedule.cancellation_status', 'pending');
                })
                ->with([
                    'route',
                    'employees' => function ($q) {
                        $q->where('employee_patroller_schedule.cancellation_status', 'pending');
                    }
                ])
                ->get();

            // Late Check-ins Logic
            $lateTolerance = (int) \App\Models\Setting::where('key', 'late_checkin_tolerance')->value('value') ?: 10;

            $lateCheckins = \App\Models\Schedule::query()
                ->where('company_id', $company_id)
                ->whereHas('employees', function ($q) use ($lateTolerance) {
                    $q->whereNotNull('actual_start_at')
                        ->where('actual_start_at', '>', \Illuminate\Support\Facades\DB::raw("DATE_ADD(schedules.from_datetime, INTERVAL {$lateTolerance} MINUTE)"));
                })
                ->where('from_datetime', '>=', now()->subDays(7)) // Last 7 days
                ->with(['employees', 'site'])
                ->orderBy('from_datetime', 'desc')
                ->limit(10)
                ->get();

            // Expired Licenses Logic
            // 1. Get employees with expired main license
            $expiredMain = \App\Models\Employee::where('company_id', $company_id)
                ->whereDate('license_expiry', '<', now())
                ->get()
                ->map(function ($emp) {
                    // Create a standardized object/array mimicking an Identification model
                    $obj = new \stdClass();
                    $obj->employee_id = $emp->id;
                    $obj->employee = $emp;
                    $obj->type = 'Security License (Main)';
                    $obj->identification_number = $emp->license_number;
                    $obj->expiry_date = $emp->license_expiry;
                    return $obj;
                });

            // 2. Get expired additional identifications
            $expiredAdditional = \App\Models\EmployeeIdentification::whereHas('employee', function ($q) use ($company_id) {
                $q->where('company_id', $company_id);
            })
                ->whereDate('expiry_date', '<', now())
                ->with('employee')
                ->get();

            // 3. Merge and Sort
            $allExpired = $expiredMain->merge($expiredAdditional)->sortByDesc('expiry_date');

            // 4. Manual Pagination
            $page = request()->get('licenses_page', 1);
            $perPage = 5;
            $expiredLicenses = new \Illuminate\Pagination\LengthAwarePaginator(
                $allExpired->forPage($page, $perPage),
                $allExpired->count(),
                $perPage,
                $page,
                ['path' => request()->url(), 'pageName' => 'licenses_page']
            );

            // Active Jobs (Pagination)
            $activeJobs = \App\Models\Schedule::where('company_id', $company_id)
                ->where('status', 'active')
                ->with(['site', 'employees'])
                ->orderBy('from_datetime', 'asc')
                ->paginate(5, ['*'], 'active_jobs_page');

            // Active Patroller Jobs
            $activePatrollerJobs = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->where('job_status', 'in_progress')
                ->with(['route.sites', 'employees'])
                ->orderBy('scheduled_date', 'asc')
                ->paginate(5, ['*'], 'active_patroller_jobs_page');

            // Calendar View: All Today's Guard Jobs (for calendar display)
            $todayCalendarJobs = \App\Models\Schedule::where('company_id', $company_id)
                ->whereDate('schedule_date', $today)
                ->with(['site', 'employees'])
                ->orderBy('from_datetime', 'asc')
                ->get();

            // Calendar View: All Today's Patroller Jobs (for calendar display)
            $todayCalendarPatrollerJobs = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->whereDate('scheduled_date', $today)
                ->with(['route.sites', 'employees'])
                ->orderBy('from_time', 'asc')
                ->get();

            // Scheduled Jobs (Pending Start) - Today Only
            $scheduledJobsToday = \App\Models\Schedule::where('company_id', $company_id)
                ->whereDate('schedule_date', $today)
                ->where(function ($q) {
                    $q->whereIn('status', ['draft', 'upcoming', 'published', 'scheduled', 'today']);
                })
                ->whereDoesntHave('employees', function ($q) {
                    $q->whereNotNull('actual_start_at'); // Filter out if ANY employee started
                })
                ->with(['site', 'employees']) // Load jobRole if needed for phone numbers (employees relation has phone)
                ->orderBy('from_datetime', 'asc')
                ->paginate(5, ['*'], 'scheduled_jobs_page');

            // Completed Jobs - Today Only
            $completedJobsToday = \App\Models\Schedule::where('company_id', $company_id)
                ->whereDate('schedule_date', $today)
                ->where('status', 'completed')
                ->whereHas('employees', function ($q) {
                    $q->whereNotNull('actual_end_at');
                })
                ->with(['site', 'employees'])
                ->orderBy('to_datetime', 'desc')
                ->paginate(5, ['*'], 'completed_jobs_page');

            // Scheduled Patroller Jobs (Pending Start) - Today Only
            $scheduledPatrollerJobsToday = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->whereDate('from_time', $today)
                ->where('job_status', 'pending')
                ->with(['route.sites', 'employees'])
                ->orderBy('from_time', 'asc')
                ->paginate(5, ['*'], 'scheduled_patroller_jobs_page');

            // Completed Patroller Jobs - Today Only
            $completedPatrollerJobsToday = \App\Models\PatrollerSchedule::where('company_id', $company_id)
                ->whereDate('from_time', $today)
                ->whereIn('job_status', ['completed', 'auto_ended'])
                ->with(['route.sites', 'employees'])
                ->orderBy('from_time', 'desc')
                ->paginate(5, ['*'], 'completed_patroller_jobs_page');

            // Upcoming Birthdays (Next 7 days, including today)
            $upcomingBirthdays = \App\Models\Employee::where('company_id', $company_id)
                ->where('active', true)
                ->whereNotNull('dob')
                ->whereRaw("DATE_FORMAT(dob, '%m-%d') BETWEEN DATE_FORMAT(NOW(), '%m-%d') AND DATE_FORMAT(DATE_ADD(NOW(), INTERVAL 7 DAY), '%m-%d')")
                ->orderByRaw("DATE_FORMAT(dob, '%m-%d') ASC")
                ->get();

            // Upcoming Work Anniversaries (Next 7 days, including today)
            $upcomingAnniversaries = \App\Models\Employee::where('company_id', $company_id)
                ->where('active', true)
                ->whereNotNull('employment_date')
                ->whereRaw("DATE_FORMAT(employment_date, '%m-%d') BETWEEN DATE_FORMAT(NOW(), '%m-%d') AND DATE_FORMAT(DATE_ADD(NOW(), INTERVAL 7 DAY), '%m-%d')")
                ->orderByRaw("DATE_FORMAT(employment_date, '%m-%d') ASC")
                ->get();

            // Guard Applications
            $guardApplications = \App\Models\GuardApplication::orderBy('created_at', 'desc')->get();
            $emailTemplates = \App\Models\EmailTemplate::all();

            // Refresh Settings
            $refreshEnabled = \App\Models\Setting::where('key', 'dashboard_refresh_enabled')->value('value');
            $refreshInterval = \App\Models\Setting::where('key', 'dashboard_refresh_interval')->value('value') ?? 5;

            return view('dashboard', compact(
                'employeeCount',
                'siteCount',
                'routeCount',
                'todaysScheduleCount',
                'upcomingScheduleCount',
                'cancellationRequests',
                'patrollerCancellationRequests',
                'lateCheckins',
                'lateTolerance',
                'expiredLicenses',
                'activeJobs',
                'scheduledJobsToday',
                'completedJobsToday',
                'todayCalendarJobs',
                'todayCalendarPatrollerJobs',
                'todaysPatrollerCount',
                'upcomingPatrollerCount',
                'activePatrollerCount',
                'pendingPatrollerCount',
                'completedPatrollerCount',
                'activePatrollerJobs',
                'scheduledPatrollerJobsToday',
                'completedPatrollerJobsToday',
                'upcomingBirthdays',
                'upcomingAnniversaries',
                'refreshEnabled',
                'refreshInterval',
                'guardApplications',
                'emailTemplates'
            ));
        })->name('dashboard');

        // Send Applicant Email
        Route::post('/dashboard/send-applicant-email', function (\Illuminate\Http\Request $request) {
            $request->validate([
                'applicant_id' => 'required|exists:guard_applications,id',
                'email_template_id' => 'required|exists:email_templates,id',
            ]);

            $applicant = \App\Models\GuardApplication::findOrFail($request->applicant_id);
            $template = \App\Models\EmailTemplate::findOrFail($request->email_template_id);

            // Replace placeholders
            $content = $template->content;
            $content = str_replace('{name}', $applicant->full_name, $content);
            // Add more replacements as needed

            // Send Email
            \Illuminate\Support\Facades\Mail::html($content, function ($message) use ($applicant, $template) {
                $message->to($applicant->email)
                    ->subject($template->subject);
            });

            return back()->with('success', 'Email sent successfully to ' . $applicant->full_name);
        })->name('dashboard.send-applicant-email');

        // Help Guide Route
        // Help Guide Route
        Route::get('/help', [\App\Http\Controllers\UserHelpController::class, 'index'])->name('user.help');
        Route::post('/help/report', [\App\Http\Controllers\UserHelpController::class, 'reportIssue'])->name('user.help.report');


        // Analytics Route for User Panel
        Route::get('/analytics', [\App\Http\Controllers\Admin\UserController::class, 'myAnalytics'])->name('user.analytics');


        // User Notifications Route
        Route::post('notifications/send', [\App\Http\Controllers\Admin\NotificationController::class, 'store'])->name('notifications.store');
        Route::get('/notifications', [\App\Http\Controllers\UserNotificationController::class, 'index'])->name('notifications.index');
        Route::get('/notifications/{id}/read', [\App\Http\Controllers\UserNotificationController::class, 'read'])->name('notifications.read');
        Route::post('/notifications/{id}/mark-read', [\App\Http\Controllers\UserNotificationController::class, 'markAsRead'])->name('notifications.markAsRead');
        Route::middleware(['can:view employees'])->group(function () {
            // Employee Import Routes
            Route::get('employees/import/settings', [\App\Http\Controllers\EmployeeImportController::class, 'showImportSettings'])->name('employees.import.settings');
            Route::get('employees/import/preview', [\App\Http\Controllers\EmployeeImportController::class, 'redirectToSettings']);
            Route::post('employees/import/preview', [\App\Http\Controllers\EmployeeImportController::class, 'previewCsv'])->name('employees.import.preview');

            // Site Import Routes
            Route::get('sites/import/settings', [\App\Http\Controllers\SiteImportController::class, 'showImportSettings'])->name('sites.import.settings');
            Route::post('sites/import/preview', [\App\Http\Controllers\SiteImportController::class, 'previewCsv'])->name('sites.import.preview');
            Route::post('sites/import/process', [\App\Http\Controllers\SiteImportController::class, 'processImport'])->name('sites.import.process');
            Route::get('sites/import/template', [\App\Http\Controllers\SiteImportController::class, 'downloadTemplate'])->name('sites.import.template');

            Route::post('employees/import/process', [\App\Http\Controllers\EmployeeImportController::class, 'processImport'])->name('employees.import.process');
            Route::get('employees/import/template', [\App\Http\Controllers\EmployeeImportController::class, 'downloadTemplate'])->name('employees.import.template');

            Route::get('employees/export-excel', [\App\Http\Controllers\EmployeeController::class, 'exportExcel'])->name('employees.export-excel');
            Route::get('employees/export-pdf', [\App\Http\Controllers\EmployeeController::class, 'exportPDF'])->name('employees.export-pdf');
            Route::post('employees/import-csv', [\App\Http\Controllers\EmployeeController::class, 'importCsv'])->name('employees.import-csv');
            Route::post('employees/bulk-update', [\App\Http\Controllers\EmployeeController::class, 'bulkUpdate'])->name('employees.bulk-update');
            Route::post('employees/copy', [\App\Http\Controllers\EmployeeController::class, 'copy'])->name('employees.copy');
            Route::post('employees/validate', [\App\Http\Controllers\EmployeeController::class, 'validateForm'])->name('employees.validate');
            Route::resource('employees', \App\Http\Controllers\EmployeeController::class);
        });

        Route::get('/employees/download-sample', [\App\Http\Controllers\EmployeeController::class, 'downloadSample'])->name('employees.download-sample');
        Route::patch('/employees/{employee}/toggle-status', [\App\Http\Controllers\EmployeeController::class, 'toggleStatus'])->name('employees.toggle-status');
        Route::post('/employees/{employee}/availability', [\App\Http\Controllers\EmployeeController::class, 'updateAvailability'])->name('employees.update_availability');
        Route::post('/employees/{employee}/assign-uniform', [\App\Http\Controllers\EmployeeController::class, 'assignUniform'])->name('employees.assign-uniform');
        Route::post('/employee/uniforms/assign', [\App\Http\Controllers\Employee\EmployeeUniformController::class, 'assign'])->name('employee.uniforms.assign');
        Route::get('/employee/uniforms', [\App\Http\Controllers\Employee\EmployeeUniformController::class, 'index'])->name('employee.uniforms.index');
        Route::get('/employees/{employee}/analytics', [\App\Http\Controllers\EmployeeController::class, 'analytics'])->name('employees.analytics');
        // Employee Utility Routes (outside resource)
        Route::get('/employees/{employee}/schedules', [\App\Http\Controllers\EmployeeController::class, 'schedules'])->name('employees.schedules');
        Route::put('/employees/{employee}/change-password', [\App\Http\Controllers\EmployeeController::class, 'changePassword'])->name('employees.change_password');
        Route::post('/employees/{employee}/upload-photo', [\App\Http\Controllers\EmployeeController::class, 'uploadPhoto'])->name('employees.upload_photo');
        Route::get('/employees/{employee}/id-card', [\App\Http\Controllers\EmployeeController::class, 'idCard'])->name('employees.id_card');
        Route::post('/employees/{employee}/ban-sites', [\App\Http\Controllers\EmployeeController::class, 'banSites'])->name('employees.ban-sites');
        // Sites Management
        Route::middleware(['can:view sites'])->group(function () {
            Route::get('sites/export-pdf', [\App\Http\Controllers\SiteController::class, 'exportPDF'])->name('sites.export-pdf');
            Route::get('sites/export-excel', [\App\Http\Controllers\SiteController::class, 'exportExcel'])->name('sites.export-excel');
            Route::get('sites', [\App\Http\Controllers\SiteController::class, 'index'])->name('sites.index');
            Route::middleware(['can:create site'])->group(function () {
                Route::get('sites/create', [\App\Http\Controllers\SiteController::class, 'create'])->name('sites.create');
                Route::post('sites/validate', [\App\Http\Controllers\SiteController::class, 'validateForm'])->name('sites.validate');
                Route::post('sites', [\App\Http\Controllers\SiteController::class, 'store'])->name('sites.store');
                Route::post('sites/copy', [\App\Http\Controllers\SiteController::class, 'copy'])->name('sites.copy');
                Route::post('sites/bulk-update', [\App\Http\Controllers\SiteController::class, 'bulkUpdate'])->name('sites.bulk-update');
            });
            Route::get('sites/{site}', [\App\Http\Controllers\SiteController::class, 'show'])->name('sites.show');

            Route::middleware(['can:edit site'])->group(function () {
                Route::get('sites/{site}/edit', [\App\Http\Controllers\SiteController::class, 'edit'])->name('sites.edit');
                Route::put('sites/{site}', [\App\Http\Controllers\SiteController::class, 'update'])->name('sites.update');
                Route::patch('sites/{site}/toggle-status', [\App\Http\Controllers\SiteController::class, 'toggleStatus'])->name('sites.toggle-status');
                Route::post('sites/{site}/ban/{employee}', [\App\Http\Controllers\SiteController::class, 'banEmployee'])->name('sites.ban');
                Route::delete('sites/{site}/ban/{employee}', [\App\Http\Controllers\SiteController::class, 'unbanEmployee'])->name('sites.unban');
            });

            Route::middleware(['can:delete site'])->group(function () {
                Route::delete('sites/{site}', [\App\Http\Controllers\SiteController::class, 'destroy'])->name('sites.destroy');
            });
        });

        // Checkpoints & Tour Routes (nested under sites)
        Route::prefix('sites/{site}')->group(function () {
            Route::get('checkpoints', [\App\Http\Controllers\CheckpointController::class, 'index'])->name('sites.checkpoints.index');
            Route::post('checkpoints', [\App\Http\Controllers\CheckpointController::class, 'store'])->name('sites.checkpoints.store');
            Route::put('checkpoints/{checkpoint}', [\App\Http\Controllers\CheckpointController::class, 'update'])->name('sites.checkpoints.update');
            Route::delete('checkpoints/{checkpoint}', [\App\Http\Controllers\CheckpointController::class, 'destroy'])->name('sites.checkpoints.destroy');

            Route::get('tour-routes', [\App\Http\Controllers\TourRouteController::class, 'index'])->name('sites.tour-routes.index');
            Route::post('tour-routes', [\App\Http\Controllers\TourRouteController::class, 'store'])->name('sites.tour-routes.store');
            Route::put('tour-routes/{tourRoute}', [\App\Http\Controllers\TourRouteController::class, 'update'])->name('sites.tour-routes.update');
            Route::delete('tour-routes/{tourRoute}', [\App\Http\Controllers\TourRouteController::class, 'destroy'])->name('sites.tour-routes.destroy');
        });


        // Vehicle Management
        Route::resource('vehicles', \App\Http\Controllers\VehicleController::class);
        Route::delete('vehicles/{vehicle}/images/{index}', [\App\Http\Controllers\VehicleController::class, 'deleteImage'])->name('vehicles.delete-image');

        // Analytics
        Route::get('analytics', [\App\Http\Controllers\AnalyticsController::class, 'index'])->name('analytics.index');

        // Route Masters Management
        Route::middleware(['can:view routes'])->group(function () {
            Route::get('routes/export-pdf', [\App\Http\Controllers\RouteController::class, 'exportPdf'])->name('routes.export-pdf');
            Route::get('routes/export-excel', [\App\Http\Controllers\RouteController::class, 'exportExcel'])->name('routes.export-excel');
            Route::patch('/routes/{route}/toggle-status', [\App\Http\Controllers\RouteController::class, 'toggleStatus'])->name('routes.toggle-status');
            Route::resource('routes', \App\Http\Controllers\RouteController::class);
        });

        // Schedules Management
        Route::middleware(['can:view schedules'])->group(function () {
            Route::get('schedules/employee-info/{id}', [\App\Http\Controllers\ScheduleController::class, 'getEmployeeInfo'])->name('schedules.employee-info');
            Route::get('schedules/site-info/{id}', [\App\Http\Controllers\ScheduleController::class, 'getSiteInfo'])->name('schedules.site-info');
            Route::get('schedules/export', [\App\Http\Controllers\ScheduleController::class, 'exportPdf'])->name('schedules.export');
            Route::get('schedules/export-excel', [\App\Http\Controllers\ScheduleController::class, 'exportExcel'])->name('schedules.export.excel');
            Route::post('schedules/email-client', [\App\Http\Controllers\ScheduleController::class, 'emailFilteredSchedules'])->name('schedules.email-client');
            Route::get('schedules/{schedule}/report/view', [\App\Http\Controllers\ScheduleController::class, 'viewReport'])->name('schedules.report.view');
            Route::get('schedules/{schedule}/report/download', [\App\Http\Controllers\ScheduleController::class, 'downloadReport'])->name('schedules.report.download');
            Route::post('schedules/{schedule}/report/email', [\App\Http\Controllers\ScheduleController::class, 'sendReportEmail'])->name('schedules.report.email');
            Route::get('schedules/{schedule}/report/zip', [\App\Http\Controllers\ScheduleController::class, 'downloadZip'])->name('schedules.report.zip');
            Route::get('schedules', [\App\Http\Controllers\ScheduleController::class, 'index'])->name('schedules.index');

            // Moved create before show to prevent wildcard conflict
            Route::middleware(['can:create schedule'])->group(function () {
                Route::get('schedules/create', [\App\Http\Controllers\ScheduleController::class, 'create'])->name('schedules.create');
                Route::post('schedules', [\App\Http\Controllers\ScheduleController::class, 'store'])->name('schedules.store');
            });

            Route::get('schedules/{schedule}', [\App\Http\Controllers\ScheduleController::class, 'show'])->name('schedules.show');

            Route::middleware(['can:edit schedule'])->group(function () {
                Route::get('schedules/{schedule}/edit', [\App\Http\Controllers\ScheduleController::class, 'edit'])->name('schedules.edit');
                Route::put('schedules/{schedule}', [\App\Http\Controllers\ScheduleController::class, 'update'])->name('schedules.update');
                Route::patch('schedules/{schedule}/toggle-status', [\App\Http\Controllers\ScheduleController::class, 'toggleStatus'])->name('schedules.toggle-status');
                Route::post('schedules/{schedule}/cancel/approve/{employee}', [\App\Http\Controllers\ScheduleController::class, 'approveCancellation'])->name('schedules.cancel.approve');
                Route::post('schedules/{schedule}/cancel/reject/{employee}', [\App\Http\Controllers\ScheduleController::class, 'rejectCancellation'])->name('schedules.cancel.reject');
                Route::post('schedules/bulk-update', [\App\Http\Controllers\ScheduleController::class, 'bulkUpdate'])->name('schedules.bulk-update');
            });

            Route::delete('schedules/bulk-destroy', [\App\Http\Controllers\ScheduleController::class, 'bulkDestroy'])->name('schedules.bulk-destroy');
            Route::delete('schedules/{schedule}', [\App\Http\Controllers\ScheduleController::class, 'destroy'])->name('schedules.destroy');
        });

        Route::get('schedules/{schedule}/locations', [\App\Http\Controllers\ScheduleController::class, 'getLocationLogs'])->name('schedules.locations');

        // Patroller Schedules Management
        Route::middleware(['can:view schedules'])->group(function () {
            // API routes for patroller schedule creation
            Route::get('patroller-schedules/api/employee-wage-types/{employee}', [\App\Http\Controllers\PatrollerScheduleController::class, 'getEmployeeWageTypes'])->name('patroller-schedules.api.employee-wage-types');
            Route::get('patroller-schedules/api/route-details/{route}', [\App\Http\Controllers\PatrollerScheduleController::class, 'getRouteDetails'])->name('patroller-schedules.api.route-details');
            Route::post('patroller-schedules/api/check-employee-availability', [\App\Http\Controllers\PatrollerScheduleController::class, 'checkEmployeeAvailability'])->name('patroller-schedules.api.check-availability');
            Route::post('patroller-schedules/api/search-available-employees', [\App\Http\Controllers\PatrollerScheduleController::class, 'searchAvailableEmployees'])->name('patroller-schedules.api.search-available');

            Route::patch('patroller-schedules/{patroller_schedule}/cancel', [\App\Http\Controllers\PatrollerScheduleController::class, 'cancel'])->name('patroller-schedules.cancel');
            Route::post('patroller-schedules/{patroller_schedule}/cancel/approve/{employee}', [\App\Http\Controllers\PatrollerScheduleController::class, 'approveCancellation'])->name('patroller-schedules.cancel.approve');
            Route::post('patroller-schedules/{patroller_schedule}/cancel/reject/{employee}', [\App\Http\Controllers\PatrollerScheduleController::class, 'rejectCancellation'])->name('patroller-schedules.cancel.reject');
            Route::get('patroller-schedules/{patroller_schedule}/export-pdf', [\App\Http\Controllers\PatrollerScheduleController::class, 'exportSinglePdf'])->name('patroller-schedules.export-single-pdf');
            Route::get('patroller-schedules/{patroller_schedule}/export-images', [\App\Http\Controllers\PatrollerScheduleController::class, 'exportTicketImages'])->name('patroller-schedules.export-images');
            Route::get('patroller-schedules/{patroller_schedule}/locations', [\App\Http\Controllers\PatrollerScheduleController::class, 'getLocations'])->name('patroller-schedules.locations');
            Route::post('patroller-schedules/export-pdf', [\App\Http\Controllers\PatrollerScheduleController::class, 'exportPdf'])->name('patroller-schedules.export-pdf');
            Route::post('patroller-schedules/export-excel', [\App\Http\Controllers\PatrollerScheduleController::class, 'exportExcel'])->name('patroller-schedules.export-excel');
            Route::resource('patroller-schedules', \App\Http\Controllers\PatrollerScheduleController::class);
        });

        // User Profile
        Route::get('/profile/password', [\App\Http\Controllers\ProfileController::class, 'showPasswordForm'])->name('profile.password');
        Route::put('/profile/password', [\App\Http\Controllers\ProfileController::class, 'updatePassword'])->name('profile.password.update');

        Route::get('/activity-logs', [\App\Http\Controllers\ActivityLogController::class, 'index'])->name('activity-logs.index');
        Route::get('/activity-logs/export-pdf', [\App\Http\Controllers\ActivityLogController::class, 'exportPdf'])->name('activity-logs.export.pdf');
        Route::get('/activity-logs/export-excel', [\App\Http\Controllers\ActivityLogController::class, 'exportExcel'])->name('activity-logs.export.excel');


        Route::get('api/countries/{country}/states', [\App\Http\Controllers\LocationController::class, 'getStates']);

        // Global Search
        Route::get('/global-search', [\App\Http\Controllers\GlobalSearchController::class, 'index'])->name('global.search');

        // Email Templates (Company Level)
        Route::get('email-templates/{email_template}/preview', [\App\Http\Controllers\EmailTemplateController::class, 'preview'])->name('email-templates.preview');

        Route::resource('email-templates', \App\Http\Controllers\EmailTemplateController::class);

        // Email Campaigns
        Route::resource('email-campaigns', \App\Http\Controllers\Admin\EmailCampaignController::class);
        Route::post('email-campaigns/{email_campaign}/send', [\App\Http\Controllers\Admin\EmailCampaignController::class, 'send'])->name('email-campaigns.send');
        Route::get('email-campaigns/{email_campaign}/logs', [\App\Http\Controllers\Admin\EmailCampaignController::class, 'showLogs'])->name('email-campaigns.logs');

        // Employee Requests (Admin View)
        Route::resource('employee-requests', \App\Http\Controllers\Admin\EmployeeRequestController::class);
        Route::post('employee-requests/{employee_request}/reply', [\App\Http\Controllers\Admin\EmployeeRequestController::class, 'reply'])->name('employee-requests.reply');
        Route::patch('employee-requests/{employee_request}/status', [\App\Http\Controllers\Admin\EmployeeRequestController::class, 'updateStatus'])->name('employee-requests.status');

        // Add-on Schedules
        Route::get('addon-schedules/history', [\App\Http\Controllers\AddonScheduleController::class, 'history'])->name('addon-schedules.history');
        Route::post('addon-schedules/{schedule}/complete', [\App\Http\Controllers\AddonScheduleController::class, 'store'])->name('addon-schedules.store');
        Route::get('addon-schedules/{schedule}/employee/{employee_id}/download-evidence', [\App\Http\Controllers\AddonScheduleController::class, 'downloadEvidence'])->name('addon-schedules.download-evidence');
        Route::get('addon-schedules', [\App\Http\Controllers\AddonScheduleController::class, 'index'])->name('addon-schedules.index');

        // Customer Care / Guard Check-in
        Route::prefix('care')->name('care.')->group(function () {
            Route::get('/active-guards', [\App\Http\Controllers\GuardCheckinController::class, 'activeGuards'])->name('active-guards');
            Route::get('/active-guards/export', [\App\Http\Controllers\GuardCheckinController::class, 'exportPdf'])->name('active-guards.export');
            Route::post('/check-in', [\App\Http\Controllers\GuardCheckinController::class, 'store'])->name('check-in.store');
            Route::get('/history', [\App\Http\Controllers\GuardCheckinController::class, 'index'])->name('history');
            Route::get('/history/export', [\App\Http\Controllers\GuardCheckinController::class, 'exportHistoryPdf'])->name('history.export');

            // NEW CHANGE: Added route for manual verification
            Route::patch('/history/{checkin}/verify', [\App\Http\Controllers\GuardCheckinController::class, 'verify'])->name('history.verify');
        });

        // Location Request (Customer Care sends to employees)
        Route::post('/location-request/send', [\App\Http\Controllers\LocationRequestController::class, 'sendRequest'])->name('location-request.send');

        // General Operational Routes (Available to all authenticated users)
        // Quotations - USER DASHBOARD ONLY (not for admins)
        // Custom routes first
        Route::get('quotations/{quotation}/pdf', [\App\Http\Controllers\QuotationController::class, 'downloadPdf'])->name('quotations.pdf');
        Route::post('quotations/{quotation}/email', [\App\Http\Controllers\QuotationController::class, 'sendEmail'])->name('quotations.email');
        Route::post('quotations/{quotation}/resend-email', [\App\Http\Controllers\QuotationController::class, 'resendEmail'])->name('quotations.resend');
        Route::resource('quotations', \App\Http\Controllers\QuotationController::class);

        // Lead Management (Using Base Controller for User Context)
        Route::post('leads/{lead}/remarks', [\App\Http\Controllers\LeadController::class, 'storeRemark'])->name('leads.remarks.store');
        Route::resource('leads', \App\Http\Controllers\LeadController::class);



        // Schedule Templates (New "Template Mode" Feature)
        Route::get('sites/{site}/schedule-templates', [\App\Http\Controllers\ScheduleTemplateController::class, 'index'])->name('schedule-templates.index');
        Route::post('schedule-templates', [\App\Http\Controllers\ScheduleTemplateController::class, 'store'])->name('schedule-templates.store');
        Route::delete('schedule-templates/{template}', [\App\Http\Controllers\ScheduleTemplateController::class, 'destroy'])->name('schedule-templates.destroy');
        Route::post('schedule-templates/{template}/shifts', [\App\Http\Controllers\ScheduleTemplateController::class, 'storeShift'])->name('schedule-templates.shifts.store');
        Route::delete('schedule-templates/shifts/{shift}', [\App\Http\Controllers\ScheduleTemplateController::class, 'destroyShift'])->name('schedule-templates.shifts.destroy');
        Route::post('schedule-templates/{template}/roll', [\App\Http\Controllers\ScheduleTemplateController::class, 'roll'])->name('schedule-templates.roll');

        // Operational Reports (User)
        // Operational Reports (User)
        Route::get('reports/operational', [\App\Http\Controllers\OperationalReportController::class, 'index'])->name('reports.operational.index');
        Route::get('reports/operational/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportPdf'])->name('reports.operational.export.pdf');
        Route::get('reports/operational/{id}/pdf', [\App\Http\Controllers\OperationalReportController::class, 'generateSinglePdf'])->name('reports.operational.single.pdf');
        Route::get('reports/operational/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportExcel'])->name('reports.operational.export.excel');
        Route::post('reports/operational/email-scans/{id}', [\App\Http\Controllers\OperationalReportController::class, 'emailScansHistoryToClient'])->name('reports.operational.email-scans');
        Route::post('reports/operational/email-shift/{schedule}/{employee}', [\App\Http\Controllers\OperationalReportController::class, 'emailShiftReport'])->name('reports.operational.email-shift-user');
        Route::get('reports/operational/payouts', [\App\Http\Controllers\OperationalReportController::class, 'payouts'])->name('reports.operational.payouts');
        Route::get('reports/operational/payouts/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportPayoutsPdf'])->name('reports.operational.payouts.export.pdf');
        Route::get('reports/operational/payouts/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportPayoutsExcel'])->name('reports.operational.payouts.export.excel');
        Route::get('reports/operational/incidents', [\App\Http\Controllers\OperationalReportController::class, 'incidents'])->name('reports.operational.incidents');
        Route::get('reports/operational/incident-analysis', [\App\Http\Controllers\OperationalReportController::class, 'incidentAnalysis'])->name('reports.operational.incident_analysis');
        Route::get('reports/operational/incidents/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportIncidentsPdf'])->name('reports.operational.incidents.export.pdf');
        Route::get('reports/operational/incidents/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportIncidentsExcel'])->name('reports.operational.incidents.export.excel');
        Route::get('reports/operational/incidents/{id}/download', [\App\Http\Controllers\OperationalReportController::class, 'downloadIncidentImages'])->name('reports.operational.incidents.download');
        Route::get('reports/operational/incident-counts', [\App\Http\Controllers\OperationalReportController::class, 'incidentCounts'])->name('reports.operational.incident_counts');
        Route::get('reports/operational/incident-counts/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportIncidentCountsPdf'])->name('reports.operational.incident_counts.export.pdf');
        Route::get('reports/operational/incident-counts/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportIncidentCountsExcel'])->name('reports.operational.incident_counts.export.excel');
        Route::get('reports/operational/guard-payouts', [\App\Http\Controllers\OperationalReportController::class, 'guardPayouts'])->name('reports.operational.guard_payouts');
        Route::match(['get', 'post'], 'reports/operational/consolidated', [\App\Http\Controllers\OperationalReportController::class, 'consolidatedReport'])->name('reports.operational.consolidated');
        Route::get('reports/operational/consolidated/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportConsolidatedPdf'])->name('reports.operational.consolidated.export.pdf');
        Route::get('reports/operational/consolidated/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportConsolidatedExcel'])->name('reports.operational.consolidated.export.excel');
        Route::get('reports/operational/guard-payouts/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportGuardPayoutsPdf'])->name('reports.operational.guard_payouts.export.pdf');
        Route::get('reports/operational/guard-payouts/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportGuardPayoutsExcel'])->name('reports.operational.guard_payouts.export.excel');
        Route::get('reports/operational/site-payouts', [\App\Http\Controllers\OperationalReportController::class, 'sitePayouts'])->name('reports.operational.site_payouts');
        Route::get('reports/operational/site-payouts/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportSitePayoutsPdf'])->name('reports.operational.site_payouts.export.pdf');
        Route::get('reports/operational/site-payouts/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportSitePayoutsExcel'])->name('reports.operational.site_payouts.export.excel');

        // Uniforms Inventory (User)
        Route::get('uniforms', [\App\Http\Controllers\Employee\EmployeeUniformController::class, 'index'])->name('user.uniforms.index');
        Route::get('uniforms/export/pdf', [\App\Http\Controllers\Employee\EmployeeUniformController::class, 'exportStockPdf'])->name('user.uniforms.export.pdf');
        Route::get('uniforms/issued/export/pdf', [\App\Http\Controllers\Employee\EmployeeUniformController::class, 'exportIssuedPdf'])->name('user.uniforms.issued.export.pdf');
        Route::get('uniforms/issued/export/excel', [\App\Http\Controllers\Employee\EmployeeUniformController::class, 'exportIssuedExcel'])->name('user.uniforms.issued.export.excel');

        Route::get('reports/operational/site-margin', [\App\Http\Controllers\OperationalReportController::class, 'siteMargin'])->name('reports.operational.site_margin');
        Route::get('reports/operational/site-margin/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportSiteMarginPdf'])->name('reports.operational.site_margin.export.pdf');
        Route::get('reports/operational/site-margin/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportSiteMarginExcel'])->name('reports.operational.site_margin.export.excel');

        Route::get('reports/operational/late-checkin', [\App\Http\Controllers\OperationalReportController::class, 'lateCheckin'])->name('reports.operational.late_checkin');
        Route::get('reports/operational/late-checkin/export/pdf', [\App\Http\Controllers\OperationalReportController::class, 'exportLateCheckinPdf'])->name('reports.operational.late_checkin.export.pdf');
        Route::get('reports/operational/late-checkin/export/excel', [\App\Http\Controllers\OperationalReportController::class, 'exportLateCheckinExcel'])->name('reports.operational.late_checkin.export.excel');
        Route::get('reports/operational/banned-employees', [\App\Http\Controllers\Admin\OperationalReportController::class, 'bannedEmployees'])->name('reports.operational.banned');
        Route::get('reports/operational/employee-passwords', [\App\Http\Controllers\Admin\OperationalReportController::class, 'employeePasswords'])->name('reports.operational.passwords');
        Route::post('reports/operational/log-password-view', [\App\Http\Controllers\Admin\OperationalReportController::class, 'logPasswordView'])->name('reports.operational.log_password_view');

        // Job Notes
        Route::get('job-notes', [\App\Http\Controllers\JobNoteController::class, 'index'])->name('job-notes.index');
        Route::get('job-notes/export/pdf', [\App\Http\Controllers\JobNoteController::class, 'exportPdf'])->name('job-notes.export.pdf');
        Route::get('job-notes/export/excel', [\App\Http\Controllers\JobNoteController::class, 'exportExcel'])->name('job-notes.export.excel');

        // Reports - Patroller (User Access)
        Route::prefix('reports/patroller')->name('reports.patroller.')->group(function () {
            Route::get('/', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'index'])->name('index');
            Route::get('/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportInOutPdf'])->name('export.pdf');
            Route::get('/export/excel', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportInOutExcel'])->name('export.excel');
            Route::get('/tickets', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'tickets'])->name('tickets');
            Route::get('/tickets/export/excel', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportTicketsExcel'])->name('tickets.export.excel');
            Route::get('/tickets/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportTicketsPdf'])->name('tickets.export.pdf');
            Route::get('/tickets/export/zip', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportTicketsZip'])->name('tickets.export.zip');
            Route::get('/monthly-tickets', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'monthlyTickets'])->name('monthly_tickets');
            Route::get('/monthly-tickets/export', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportMonthlyTickets'])->name('monthly_tickets.export');
            Route::get('/payout', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'payout'])->name('payout');
            Route::get('/payout/export', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'payoutExport'])->name('payout.export');
            Route::get('/payout/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'payoutExportPdf'])->name('payout.export.pdf');
            Route::get('/route-payout', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routePayout'])->name('route_payout');
            Route::get('/route-payout/export', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routePayoutExport'])->name('route_payout.export');
            Route::get('/route-payout/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routePayoutExportPdf'])->name('route_payout.export.pdf');
            Route::get('/route-margin', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routeMargin'])->name('route_margin');
            Route::get('/route-margin/export', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routeMarginExport'])->name('route_margin.export');
            Route::get('/route-margin/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routeMarginExportPdf'])->name('route_margin.export.pdf');
        });

        // Attendance Chart (User Access)
        Route::get('attendance/chart', [\App\Http\Controllers\Admin\AttendanceController::class, 'chart'])->name('attendance.chart');

        // Developer Settings
        Route::get('developer-settings', [\App\Http\Controllers\DeveloperSettingsController::class, 'index'])->name('developer.settings');
        Route::post('developer-settings/run-command', [\App\Http\Controllers\DeveloperSettingsController::class, 'runCommand'])->name('developer.run-command');
    });
});

use App\Http\Controllers\Admin\AdminSettingController;
use App\Http\Controllers\Admin\CountryController;
use App\Http\Controllers\Admin\JobRoleController;
use App\Http\Controllers\Admin\StateController;
use App\Http\Controllers\Admin\WageTypeController;

Route::prefix('admin')->name('admin.')->group(function () {
    Route::get('/login', [AdminAuthController::class, 'showLogin'])->name('login');
    Route::post('/login', [AdminAuthController::class, 'login']);

    Route::middleware(['auth', \App\Http\Middleware\EnsureAdminSession::class, 'admin'])->group(function () {
        Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
        Route::post('/logout', [AdminAuthController::class, 'logout'])->name('logout');

        // Masters - Time Zones
        Route::middleware(['can:view time zones'])->group(function () {
            Route::resource('time-zones', TimeZoneController::class);
            Route::patch('time-zones/{time_zone}/toggle-status', [TimeZoneController::class, 'toggleStatus'])->name('time-zones.toggle-status');
        });

        // Companies
        Route::middleware(['can:view companies'])->group(function () {
            Route::get('companies/export/pdf', [CompanyController::class, 'exportPDF'])->name('companies.export-pdf');
            Route::resource('companies', CompanyController::class);
            Route::patch('companies/{company}/toggle-status', [CompanyController::class, 'toggleStatus'])->name('companies.toggle-status');
        });

        // Masters - Departments
        Route::middleware(['can:view departments'])->group(function () {
            Route::get('departments/export/pdf', [DepartmentController::class, 'exportPDF'])->name('departments.export-pdf');
            Route::resource('departments', DepartmentController::class);
            Route::patch('departments/{department}/toggle-status', [DepartmentController::class, 'toggleStatus'])->name('departments.toggle-status');
        });

        // Masters - Job Roles
        Route::middleware(['can:view job roles'])->group(function () {
            Route::get('job-roles/export/pdf', [JobRoleController::class, 'exportPDF'])->name('job-roles.export-pdf');

            Route::resource('job-roles', JobRoleController::class);
            Route::patch('job-roles/{job_role}/toggle-status', [JobRoleController::class, 'toggleStatus'])->name('job-roles.toggle-status');
        });

        // Masters - Positions
        Route::middleware(['can:view positions'])->group(function () {
            Route::get('positions/export/pdf', [\App\Http\Controllers\Admin\PositionController::class, 'exportPDF'])->name('positions.export-pdf');
            Route::resource('positions', \App\Http\Controllers\Admin\PositionController::class);
            Route::patch('positions/{position}/toggle-status', [\App\Http\Controllers\Admin\PositionController::class, 'toggleStatus'])->name('positions.toggle-status');
        });

        // Masters - Wage Types
        Route::middleware(['can:view wage types'])->group(function () {
            Route::resource('wage-types', WageTypeController::class);
            Route::patch('wage-types/{wage_type}/toggle-status', [WageTypeController::class, 'toggleStatus'])->name('wage-types.toggle-status');
        });

        // Masters - Stat Holidays
        Route::middleware(['can:view stat holidays'])->group(function () {
            Route::resource('stat-holidays', \App\Http\Controllers\Admin\StatHolidayController::class);
            Route::patch('stat-holidays/{stat_holiday}/toggle-status', [\App\Http\Controllers\Admin\StatHolidayController::class, 'toggleStatus'])->name('stat-holidays.toggle-status');
        });

        // Users
        Route::middleware(['can:view users'])->group(function () {
            Route::get('users/export/pdf', [UserController::class, 'exportPDF'])->name('users.export-pdf');
            Route::resource('users', UserController::class)->except(['show']);
            Route::post('users/{user}/sync-companies', [UserController::class, 'syncCompanies'])->name('users.sync-companies');
            Route::patch('users/{user}/toggle-status', [UserController::class, 'toggleStatus'])->name('users.toggle-status');
            Route::patch('users/{user}/password', [UserController::class, 'changePassword'])->name('users.password');
            Route::get('users/{user}/analytics', [UserController::class, 'analytics'])->name('users.analytics');
        });

        // Masters - Countries
        Route::middleware(['can:view countries'])->group(function () {
            Route::get('countries/export/pdf', [CountryController::class, 'exportPDF'])->name('countries.export-pdf');
            Route::resource('countries', CountryController::class);
            Route::patch('countries/{country}/toggle-status', [CountryController::class, 'toggleStatus'])->name('countries.toggle-status');
        });

        // Masters - States
        Route::middleware(['can:view states'])->group(function () {
            Route::get('states/export/pdf', [StateController::class, 'exportPDF'])->name('states.export-pdf');
            Route::resource('states', StateController::class);
            Route::patch('states/{state}/toggle-status', [StateController::class, 'toggleStatus'])->name('states.toggle-status');
        });

        // Masters - Zones
        Route::middleware(['can:view zones'])->group(function () {
            Route::resource('zones', \App\Http\Controllers\Admin\ZoneController::class);
            Route::patch('zones/{zone}/toggle-status', [\App\Http\Controllers\Admin\ZoneController::class, 'toggleStatus'])->name('zones.toggle-status');
            Route::get('zones/get-states/{country}', [\App\Http\Controllers\Admin\ZoneController::class, 'getStates'])->name('zones.get-states');
        });

        // Masters - Incident Categories
        Route::get('incident-categories', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'index'])->name('incident-categories.index');

        // Parent
        Route::post('incident-categories/parents', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'storeParent'])->name('incident-categories.parents.store');
        Route::put('incident-categories/parents/{parent}', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'updateParent'])->name('incident-categories.parents.update');
        Route::delete('incident-categories/parents/{parent}', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'destroyParent'])->name('incident-categories.parents.destroy');
        Route::patch('incident-categories/parents/{parent}/toggle-status', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'toggleStatusParent'])->name('incident-categories.parents.toggle-status');
        Route::post('incident-categories/parents/import', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'importParents'])->name('incident-categories.parents.import');

        // Child
        Route::post('incident-categories/children', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'storeCategory'])->name('incident-categories.children.store');
        Route::put('incident-categories/children/{category}', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'updateCategory'])->name('incident-categories.children.update');
        Route::delete('incident-categories/children/{category}', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'destroyCategory'])->name('incident-categories.children.destroy');
        Route::patch('incident-categories/children/{category}/toggle-status', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'toggleStatusCategory'])->name('incident-categories.children.toggle-status');
        Route::post('incident-categories/children/import', [\App\Http\Controllers\Admin\IncidentCategoryController::class, 'importCategories'])->name('incident-categories.children.import');

        // Settings - CSS
        Route::middleware(['can:manage css settings'])->group(function () {
            Route::get('settings/css', [AdminSettingController::class, 'cssSettings'])->name('settings.css');
            Route::post('settings/css', [AdminSettingController::class, 'updateCssSettings'])->name('settings.css.update');
        });

        // Settings - Password (Open to all admins/users to change their own password)
        Route::get('settings/password', [AdminSettingController::class, 'changePassword'])->name('settings.password');
        Route::post('settings/password', [AdminSettingController::class, 'updatePassword'])->name('settings.password.update');

        // Settings - Integrations
        Route::middleware(['can:manage integrations'])->group(function () {
            Route::get('settings/integrations', [AdminSettingController::class, 'integrations'])->name('settings.integrations');
            Route::post('settings/integrations', [AdminSettingController::class, 'updateIntegrations'])->name('settings.integrations.update');
        });

        // Settings - Operational
        Route::middleware(['can:manage operational rules'])->group(function () {
            Route::get('settings/operational', [AdminSettingController::class, 'operational'])->name('settings.operational');
            Route::post('settings/operational', [AdminSettingController::class, 'updateOperational'])->name('settings.operational.update');
        });

        // Security Settings
        Route::middleware(['can:manage security settings'])->group(function () {
            Route::get('settings/security', [AdminSettingController::class, 'security'])->name('settings.security');
            Route::post('settings/security', [AdminSettingController::class, 'updateSecurity'])->name('settings.security.update');
        });

        // Appearance Settings
        Route::middleware(['can:manage settings'])->group(function () {
            Route::get('settings/appearance', [AdminSettingController::class, 'appearance'])->name('settings.appearance');
            Route::post('settings/appearance', [AdminSettingController::class, 'updateAppearance'])->name('settings.appearance.update');
        });

        // System Maintenance Settings
        Route::middleware(['can:manage settings'])->group(function () {
            Route::get('settings/system', [AdminSettingController::class, 'system'])->name('settings.system');
            Route::post('settings/system/migrate', [AdminSettingController::class, 'runMigration'])->name('settings.system.migrate');
            Route::post('settings/system/clear-cache', [AdminSettingController::class, 'clearCache'])->name('settings.system.clear-cache');
        });

        Route::patch('wage-types/{wageType}/toggle-status', [\App\Http\Controllers\Admin\WageTypeController::class, 'toggleStatus'])->name('wage-types.toggle-status');
        Route::resource('wage-types', \App\Http\Controllers\Admin\WageTypeController::class);

        Route::patch('request-types/{requestType}/toggle-status', [\App\Http\Controllers\Admin\RequestTypeController::class, 'toggleStatus'])->name('request-types.toggle-status');
        Route::resource('request-types', \App\Http\Controllers\Admin\RequestTypeController::class);

        Route::patch('uniforms/{uniform}/toggle-status', [\App\Http\Controllers\Admin\UniformController::class, 'toggleStatus'])->name('uniforms.toggle-status');
        Route::resource('uniforms', \App\Http\Controllers\Admin\UniformController::class);

        Route::patch('stat-holidays/{statHoliday}/toggle-status', [\App\Http\Controllers\Admin\StatHolidayController::class, 'toggleStatus'])->name('stat-holidays.toggle-status');

        // System Logs
        Route::middleware(['can:view system logs'])->group(function () {
            Route::delete('logs/clear', [\App\Http\Controllers\Admin\SystemLogController::class, 'clear'])->name('logs.clear');
            Route::resource('logs', \App\Http\Controllers\Admin\SystemLogController::class)->only(['index', 'show', 'destroy']);
        });

        // Activity Logs
        Route::middleware(['can:view activity logs'])->group(function () {
            Route::delete('activity-logs/clear', [\App\Http\Controllers\Admin\ActivityLogController::class, 'clear'])->name('activity-logs.clear');
            Route::resource('activity-logs', \App\Http\Controllers\Admin\ActivityLogController::class)->only(['index', 'destroy']);
        });

        // Communication - Email Signatures
        Route::middleware(['can:view email signatures'])->group(function () {
            Route::resource('email-signatures', \App\Http\Controllers\Admin\EmailSignatureController::class);
        });

        // Communication - Email Templates
        // Communication - Email Templates
        Route::resource('email-templates', \App\Http\Controllers\Admin\EmailTemplateController::class);



        // Access Control - Roles
        Route::middleware(['can:manage roles'])->group(function () {
            Route::resource('roles', \App\Http\Controllers\Admin\RoleController::class);
        });

        // Access Control - Permissions
        Route::middleware(['can:manage permissions'])->group(function () {
            Route::resource('permissions', \App\Http\Controllers\Admin\PermissionController::class);
        });

        // Employees
        Route::middleware(['can:view employees'])->group(function () {
            Route::get('employees/export/pdf', [\App\Http\Controllers\Admin\EmployeeController::class, 'exportPDF'])->name('employees.export-pdf');
            Route::resource('employees', \App\Http\Controllers\Admin\EmployeeController::class)->only(['index', 'show', 'destroy']);
            Route::patch('employees/{employee}/toggle-status', [\App\Http\Controllers\Admin\EmployeeController::class, 'toggleStatus'])->name('employees.toggle-status');
            Route::get('employees/{employee}/analytics', [\App\Http\Controllers\EmployeeController::class, 'analytics'])->name('employees.analytics');
        });

        // Masters - Skills
        Route::middleware(['can:view skills'])->group(function () {
            Route::resource('skills', \App\Http\Controllers\Admin\SkillController::class);
            Route::patch('skills/{skill}/toggle-status', [\App\Http\Controllers\Admin\SkillController::class, 'toggleStatus'])->name('skills.toggle-status');
        });

        // Admin Notifications
        Route::post('notifications/send', [\App\Http\Controllers\Admin\NotificationController::class, 'store'])->name('notifications.store');
        Route::get('/notifications', [\App\Http\Controllers\UserNotificationController::class, 'index'])->name('notifications.index');
        Route::get('/notifications/{id}/read', [\App\Http\Controllers\UserNotificationController::class, 'read'])->name('notifications.read');
        Route::post('/notifications/{id}/mark-read', [\App\Http\Controllers\UserNotificationController::class, 'markAsRead'])->name('notifications.markAsRead');

        // ID Settings
        Route::middleware(['can:manage id settings'])->group(function () {
            Route::resource('id-settings', \App\Http\Controllers\Admin\IdSettingController::class);
        });

        // Sites
        Route::middleware(['can:view sites'])->group(function () {
            Route::get('sites/export/pdf', [\App\Http\Controllers\Admin\SiteController::class, 'exportPDF'])->name('sites.export-pdf');
            Route::resource('sites', \App\Http\Controllers\Admin\SiteController::class)->only(['index', 'destroy']);
            Route::patch('sites/{site}/toggle-status', [\App\Http\Controllers\Admin\SiteController::class, 'toggleStatus'])->name('sites.toggle-status');
        });

        // Routes
        Route::middleware(['can:view routes'])->group(function () {
            Route::get('routes/export/pdf', [\App\Http\Controllers\Admin\RouteController::class, 'exportPDF'])->name('routes.export-pdf');
            Route::resource('routes', \App\Http\Controllers\Admin\RouteController::class)->only(['index', 'destroy']);
            Route::patch('/routes/{route}/toggle-status', [\App\Http\Controllers\Admin\RouteController::class, 'toggleStatus'])->name('routes.toggle-status');
        });

        // Quotations (Available to users)
        // Route::resource('quotations', \App\Http\Controllers\QuotationController::class); // Moving out

        // Operational Reports (Admin)
        Route::get('reports/operational', [\App\Http\Controllers\Admin\OperationalReportController::class, 'index'])->name('reports.operational.index');
        Route::get('reports/operational/export/pdf', [\App\Http\Controllers\Admin\OperationalReportController::class, 'exportPdf'])->name('reports.operational.export.pdf');
        Route::get('reports/operational/export/excel', [\App\Http\Controllers\Admin\OperationalReportController::class, 'exportExcel'])->name('reports.operational.export.excel');
        Route::post('reports/operational/email-shift/{schedule}/{employee}', [\App\Http\Controllers\Admin\OperationalReportController::class, 'emailShiftReport'])->name('reports.operational.email-shift');

        // Reports - Patroller
        Route::middleware(['can:view operational reports'])->prefix('reports/patroller')->name('reports.patroller.')->group(function () {
            Route::get('/', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'index'])->name('index');
            Route::get('/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportInOutPdf'])->name('export.pdf');
            Route::get('/export/excel', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportInOutExcel'])->name('export.excel');
            Route::get('/tickets', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'tickets'])->name('tickets');
            Route::get('/tickets/export/excel', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportTicketsExcel'])->name('tickets.export.excel');
            Route::get('/tickets/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportTicketsPdf'])->name('tickets.export.pdf');
            Route::get('/tickets/export/zip', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportTicketsZip'])->name('tickets.export.zip');
            Route::get('/monthly-tickets', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'monthlyTickets'])->name('monthly_tickets');
            Route::get('/monthly-tickets/export', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'exportMonthlyTickets'])->name('monthly_tickets.export');
            Route::get('/payout', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'payout'])->name('payout');
            Route::get('/payout/export', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'payoutExport'])->name('payout.export');
            Route::get('/payout/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'payoutExportPdf'])->name('payout.export.pdf');
            Route::get('/route-payout', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routePayout'])->name('route_payout');
            Route::get('/route-payout/export', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routePayoutExport'])->name('route_payout.export');
            Route::get('/route-payout/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routePayoutExportPdf'])->name('route_payout.export.pdf');
            Route::get('/route-margin', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routeMargin'])->name('route_margin');
            Route::get('/route-margin/export', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routeMarginExport'])->name('route_margin.export');
            Route::get('/route-margin/export/pdf', [\App\Http\Controllers\Admin\PatrollerReportController::class, 'routeMarginExportPdf'])->name('route_margin.export.pdf');
        });



        // Attendance Chart
        Route::get('attendance/chart', [\App\Http\Controllers\Admin\AttendanceController::class, 'chart'])->name('attendance.chart');

        // Lead Management





    });

    // Operational Routes (Admin Prefix, Auth Required, NO Extra Admin Login)
    // Removed Leads and Quotations from here to move them to global scope as per user request.
    // Route::middleware(['auth'])->group(function () {
    //      // Uniforms Inventory (User View) - Moved to top
    // });

});

// Employee Portal Routes
Route::prefix('employee')->name('employee.')->group(function () {
    Route::get('/login', [\App\Http\Controllers\Employee\EmployeeAuthController::class, 'showLogin'])->name('login')->middleware('guest:employee');
    Route::post('/login', [\App\Http\Controllers\Employee\EmployeeAuthController::class, 'login'])->name('login.post')->middleware('guest:employee');

    // Password Reset Routes
    Route::get('/forgot-password', [\App\Http\Controllers\Employee\EmployeePasswordResetController::class, 'showLinkRequestForm'])->name('password.request');
    Route::post('/forgot-password', [\App\Http\Controllers\Employee\EmployeePasswordResetController::class, 'sendResetLinkEmail'])->name('password.email');
    Route::get('/reset-password/{token}', [\App\Http\Controllers\Employee\EmployeePasswordResetController::class, 'showResetForm'])->name('password.reset');
    Route::post('/reset-password', [\App\Http\Controllers\Employee\EmployeePasswordResetController::class, 'reset'])->name('password.update');

    Route::middleware(['auth:employee', \App\Http\Middleware\EnsureEmployeeTimezone::class])->group(function () {
        Route::match(['get', 'post'], '/logout', [\App\Http\Controllers\Employee\EmployeeAuthController::class, 'logout'])->name('logout');

        // Dashboard
        Route::get('/dashboard', [\App\Http\Controllers\Employee\EmployeeDashboardController::class, 'index'])->name('dashboard');

        // My Profile
        Route::get('/profile', [\App\Http\Controllers\Employee\EmployeeProfileController::class, 'index'])->name('profile');
        Route::put('/profile', [\App\Http\Controllers\Employee\EmployeeProfileController::class, 'update'])->name('profile.update');
        Route::get('/profile/id-card', [\App\Http\Controllers\Employee\EmployeeProfileController::class, 'downloadIdCard'])->name('profile.id_card');

        // My Jobs
        Route::get('/jobs', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'index'])->name('jobs.index');
        Route::post('/jobs/{schedule}/location', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'logLocation'])->name('jobs.location');
        Route::post('/jobs/{schedule}/start', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'startJob'])->name('jobs.start');
        Route::post('/jobs/{schedule}/checkin', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'checkin'])->name('jobs.checkin');
        Route::post('/jobs/{schedule}/scan', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'scanCheckpoint'])->name('jobs.scan');
        Route::post('/jobs/{schedule}/incident', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'reportIncident'])->name('jobs.incident');
        Route::post('/jobs/{schedule}/checkout-evidence', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'uploadCheckoutEvidence'])->name('jobs.checkout.evidence');
        Route::post('/jobs/{schedule}/end', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'endJob'])->name('jobs.end');
        Route::get('/jobs/{schedule}/report', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'downloadReport'])->name('jobs.report');
        Route::get('/jobs/{schedule}/report/view', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'viewReport'])->name('jobs.report.view');
        Route::delete('/jobs/{schedule}/evidence/{type}/{index}', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'deleteEvidence'])->name('jobs.evidence.delete');
        Route::get('/jobs/{schedule}/reliever', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'getRelieverInfo'])->name('jobs.reliever');
        Route::post('/jobs/{schedule}/note', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'storeNote'])->name('jobs.note');
        Route::get('/jobs/{schedule}/notes', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'getNotes'])->name('jobs.notes');

        // Notifications
        Route::get('/notifications', [\App\Http\Controllers\Employee\EmployeeNotificationController::class, 'index'])->name('notifications.index');
        Route::post('/notifications/reminder', [\App\Http\Controllers\Employee\EmployeeNotificationController::class, 'sendReminder'])->name('notifications.reminder');
        Route::post('/notifications/{id}/read', [\App\Http\Controllers\Employee\EmployeeNotificationController::class, 'markAsRead'])->name('notifications.markAsRead');
        Route::get('/notifications/{id}/read', [\App\Http\Controllers\Employee\EmployeeNotificationController::class, 'read'])->name('notifications.read');
        Route::get('/jobs/{schedule}/report/checkin', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'downloadCheckinReport'])->name('jobs.report.checkin');
        Route::get('/jobs/{schedule}/report/checkout', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'downloadCheckoutReport'])->name('jobs.report.checkout');
        Route::get('/jobs/{schedule}/report/payout', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'downloadPayoutReport'])->name('jobs.report.payout');
        Route::get('/jobs/{schedule}/report/zip', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'downloadZip'])->name('jobs.report.zip');
        Route::post('/jobs/{schedule}/cancel', [\App\Http\Controllers\Employee\EmployeeJobController::class, 'requestCancellation'])->name('jobs.cancel');

        // Settings
        Route::get('/settings', [\App\Http\Controllers\Employee\EmployeeSettingsController::class, 'index'])->name('settings');
        Route::put('/settings/password', [\App\Http\Controllers\Employee\EmployeeSettingsController::class, 'updatePassword'])->name('settings.password');

        // ChatBot
        Route::post('/chatbot', [\App\Http\Controllers\Employee\ChatBotController::class, 'chat'])->name('chatbot');

        // Requests
        Route::get('/requests', [\App\Http\Controllers\Employee\EmployeeRequestController::class, 'index'])->name('requests.index');
        Route::get('/requests/create', [\App\Http\Controllers\Employee\EmployeeRequestController::class, 'create'])->name('requests.create');
        Route::post('/requests', [\App\Http\Controllers\Employee\EmployeeRequestController::class, 'store'])->name('requests.store');
        Route::get('/requests/{request}', [\App\Http\Controllers\Employee\EmployeeRequestController::class, 'show'])->name('requests.show');
        Route::post('/requests/{request}/reply', [\App\Http\Controllers\Employee\EmployeeRequestController::class, 'reply'])->name('requests.reply');

        // Operational Reports
        Route::get('/reports', [\App\Http\Controllers\Employee\EmployeeReportController::class, 'index'])->name('reports.index');
        Route::get('/reports/export/pdf', [\App\Http\Controllers\Employee\EmployeeReportController::class, 'exportPdf'])->name('reports.export.pdf');
        Route::get('/reports/export/excel', [\App\Http\Controllers\Employee\EmployeeReportController::class, 'exportExcel'])->name('reports.export.excel');

        // Help & Support
        Route::get('/support', [\App\Http\Controllers\Employee\EmployeeDashboardController::class, 'support'])->name('support');
        Route::post('/support', [\App\Http\Controllers\Employee\EmployeeDashboardController::class, 'submitSupport'])->name('support.submit');

        // Analytics
        Route::get('/analytics', [\App\Http\Controllers\Employee\EmployeeAnalyticsController::class, 'index'])->name('analytics.index');

        // Uniforms Inventory (Read Only)
        Route::get('/uniforms', [\App\Http\Controllers\Employee\EmployeeUniformController::class, 'index'])->name('uniforms.index');

        // Placeholders for other links
        Route::get('/payouts', [\App\Http\Controllers\Employee\EmployeePayoutController::class, 'index'])->name('payouts');

        // Patroller Jobs
        Route::prefix('patroller-jobs')->name('patroller-jobs.')->group(function () {
            Route::get('/', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'index'])->name('index');
            Route::get('history', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'history'])->name('history');
            Route::get('{schedule}', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'show'])->name('show');
            Route::post('{schedule}/start', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'start'])->name('start');
            Route::post('{schedule}/sites/{site}/checkin', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'checkin'])->name('checkin');
            Route::post('{schedule}/sites/{site}/checkout', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'checkout'])->name('checkout');
            Route::post('{schedule}/sites/{site}/tickets', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'createTicket'])->name('create-ticket');
            Route::post('{schedule}/end', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'endJob'])->name('end');
            Route::get('{schedule}/report', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'report'])->name('report');
            Route::post('{schedule}/note', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'storeNote'])->name('note');
            Route::get('{schedule}/notes', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'getNotes'])->name('notes');
            Route::get('{schedule}/export-tickets', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'exportTickets'])->name('export-tickets');
            Route::get('{schedule}/export-pdf', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'exportPdf'])->name('export-pdf');
            Route::get('{schedule}/reliever', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'getRelieverInfo'])->name('reliever');
            Route::post('{schedule}/cancel', [\App\Http\Controllers\Employee\PatrollerJobController::class, 'requestCancellation'])->name('cancel');
        });

        // Location Request Response
        Route::get('/location-request', [\App\Http\Controllers\Employee\LocationResponseController::class, 'show'])->name('location-request.show');
        Route::post('/location-request/submit', [\App\Http\Controllers\Employee\LocationResponseController::class, 'store'])->name('location-request.submit');
    });
});
