@extends('layouts.user')

@section('title', 'Attendances & Schedules')

@section('content')
    <div class="p-6" x-data="bulkScheduleEdit()">
        <div class="flex justify-between items-center mb-6">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">Guard Schedules</h1>
                <p class="text-gray-500">Manage and track guard duties across all sites.</p>
            </div>

            <div class="flex items-center gap-4">
                <a href="{{ route('schedules.export', request()->query()) }}" target="_blank"
                    class="flex items-center gap-2 px-4 py-2 bg-slate-800 text-white text-sm font-bold rounded-lg hover:bg-slate-900 transition-all shadow-md">
                    <svg class="w-4 h-4 text-emerald-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z">
                        </path>
                    </svg>
                    Export PDF
                </a>
                <a href="{{ route('schedules.export.excel', request()->query()) }}" target="_blank"
                    class="flex items-center gap-2 px-4 py-2 bg-emerald-600 text-white text-sm font-bold rounded-lg hover:bg-emerald-700 transition-all shadow-md">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z">
                        </path>
                    </svg>
                    Export Excel
                </a>
                <a href="{{ route('schedules.create') }}"
                    class="px-6 py-2 bg-blue-600 text-white font-bold rounded-lg hover:bg-blue-700 transition-all shadow-md shadow-blue-500/20 flex items-center gap-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                    </svg>
                    Create New Schedule
                </a>
            </div>
        </div>

        <!-- Bulk Action Bar -->
        <div x-show="selectedSchedules.length > 0" x-cloak
            class="fixed bottom-6 left-1/2 -translate-x-1/2 z-50 bg-slate-900 text-white px-6 py-3 rounded-2xl shadow-2xl flex items-center gap-6 border border-slate-700"
            x-transition:enter="transition ease-out duration-300" x-transition:enter-start="opacity-0 translate-y-full"
            x-transition:enter-end="opacity-100 translate-y-0" x-transition:leave="transition ease-in duration-200"
            x-transition:leave-start="opacity-100 translate-y-0" x-transition:leave-end="opacity-0 translate-y-full">
            <div class="flex items-center gap-2">
                <span class="bg-blue-600 text-white text-xs font-bold px-2 py-0.5 rounded"
                    x-text="selectedSchedules.length"></span>
                <span class="text-sm font-medium text-slate-300">Selected</span>
            </div>
            <div class="h-4 w-px bg-slate-700"></div>
            <button @click="openBulkEditModal"
                class="text-sm font-bold text-blue-500 hover:text-blue-600 transition-colors flex items-center gap-2">
                <svg class="w-4 h-4 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z">
                    </path>
                </svg>
                Bulk Edit
            </button>
            <div class="h-4 w-px bg-slate-700"></div>
            <button @click="confirmBulkDelete"
                class="text-sm font-bold text-red-500 hover:text-red-600 transition-colors flex items-center gap-2">
                <svg class="w-4 h-4 text-red-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16">
                    </path>
                </svg>
                Delete
            </button>
            <button @click="selectedSchedules = []" class="text-slate-500 hover:text-white transition-colors">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>

        <!-- Bulk Edit Modal -->
        <div x-show="showBulkEditModal" class="fixed inset-0 z-50 flex items-center justify-center p-4" x-cloak>
            <div class="absolute inset-0 bg-slate-900/50 backdrop-blur-sm" @click="showBulkEditModal = false"></div>
            <div class="relative bg-white rounded-3xl shadow-xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
                <form action="{{ route('schedules.bulk-update') }}" method="POST">
                    @csrf
                    <div class="p-8 border-b border-gray-100 flex justify-between items-center bg-gray-50/50">
                        <h2 class="text-xl font-bold text-gray-900">Bulk Edit Schedules</h2>
                        <button type="button" @click="showBulkEditModal = false" class="text-gray-400 hover:text-gray-600">
                            <svg class="w-6 h-6 text-red-400 hover:text-red-600" fill="none" stroke="currentColor"
                                viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M6 18L18 6M6 6l12 12"></path>
                            </svg>
                        </button>
                    </div>

                    <div class="p-8 space-y-6">
                        <input type="hidden" name="schedule_ids" :value="selectedSchedules.join(',')">

                        <div class="p-4 bg-blue-50 border border-blue-100 rounded-xl mb-6">
                            <p class="text-sm text-blue-700 text-center">
                                You are editing <span class="font-bold" x-text="selectedSchedules.length"></span> schedules.
                                Only fill fields you want to update. Leave blank to keep existing values.
                            </p>
                        </div>

                        <!-- Site Selection -->
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Change Site</label>
                            <select name="site_id"
                                class="w-full px-4 py-2 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 outline-none text-sm font-bold text-gray-700 bg-gray-50">
                                <option value="">No Change</option>
                                @foreach($sites as $site)
                                    <option value="{{ $site->id }}">{{ $site->name }}</option>
                                @endforeach
                            </select>
                        </div>

                        <div class="grid grid-cols-2 gap-6">
                            <!-- Date & Time -->
                            <div>
                                <label class="block text-xs font-bold text-gray-500 uppercase mb-2">New Start
                                    Date/Time</label>
                                <input type="datetime-local" name="from_datetime"
                                    class="w-full px-4 py-2 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 outline-none text-sm font-bold text-gray-700">
                            </div>
                            <div>
                                <label class="block text-xs font-bold text-gray-500 uppercase mb-2">New End
                                    Date/Time</label>
                                <input type="datetime-local" name="to_datetime"
                                    class="w-full px-4 py-2 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 outline-none text-sm font-bold text-gray-700">
                            </div>
                        </div>

                        <!-- Guards Update Logic -->
                        <div class="border-t border-gray-100 pt-6">
                            <h3 class="text-sm font-black text-gray-900 uppercase tracking-wide mb-4">Guard Assignment</h3>

                            <div class="mb-4">
                                <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Action</label>
                                <select name="guard_action"
                                    class="w-full px-4 py-2 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 outline-none text-sm font-bold text-gray-700 bg-gray-50"
                                    x-model="guardAction">
                                    <option value="">No Change</option>
                                    <option value="replace">Replace All Guards (Assign New)</option>
                                    <option value="add">Add Guards (Keep Existing)</option>
                                </select>
                            </div>

                            <div x-show="guardAction === 'replace' || guardAction === 'add'" x-cloak
                                class="space-y-4 p-4 bg-gray-50 rounded-xl border border-gray-200">
                                <div>
                                    <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Select
                                        Employees</label>
                                    <select name="employee_ids[]" multiple
                                        class="w-full h-32 px-4 py-2 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 outline-none text-sm font-bold text-gray-700">
                                        @foreach($employees as $emp)
                                            <option value="{{ $emp->id }}">{{ $emp->first_name }} {{ $emp->last_name }}</option>
                                        @endforeach
                                    </select>
                                    <p class="text-[10px] text-gray-400 mt-1">Hold Ctrl/Cmd to select multiple.</p>
                                </div>

                                <div>
                                    <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Wage Rate Override
                                        ($/hr)</label>
                                    <input type="number" step="0.01" name="wage_rate" placeholder="Default Employee Rate"
                                        class="w-full px-4 py-2 border border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 outline-none text-sm font-bold text-gray-700">
                                    <p class="text-[10px] text-gray-400 mt-1">Leave blank to use each employee's default
                                        base rate.</p>
                                </div>
                            </div>
                        </div>

                    </div>

                    <div class="p-8 border-t border-gray-100 flex justify-end gap-4 bg-gray-50/50">
                        <button type="button" @click="showBulkEditModal = false"
                            class="px-6 py-2.5 text-gray-600 font-bold hover:bg-gray-100 rounded-xl transition-colors text-sm">Cancel</button>
                        <button type="submit"
                            class="px-6 py-2.5 bg-blue-600 text-white font-bold rounded-xl shadow-lg shadow-blue-500/30 hover:bg-blue-700 transition-all text-sm uppercase tracking-wider">Update
                            Schedules</button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Labs Container -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-100 mb-6">
            <div class="flex border-b border-gray-100 overflow-x-auto whitespace-nowrap scrollbar-hide">
                @php
                    $tabs = [
                        'all' => 'All',
                        'active' => 'Active',
                        'upcoming' => 'Upcoming',
                        'completed' => 'Completed',
                        'missed' => 'Missed',
                        'cancelled' => 'Cancelled'
                    ];
                @endphp
                @foreach($tabs as $key => $label)
                    <a href="{{ route('schedules.index', ['type' => $key]) }}"
                        class="px-6 py-4 text-sm font-bold border-b-2 transition-all {{ $type === $key ? 'border-blue-600 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:bg-gray-50' }}">
                        {{ $label }}
                        <span
                            class="ml-2 px-2 py-0.5 rounded-full text-xs {{ $type === $key ? 'bg-blue-100 text-blue-800' : 'bg-gray-100 text-gray-600' }}">
                            {{ $counts[$key] ?? 0 }}
                        </span>
                    </a>
                @endforeach
            </div>
        </div>

        <!-- Filter & Export Toolbar -->
        <div
            class="flex flex-col md:flex-row justify-between items-center gap-4 mb-6 bg-white p-4 rounded-xl border border-gray-100 shadow-sm">
            <form method="GET" action="{{ route('schedules.index') }}"
                class="flex flex-wrap items-center gap-4 w-full md:w-auto">
                <input type="hidden" name="type" value="{{ $type }}">

                <!-- Site Filter -->
                <div class="relative">
                    <select name="site_id"
                        class="pl-4 pr-10 py-2 bg-gray-50 border border-gray-200 rounded-lg text-sm font-bold text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 appearance-none min-w-[200px]"
                        onchange="this.form.submit()">
                        <option value="">All Sites</option>
                        @foreach($sites as $site)
                            <option value="{{ $site->id }}" {{ request('site_id') == $site->id ? 'selected' : '' }}>
                                {{ $site->name }}
                            </option>
                        @endforeach
                    </select>
                    <div class="absolute inset-y-0 right-0 flex items-center px-2 pointer-events-none">
                        <svg class="w-4 h-4 text-indigo-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                        </svg>
                    </div>
                </div>

                <!-- Employee Filter -->
                <div class="relative">
                    <select name="employee_id"
                        class="pl-4 pr-10 py-2 bg-gray-50 border border-gray-200 rounded-lg text-sm font-bold text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 appearance-none min-w-[200px]"
                        onchange="this.form.submit()">
                        <option value="">All Employees</option>
                        @foreach($employees as $emp)
                            <option value="{{ $emp->id }}" {{ request('employee_id') == $emp->id ? 'selected' : '' }}>
                                {{ $emp->first_name }} {{ $emp->last_name }}
                            </option>
                        @endforeach
                    </select>
                    <div class="absolute inset-y-0 right-0 flex items-center px-2 pointer-events-none">
                        <svg class="w-4 h-4 text-indigo-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"></path>
                        </svg>
                    </div>
                </div>
                <!-- Date Range Filter -->
                <div class="flex items-center gap-2">
                    <div class="relative">
                        <input type="date" name="from_date" value="{{ request('from_date') }}"
                            class="pl-4 pr-4 py-2 bg-gray-50 border border-gray-200 rounded-lg text-sm font-bold text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 w-[140px]"
                            placeholder="From Date" onchange="this.form.submit()">
                    </div>
                    <span class="text-gray-400 font-bold">-</span>
                    <div class="relative">
                        <input type="date" name="to_date" value="{{ request('to_date') }}"
                            class="pl-4 pr-4 py-2 bg-gray-50 border border-gray-200 rounded-lg text-sm font-bold text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 w-[140px]"
                            placeholder="To Date" onchange="this.form.submit()">
                    </div>

                    <!-- Quick Date Filters -->
                    <div class="flex items-center gap-1 ml-2 bg-gray-50 rounded-lg p-1 border border-gray-200">
                        @php
                            $rangeFilters = ['today' => 'Today', 'this_week' => 'This Week', 'this_month' => 'This Month'];
                        @endphp
                        @foreach($rangeFilters as $key => $label)
                            <a href="{{ route('schedules.index', array_merge(request()->query(), ['type' => $key])) }}"
                                class="px-2 py-1 text-[10px] font-bold uppercase tracking-wider rounded transition-colors {{ $type === $key ? 'bg-white text-blue-600 shadow-sm' : 'text-gray-500 hover:text-gray-700 hover:bg-gray-100' }}">
                                {{ $label }}
                            </a>
                        @endforeach
                    </div>
                </div>

                @if($type === 'completed')
                    <div class="flex items-center gap-2 bg-gray-50 border border-gray-200 rounded-lg px-3 py-2">
                        <input type="checkbox" name="is_addon" id="is_addon" value="1" {{ request('is_addon') ? 'checked' : '' }} onchange="this.form.submit()"
                            class="w-4 h-4 text-blue-600 rounded border-gray-300 focus:ring-blue-500 cursor-pointer">
                        <label for="is_addon" class="text-xs font-bold text-gray-700 cursor-pointer select-none">
                            Manual Addon
                        </label>
                    </div>
                @endif

                <a href="{{ route('schedules.index', ['type' => $type]) }}"
                    class="text-sm font-bold text-red-400 hover:text-red-600" title="Clear Filters">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                        </path>
                    </svg>
                </a>

        </div>
        </form>



        <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
            <table class="w-full text-left border-collapse">
                <thead class="bg-gray-50 border-b border-gray-100">
                    <tr>
                        <th class="px-6 py-4 w-10">
                            @if($type === 'upcoming')
                                <input type="checkbox" @change="toggleAll" x-model="allSelected"
                                    class="w-4 h-4 text-blue-600 rounded border-gray-300 focus:ring-blue-500 cursor-pointer">
                            @endif
                        </th>
                        <th class="px-6 py-4 text-[11px] font-bold text-gray-400 uppercase tracking-wider">Duty #</th>
                        <th class="px-6 py-4 text-[11px] font-bold text-gray-400 uppercase tracking-wider">
                            <a href="{{ request()->fullUrlWithQuery(['sort' => 'site', 'direction' => request('sort') === 'site' && request('direction') === 'asc' ? 'desc' : 'asc']) }}"
                                class="flex items-center gap-1 hover:text-gray-600 transition-colors">
                                Site
                                @if(request('sort') === 'site')
                                    <svg class="w-3 h-3 text-gray-900 {{ request('direction') === 'asc' ? 'rotate-180' : '' }}"
                                        fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 9l-7 7-7-7"></path>
                                    </svg>
                                @else
                                    <svg class="w-3 h-3 text-gray-900" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M8 9l4-4 4 4m0 6l-4 4-4-4"></path>
                                    </svg>
                                @endif
                            </a>
                        </th>
                        <th class="px-6 py-4 text-[11px] font-bold text-gray-400 uppercase tracking-wider">
                            <a href="{{ request()->fullUrlWithQuery(['sort' => 'employee', 'direction' => request('sort') === 'employee' && request('direction') === 'asc' ? 'desc' : 'asc']) }}"
                                class="flex items-center gap-1 hover:text-gray-600 transition-colors">
                                Employee
                                @if(request('sort') === 'employee')
                                    <svg class="w-3 h-3 text-gray-900 {{ request('direction') === 'asc' ? 'rotate-180' : '' }}"
                                        fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 9l-7 7-7-7"></path>
                                    </svg>
                                @else
                                    <svg class="w-3 h-3 text-gray-900" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M8 9l4-4 4 4m0 6l-4 4-4-4"></path>
                                    </svg>
                                @endif
                            </a>
                        </th>
                        <th class="px-6 py-4 text-[11px] font-bold text-gray-400 uppercase tracking-wider">
                            <a href="{{ request()->fullUrlWithQuery(['sort' => 'data_time', 'direction' => request('sort') === 'data_time' && request('direction') === 'asc' ? 'desc' : 'asc']) }}"
                                class="flex items-center gap-1 hover:text-gray-600 transition-colors">
                                Date & Time
                                @if(request('sort') === 'data_time')
                                    <svg class="w-3 h-3 text-gray-900 {{ request('direction') === 'asc' ? 'rotate-180' : '' }}"
                                        fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 9l-7 7-7-7"></path>
                                    </svg>
                                @else
                                    <svg class="w-3 h-3 text-gray-900" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M8 9l4-4 4 4m0 6l-4 4-4-4"></path>
                                    </svg>
                                @endif
                            </a>
                        </th>
                        <th class="px-6 py-4 text-[11px] font-bold text-gray-400 uppercase tracking-wider">
                            <a href="{{ request()->fullUrlWithQuery(['sort' => 'status', 'direction' => request('sort') === 'status' && request('direction') === 'asc' ? 'desc' : 'asc']) }}"
                                class="flex items-center gap-1 hover:text-gray-600 transition-colors">
                                Status
                                @if(request('sort') === 'status')
                                    <svg class="w-3 h-3 text-gray-900 {{ request('direction') === 'asc' ? 'rotate-180' : '' }}"
                                        fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 9l-7 7-7-7"></path>
                                    </svg>
                                @else
                                    <svg class="w-3 h-3 text-gray-900" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M8 9l4-4 4 4m0 6l-4 4-4-4"></path>
                                    </svg>
                                @endif
                            </a>
                        </th>
                        <th class="px-6 py-4 text-[11px] font-bold text-gray-400 uppercase tracking-wider text-right">
                            Actions</th>
                    </tr>
                </thead>
                <tbody class="divide-y divide-gray-100">
                    @forelse($schedules as $schedule)
                        <tr class="hover:bg-gray-50 transition-colors group">
                            <td class="px-6 py-4">
                                @if($type === 'upcoming')
                                    <input type="checkbox" value="{{ $schedule->id }}" x-model="selectedSchedules"
                                        class="w-4 h-4 text-blue-600 rounded border-gray-300 focus:ring-blue-500 cursor-pointer">
                                @endif
                            </td>
                            <td class="px-6 py-4">
                                <div class="text-sm font-bold text-gray-900">{{ $schedule->duty_number }}</div>
                                @if($schedule->customer_reference)
                                    <div class="text-[10px] text-gray-400 font-bold uppercase mt-1">Ref:
                                        {{ $schedule->customer_reference }}</div>
                                @endif
                            </td>
                            <td class="px-6 py-4">
                                <div class="flex items-center gap-3">
                                    <div
                                        class="w-8 h-8 rounded-lg bg-blue-50 text-blue-600 flex items-center justify-center font-bold text-xs">
                                        {{ substr($schedule->site->name, 0, 1) }}
                                    </div>
                                    <div>
                                        <div class="text-sm font-bold text-gray-900">{{ $schedule->site->name }}</div>
                                        <div class="text-[10px] text-gray-400 font-medium">ID: {{ $schedule->site->site_id }}
                                        </div>
                                    </div>
                                </div>
                            </td>
                            <td class="px-6 py-4">
                                <div class="flex flex-col gap-2">
                                    @foreach($schedule->employees as $employee)
                                        <div class="flex items-center gap-2">
                                            @if($employee->profile_picture)
                                                <img src="{{ asset('storage/' . $employee->profile_picture) }}"
                                                    class="h-8 w-8 rounded-full border border-slate-100 object-cover shrink-0">
                                            @else
                                                <div
                                                    class="h-8 w-8 rounded-full bg-slate-100 text-slate-500 border border-slate-200 flex items-center justify-center font-bold text-[10px] uppercase shrink-0">
                                                    {{ substr($employee->first_name, 0, 1) }}{{ substr($employee->last_name, 0, 1) }}
                                                </div>
                                            @endif
                                            <div>
                                                <div class="text-xs font-bold text-slate-700 leading-tight">
                                                    {{ $employee->first_name }} {{ $employee->last_name }}
                                                </div>
                                                <div class="text-[10px] font-bold text-slate-400 uppercase tracking-widest">
                                                    {{ $employee->employee_id }}
                                                </div>
                                            </div>
                                        </div>
                                    @endforeach
                                    @if($schedule->employees->count() === 0)
                                        <span class="text-xs font-bold text-slate-400 italic">Unassigned</span>
                                    @endif
                                </div>
                            </td>
                            <td class="px-6 py-4">
                                <div class="text-sm text-gray-700 font-bold">{{ $schedule->from_datetime->format('M d, Y') }}
                                </div>
                                <div class="text-[11px] text-gray-500 font-medium italic">
                                    {{ $schedule->from_datetime->format('h:i A') }} -
                                    {{ $schedule->to_datetime->format('h:i A') }}
                                </div>
                            </td>
                            <td class="px-6 py-4">
                                @php
                                    $displayStatus = $schedule->status;

                                    // Dynamic Status Logic for Display
                                    if ($schedule->active && $schedule->status !== 'completed' && $schedule->status !== 'active' && $schedule->status !== 'cancelled') {
                                        if ($schedule->to_datetime->isPast()) {
                                            $displayStatus = 'missed';
                                        } elseif ($schedule->from_datetime->isToday()) {
                                            $displayStatus = 'today';
                                        } else {
                                            $displayStatus = 'upcoming';
                                        }
                                    }

                                    $statusColors = [
                                        'upcoming' => 'bg-blue-100 text-blue-700',
                                        'today' => 'bg-amber-100 text-amber-700',
                                        'completed' => 'bg-emerald-100 text-emerald-700',
                                        'missed' => 'bg-rose-100 text-rose-700',
                                        'archived' => 'bg-slate-100 text-slate-700',
                                        'active' => 'bg-green-100 text-green-700',
                                        'cancelled' => 'bg-red-100 text-red-700'
                                    ];

                                    $dotColors = [
                                        'upcoming' => 'bg-blue-500',
                                        'today' => 'bg-amber-500',
                                        'completed' => 'bg-emerald-500',
                                        'missed' => 'bg-rose-500',
                                        'archived' => 'bg-slate-500',
                                        'active' => 'bg-green-500',
                                        'cancelled' => 'bg-red-500'
                                    ];

                                    $isAddon = $schedule->employees->contains(fn($e) => $e->pivot->is_addon);
                                @endphp
                                <div class="flex flex-col items-start gap-1">
                                    <span
                                        class="inline-flex items-center gap-1.5 px-3 py-1 rounded-full text-[10px] font-bold uppercase tracking-wider {{ !$schedule->active ? 'bg-gray-100 text-gray-500' : ($statusColors[$displayStatus] ?? 'bg-gray-100 text-gray-600') }}">
                                        <span
                                            class="w-1.5 h-1.5 rounded-full {{ !$schedule->active ? 'bg-gray-400' : ($dotColors[$displayStatus] ?? 'bg-gray-400') }}"></span>
                                        {{ !$schedule->active ? 'Inactive' : $displayStatus }}
                                    </span>
                                    @if($isAddon)
                                        <span
                                            class="inline-flex items-center gap-1 px-2 py-0.5 rounded text-[10px] font-bold bg-purple-100 text-purple-700 border border-purple-200">
                                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M15.232 5.232l3.536 3.536m-2.036-5.036a2.5 2.5 0 113.536 3.536L6.5 21.036H3v-3.572L16.732 3.732z">
                                                </path>
                                            </svg>
                                            Manual/Addon
                                        </span>
                                    @endif
                                </div>
                            </td>
                            <td class="px-6 py-4 text-right">
                                <div class="flex justify-end gap-1">
                                    <a href="{{ route('schedules.show', $schedule) }}"
                                        class="p-2 text-blue-500 hover:text-blue-700 hover:bg-blue-50 rounded-xl transition-all group/tooltip relative">
                                        <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                                        </svg>
                                        <span
                                            class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">View
                                            Details</span>
                                    </a>
                                    @if($schedule->status !== 'completed' && $schedule->status !== 'active' && $type !== 'active')
                                        <form action="{{ route('schedules.toggle-status', $schedule) }}" method="POST"
                                            class="inline-block">
                                            @csrf
                                            @method('PATCH')
                                            <button type="submit"
                                                class="p-2 rounded-xl transition-all group/tooltip relative {{ $schedule->active ? 'text-red-500 hover:text-red-700 hover:bg-red-50' : 'text-emerald-500 hover:text-emerald-700 hover:bg-emerald-50' }}">
                                                @if($schedule->active)
                                                    <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                            d="M18.364 18.364A9 9 0 005.636 5.636m12.728 12.728A9 9 0 015.636 5.636m12.728 12.728L5.636 5.636" />
                                                    </svg>
                                                    <span
                                                        class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">Deactivate</span>
                                                @else
                                                    <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                            d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                                    </svg>
                                                    <span
                                                        class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">Activate</span>
                                                @endif
                                            </button>
                                        </form>
                                    @endif
                                    @if($schedule->status !== 'completed' && $schedule->status !== 'active' && $type !== 'active')
                                        <a href="{{ route('schedules.edit', $schedule) }}"
                                            class="p-2 text-amber-500 hover:text-amber-700 hover:bg-amber-50 rounded-xl transition-all group/tooltip relative">
                                            <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                                            </svg>
                                            <span
                                                class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">Edit
                                                Schedule</span>
                                        </a>
                                    @endif
                                    @if($schedule->status !== 'active' && $schedule->status !== 'completed' && !$schedule->employees->contains(fn($e) => $e->pivot->actual_start_at !== null))
                                        <form action="{{ route('schedules.destroy', $schedule) }}" method="POST"
                                            class="inline-block"
                                            onsubmit="return confirm('Are you sure you want to delete this schedule?');">
                                            @csrf
                                            @method('DELETE')
                                            <button type="submit"
                                                class="p-2 text-rose-500 hover:text-rose-700 hover:bg-red-50 rounded-xl transition-all group/tooltip relative">
                                                <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                                </svg>
                                                <span
                                                    class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">Delete</span>
                                            </button>
                                        </form>
                                    @endif

                                    @if($schedule->status === 'completed' || $schedule->active)
                                        <button type="button"
                                            @click="console.log('Clicked schedule {{ $schedule->id }}'); viewLocations({{ $schedule->id }})"
                                            class="p-2 text-cyan-500 hover:text-cyan-700 hover:bg-cyan-50 rounded-xl transition-all group/tooltip relative">
                                            <svg class="w-4.5 h-4.5 pointer-events-none" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z">
                                                </path>
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                                            </svg>
                                            <span
                                                class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">View
                                                Location</span>
                                        </button>
                                    @endif

                                    @if($schedule->status === 'completed' || $schedule->status === 'cancelled')
                                        <div class="flex gap-1 border-l border-gray-200 pl-1 mx-1">
                                            <a href="{{ route('schedules.report.view', $schedule) }}" target="_blank"
                                                class="p-2 text-indigo-500 hover:text-indigo-700 hover:bg-blue-50 rounded-xl transition-all group/tooltip relative">
                                                <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                                                </svg>
                                                <span
                                                    class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">View
                                                    Report</span>
                                            </a>
                                            <a href="{{ route('schedules.report.download', $schedule) }}"
                                                class="p-2 text-red-500 hover:text-red-700 hover:bg-red-50 rounded-xl transition-all group/tooltip relative">
                                                <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z">
                                                    </path>
                                                </svg>
                                                <span
                                                    class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">Download
                                                    Report</span>
                                            </a>
                                            <a href="{{ route('schedules.report.zip', $schedule) }}"
                                                class="p-2 text-purple-500 hover:text-purple-700 hover:bg-amber-50 rounded-xl transition-all group/tooltip relative">
                                                <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M5 8h14M5 8a2 2 0 110-4h14a2 2 0 110 4M5 8v10a2 2 0 002 2h10a2 2 0 002-2V8m-9 4h4">
                                                    </path>
                                                </svg>
                                                <span
                                                    class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">Download
                                                    Photos (ZIP)</span>
                                            </a>
                                            <form action="{{ route('schedules.report.email', $schedule) }}" method="POST"
                                                class="inline-block">
                                                @csrf
                                                <button type="submit"
                                                    class="p-2 {{ $schedule->site->contact_email ? 'text-emerald-500 hover:text-emerald-700 hover:bg-emerald-50' : 'text-gray-300 cursor-not-allowed' }} rounded-xl transition-all group/tooltip relative"
                                                    @if($schedule->site->contact_email) onclick="return confirm('Send this report to {{ $schedule->site->contact_email }}?');" @endif
                                                    {{ !$schedule->site->contact_email ? 'disabled' : '' }}>
                                                    <svg class="w-4.5 h-4.5" fill="none" stroke="currentColor"
                                                        viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round"
                                                            stroke-width="2"
                                                            d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                                                    </svg>
                                                    <span
                                                        class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 px-2 py-1 bg-slate-900 text-white text-[10px] font-bold rounded opacity-0 group-hover/tooltip:opacity-100 transition-opacity whitespace-nowrap pointer-events-none z-20">{{ $schedule->site->contact_email ? 'Email Report to Client' : 'No Client Email Set' }}</span>
                                                </button>
                                            </form>
                                        </div>
                                    @endif
                                </div>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="px-6 py-12 text-center text-gray-400 italic">
                                No schedules found for this category.
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <div class="mt-6">
            {{ $schedules->appends(request()->query())->links() }}
        </div>
        <!-- Map Modal -->
        <div x-show="showMapModal" class="fixed inset-0 z-50 flex items-center justify-center p-4" x-cloak
            style="display: none;">
            <div class="absolute inset-0 bg-slate-900/50 backdrop-blur-sm" @click="showMapModal = false"></div>
            <div class="relative bg-white rounded-3xl shadow-xl w-full max-w-4xl h-[600px] flex flex-col overflow-hidden">
                <div class="p-4 border-b border-gray-100 flex justify-between items-center bg-gray-50/50">
                    <h2 class="text-xl font-bold text-gray-900">Location History</h2>
                    <button type="button" @click="showMapModal = false" class="text-gray-400 hover:text-gray-600">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                            </path>
                        </svg>
                    </button>
                </div>
                <div class="flex-1 relative bg-slate-50">
                    <div id="map" class="absolute inset-0 z-10 w-full h-full"></div>
                </div>
            </div>
        </div>
    </div>
    </div>

    @push('scripts')
        <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" crossorigin="" />
        <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js" crossorigin=""></script>
    @endpush

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            // Auto-Refresh Logic
            var refreshEnabled = @json($refreshEnabled ?? 0);
            var refreshInterval = @json($refreshInterval ?? 5);

            // Check loosely for truthy values (1, '1', true, 'true', 'on')
            var isEnabled = refreshEnabled == 1 || refreshEnabled === 'true' || refreshEnabled === true || refreshEnabled === 'on';

            if (isEnabled) {
                var minutes = parseInt(refreshInterval);
                if (isNaN(minutes) || minutes < 1) {
                    minutes = 5; // Fallback default
                }

                var intervalMs = minutes * 60 * 1000;
                console.log('Schedules Auto-Refresh is ACTIVE. Scheduling reload in ' + minutes + ' minutes.');

                setTimeout(function () {
                    console.log('Reloading schedules page...');
                    window.location.reload(); // Reloads the current page with all query parameters (e.g. ?type=upcoming) preserved
                }, intervalMs);
            } else {
                console.log('Schedules Auto-Refresh is DISABLED.');
            }
        });
    </script>

    <script>
        function bulkScheduleEdit() {
            return {
                selectedSchedules: [],
                allSelected: false,
                showBulkEditModal: false,
                guardAction: '',
                showMapModal: false,
                map: null,

                toggleAll() {
                    if (this.allSelected) {
                        this.selectedSchedules = Array.from(document.querySelectorAll('input[type="checkbox"][value]'))
                            .map(el => el.value)
                            .filter(val => val !== 'on');
                    } else {
                        this.selectedSchedules = [];
                    }
                },

                openBulkEditModal() {
                    this.showBulkEditModal = true;
                },

                confirmBulkDelete() {
                    if (confirm('Are you sure you want to delete ' + this.selectedSchedules.length + ' schedules? This action cannot be undone.')) {
                        let form = document.createElement('form');
                        form.action = "{{ route('schedules.bulk-destroy') }}";
                        form.method = 'POST';
                        form.style.display = 'none';

                        let csrfToken = document.createElement('input');
                        csrfToken.type = 'hidden';
                        csrfToken.name = '_token';
                        csrfToken.value = "{{ csrf_token() }}";
                        form.appendChild(csrfToken);

                        let method = document.createElement('input');
                        method.type = 'hidden';
                        method.name = '_method';
                        method.value = 'DELETE';
                        form.appendChild(method);

                        let ids = document.createElement('input');
                        ids.type = 'hidden';
                        ids.name = 'schedule_ids';
                        ids.value = this.selectedSchedules.join(',');
                        form.appendChild(ids);

                        document.body.appendChild(form);
                        form.submit();
                    }
                },

                viewLocations(scheduleId) {
                    console.log('Opening map for schedule:', scheduleId);
                    this.showMapModal = true;
                    this.$nextTick(() => {
                        if (!this.map) {
                            if (typeof L === 'undefined') {
                                console.error('Leaflet L is undefined');
                                alert('Map library not loaded. Please refresh.');
                                return;
                            }
                            this.map = L.map('map').setView([0, 0], 2);
                            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                                attribution: '&copy; OpenStreetMap contributors'
                            }).addTo(this.map);
                        }

                        // Clear layers
                        this.map.eachLayer((layer) => {
                            if (layer instanceof L.Marker || layer instanceof L.Polyline) {
                                this.map.removeLayer(layer);
                            }
                        });

                        fetch(`/schedules/${scheduleId}/locations`)
                            .then(res => res.json())
                            .then(data => {
                                console.log('Locations loaded:', data.length);
                                if (data.length === 0) {
                                    alert('No location history recorded for this schedule.');
                                    return;
                                }

                                const latlngs = data.map(d => [d.lat, d.lng]);
                                const polyline = L.polyline(latlngs, { color: 'blue' }).addTo(this.map);

                                data.forEach(d => {
                                    L.marker([d.lat, d.lng]).addTo(this.map)
                                        .bindPopup(`<b>${d.name}</b><br>${d.time}`);
                                });

                                if (latlngs.length > 0) {
                                    this.map.fitBounds(polyline.getBounds());
                                }
                            })
                            .catch(err => {
                                console.error('Error fetching locations:', err);
                                alert('Failed to load location data.');
                            });

                        setTimeout(() => { this.map.invalidateSize(); }, 200);
                    });
                }
            }
        }
        window.bulkScheduleEdit = bulkScheduleEdit;
    </script>
@endsection