@extends('layouts.user')

@section('title', 'Create Guard Schedule')

@section('content')
    <div class="p-6" x-data="scheduleForm()">
        <div
            class="flex flex-col md:flex-row md:items-center justify-between gap-4 bg-slate-900 p-8 rounded-[2rem] text-white shadow-2xl mb-6">
            <div>
                <div class="flex items-center gap-2 text-[10px] font-bold uppercase tracking-widest text-blue-400 mb-1">
                    <a href="{{ route('schedules.index') }}" class="hover:text-white transition-colors">Duty Schedules</a>
                    <svg class="w-3 h-3 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                    </svg>
                    <span class="text-white">Create</span>
                </div>
                <h1 class="text-3xl font-black tracking-tight">Create Duty Schedule</h1>
                <p class="text-xs text-slate-400 font-medium mt-1">Assign guard(s) to a site and define duty timings.
                </p>
            </div>

            <a href="{{ route('schedules.index') }}"
                class="flex items-center gap-2 px-6 py-2.5 bg-white/10 hover:bg-white/20 text-white text-xs font-bold rounded-2xl backdrop-blur-md transition-all border border-white/10">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18">
                    </path>
                </svg>
                Back to List
            </a>
        </div>

        <!-- Meta Alerts Container -->
        <div class="space-y-4 mb-6">
            <!-- License Expiry Alert Section -->
            <template x-for="emp in selectedEmployeesData.filter(e => e.is_license_expired)" :key="'exp-' + emp.id">
                <div class="p-6 bg-amber-50 border-2 border-amber-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-amber-500 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-amber-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                </path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-amber-800 mb-1">Warning: Guard License Expired!</h3>
                            <p class="text-amber-600 text-sm mb-4 font-medium"
                                x-text="'The security license for ' + emp.name + ' expired on ' + emp.license_expiry + '.'">
                            </p>
                            <button type="button" @click="toggleEmployee(emp.id)"
                                class="px-4 py-2 bg-amber-600 text-white text-xs font-bold rounded-lg hover:bg-amber-700">Remove
                                This Guard</button>
                        </div>
                    </div>
                </div>
            </template>

            <!-- Banned Site Alert Section -->
            <template
                x-for="emp in selectedEmployeesData.filter(e => e.banned_sites && e.banned_sites.map(String).includes(String(selectedSiteId)))"
                :key="'ban-' + emp.id">
                <div class="p-6 bg-red-50 border-2 border-red-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-red-600 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-red-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M18.364 18.364A9 9 0 005.636 5.636m12.728 12.728A9 9 0 015.636 5.636m12.728 12.728L5.636 5.636">
                                </path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-red-800 mb-1">Warning: Guard Banned from Site!</h3>
                            <p class="text-red-600 text-sm mb-4 font-medium"
                                x-text="'The guard ' + emp.name + ' has been BANNED from this site. You should generally not assign them.'">
                            </p>
                            <button type="button" @click="toggleEmployee(emp.id)"
                                class="px-4 py-2 bg-red-600 text-white text-xs font-bold rounded-lg hover:bg-red-700">Remove
                                This Guard</button>
                        </div>
                    </div>
                </div>
            </template>

            <!-- Conflict Alert Section -->
            <template x-if="allConflicts.length > 0">
                <div class="p-6 bg-rose-50 border-2 border-rose-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-rose-500 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-rose-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                </path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-rose-800 mb-1">Schedule Conflict Detected!</h3>
                            <p class="text-rose-600 text-sm mb-4 font-medium">One or more selected guards are already
                                scheduled during this period.</p>
                            <div class="space-y-3 max-h-40 overflow-y-auto pr-2">
                                <template x-for="c in allConflicts" :key="c.duty_number + c.guard_name">
                                    <div
                                        class="bg-white/80 p-3 rounded-xl flex items-center justify-between border border-rose-200">
                                        <div>
                                            <div class="text-[10px] font-bold text-rose-500 uppercase"
                                                x-text="c.guard_name"></div>
                                            <div class="text-sm font-bold text-gray-900" x-text="c.site_name"></div>
                                        </div>
                                        <div class="text-right">
                                            <div class="text-[10px] text-gray-500" x-text="c.from + ' - ' + c.to"></div>
                                            <div class="text-[10px] font-bold text-gray-400" x-text="c.duty_number"></div>
                                        </div>
                                    </div>
                                </template>
                            </div>
                        </div>
                    </div>
                </div>
            </template>

            <!-- Unavailable Day Alert Section -->
            <template x-if="getUnavailableConflicts().length > 0">
                <div
                    class="p-6 bg-orange-50 border-2 border-orange-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-orange-500 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-orange-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-orange-800 mb-1">Guard Unavailable!</h3>
                            <p class="text-orange-600 text-sm mb-4 font-medium">The following guards are marked as
                                unavailable on <span class="font-black uppercase" x-text="getDayName(scheduleDate)"></span>.
                            </p>
                            <div class="space-y-3 max-h-40 overflow-y-auto pr-2">
                                <template x-for="emp in getUnavailableConflicts()" :key="emp.id">
                                    <div
                                        class="bg-white/80 p-3 rounded-xl flex items-center justify-between border border-orange-200">
                                        <div>
                                            <div class="text-sm font-bold text-gray-900" x-text="emp.name"></div>
                                            <div class="text-[10px] text-gray-500" x-text="emp.employee_id"></div>
                                        </div>
                                        <button type="button" @click="toggleEmployee(emp.id)"
                                            class="px-3 py-1 bg-orange-100 text-orange-700 text-[10px] font-bold rounded-lg hover:bg-orange-200">
                                            Remove
                                        </button>
                                    </div>
                                </template>
                            </div>
                        </div>
                    </div>
                </div>
            </template>

            <!-- Max Shift Hours Alert Section -->
            <template x-if="isUserRole && calculateDuration() > maxShiftHours">
                <div class="p-6 bg-red-50 border-2 border-red-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-red-600 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-red-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                </path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-red-800 mb-1">Shift Duration Error!</h3>
                            <p class="text-red-600 text-sm font-medium"
                                x-text="'Total shift duration (' + getDurationString() + ') exceeds the maximum allowed shift hours of ' + maxShiftHours + ' hours.'">
                            </p>
                        </div>
                    </div>
                </div>
            </template>

            <!-- Validation Errors Section -->
            @if ($errors->any())
                <div class="p-6 bg-red-50 border-2 border-red-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-red-500 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-red-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-red-800 mb-2">Input Validation Errors</h3>
                            <ul class="list-disc list-inside text-red-600 text-sm space-y-1 font-medium">
                                @foreach ($errors->all() as $error)
                                    <li>{{ $error }}</li>
                                @endforeach
                            </ul>
                        </div>
                    </div>
                </div>
            @endif
        </div>

        <form action="{{ route('schedules.store') }}" method="POST" class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            @csrf

            <!-- Left Column: Form Fields -->
            <div class="lg:col-span-2 space-y-6">
                <!-- Site & Timing Section -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Duty # -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Duty Number</label>
                            <input type="text" name="duty_number" value="{{ $dutyNumber }}" readonly
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl text-gray-500 font-mono font-bold outline-none cursor-not-allowed">
                        </div>

                        <!-- Schedule Date -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Schedule Date <span
                                    class="text-red-500">*</span></label>
                            <input type="date" name="schedule_date" x-model="scheduleDate" @change="fetchEmployeesInfo"
                                required
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                        </div>

                        <!-- Select Site (Searchable) -->
                        <div class="md:col-span-1 relative" @click.away="siteSearchOpen = false">
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Select Site <span
                                    class="text-red-500">*</span></label>
                            <div class="relative">
                                <input type="text" x-model="siteSearch" @focus="siteSearchOpen = true"
                                    placeholder="Search by Site Name or ID..."
                                    class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                                <input type="hidden" name="site_id" :value="selectedSiteId" required>

                                <!-- Dropdown Results -->
                                <div x-show="siteSearchOpen" x-cloak
                                    class="absolute z-50 w-full mt-2 bg-white border border-gray-100 rounded-2xl shadow-2xl max-h-64 overflow-y-auto p-2 space-y-1 animate-in fade-in zoom-in duration-200">
                                    <template x-for="site in getFilteredSites()" :key="site.id">
                                        <button type="button" @click="selectSite(site)"
                                            class="w-full text-left px-4 py-3 hover:bg-blue-50 rounded-xl transition-all group">
                                            <div class="flex items-center justify-between">
                                                <div>
                                                    <div class="text-sm font-bold text-gray-900 group-hover:text-blue-700"
                                                        x-text="site.name"></div>
                                                    <div class="text-[10px] text-gray-400 font-bold uppercase tracking-wider"
                                                        x-text="'ID: ' + site.site_id"></div>
                                                </div>
                                                <div x-show="selectedSiteId == site.id">
                                                    <svg class="w-5 h-5 text-blue-500" fill="none" stroke="currentColor"
                                                        viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round"
                                                            stroke-width="3" d="M5 13l4 4L19 7"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                        </button>
                                    </template>
                                    <div x-show="getFilteredSites().length === 0"
                                        class="p-4 text-center text-gray-400 text-sm italic">
                                        No sites found matching "<span x-text="siteSearch"></span>"
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Site Rate -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Site Rate</label>
                            <div class="relative">
                                <span class="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 font-bold">$</span>
                                <input type="number" step="0.01" name="site_rate" x-model="siteRate" required
                                    class="w-full pl-8 pr-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                            </div>
                        </div>

                        <!-- Tour Routes Selection -->
                        <div x-show="availableTourRoutes.length > 0" x-transition class="md:col-span-2">
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Select Tour Route(s)</label>
                            <div
                                class="grid grid-cols-1 md:grid-cols-2 gap-3 p-4 bg-slate-50 rounded-2xl border border-slate-100">
                                <template x-for="tour in availableTourRoutes" :key="tour.id">
                                    <label
                                        class="flex items-center gap-3 p-3 bg-white border border-slate-100 rounded-xl cursor-pointer hover:border-blue-300 transition-all select-none">
                                        <input type="checkbox" name="tour_route_ids[]" :value="tour.id"
                                            x-model="selectedTourRouteIds"
                                            class="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500">
                                        <div>
                                            <div class="text-sm font-bold text-slate-700" x-text="tour.description"></div>
                                            <div class="text-[10px] font-bold text-slate-400 uppercase tracking-wider"
                                                x-text="tour.estimated_duration_minutes + ' min • ' + tour.checkpoints_count + ' checkpoints'">
                                            </div>
                                        </div>
                                    </label>
                                </template>
                            </div>
                        </div>

                        <!-- Assign Guards -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Add Guards <span
                                    class="text-red-500">*</span></label>
                            <div class="relative" x-data="{ open: false }" @click.away="open = false">
                                <button type="button" @click="open = !open"
                                    class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl text-left flex items-center justify-between">
                                    <span class="text-sm font-bold text-gray-700"
                                        x-text="selectedEmployeeIds.length > 0 ? selectedEmployeeIds.length + ' Selected' : 'Choose Guard(s)'"></span>
                                    <svg class="w-4 h-4 text-gray-400 transition-transform"
                                        :class="open ? 'rotate-180' : ''" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 9l-7 7-7-7"></path>
                                    </svg>
                                </button>
                                <div x-show="open" x-cloak
                                    class="absolute z-10 w-full mt-2 bg-white border border-gray-100 rounded-2xl shadow-2xl max-h-60 overflow-y-auto p-2 space-y-1">

                                    <div class="sticky top-0 bg-white p-2 border-b border-gray-100 mb-2 z-10">
                                        <input type="text" x-model="employeeSearch"
                                            class="w-full px-3 py-2 bg-gray-50 border border-gray-200 rounded-lg text-xs font-bold focus:ring-2 focus:ring-blue-500/10 focus:border-blue-500 outline-none"
                                            placeholder="Search guards..." @click.stop>
                                    </div>

                                    <template x-for="emp in getFilteredEmployees()" :key="emp.id">
                                        <label
                                            class="flex items-center gap-3 p-2 hover:bg-gray-50 rounded-lg cursor-pointer transition-colors"
                                            :class="selectedEmployeeIds.includes(String(emp.id)) ? 'bg-blue-50' : ''">
                                            <input type="checkbox" :value="emp.id" x-model="selectedEmployeeIds"
                                                @change="updateSiteRate(); fetchEmployeesInfo()"
                                                class="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500">
                                            <div class="flex-1">
                                                <div class="text-sm font-bold text-gray-900"
                                                    x-text="emp.first_name + ' ' + emp.last_name"></div>
                                                <div class="text-[10px] text-gray-400 font-semibold uppercase tracking-wider"
                                                    x-text="emp.employee_id">
                                                </div>
                                                <template x-if="emp.unavailable_days && emp.unavailable_days.length > 0">
                                                    <div class="text-[9px] font-bold text-amber-500 mt-0.5">
                                                        Unavail: <span x-text="emp.unavailable_days.join(', ')"></span>
                                                    </div>
                                                </template>
                                            </div>
                                        </label>
                                    </template>

                                    <div x-show="getFilteredEmployees().length === 0"
                                        class="p-4 text-center text-gray-400 text-xs font-bold italic">
                                        No guards found.
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- From & To Date/Time -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">From Date & Time <span
                                    class="text-red-500">*</span></label>
                            <input type="datetime-local" x-model="fromDatetime" @change="fetchEmployeesInfo"
                                name="from_datetime" required
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                        </div>
                        <div>
                            <div class="flex items-center justify-between mb-2">
                                <label class="block text-xs font-bold text-gray-400 uppercase">To Date & Time <span
                                        class="text-red-500">*</span></label>
                                <template x-if="getDurationString()">
                                    <span
                                        class="text-[10px] font-black px-2 py-0.5 bg-blue-50 text-blue-600 rounded-lg uppercase tracking-wider animate-in fade-in slide-in-from-right-2"
                                        x-text="'Duration: ' + getDurationString()"></span>
                                </template>
                                <template
                                    x-if="fromDatetime && toDatetime && new Date(toDatetime) <= new Date(fromDatetime)">
                                    <span
                                        class="text-[10px] font-black px-2 py-0.5 bg-red-50 text-red-600 rounded-lg uppercase tracking-wider animate-pulse">
                                        End time must be after start time
                                    </span>
                                </template>
                            </div>
                            <input type="datetime-local" x-model="toDatetime" @change="fetchEmployeesInfo"
                                name="to_datetime" required
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold"
                                :class="fromDatetime && toDatetime && new Date(toDatetime) <= new Date(fromDatetime) ? 'border-red-300 ring-red-50' : ''">
                        </div>

                        <!-- Time Zone -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Time Zone</label>
                            <select name="time_zone"
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                                @foreach($timeZones as $tz)
                                    <option value="{{ $tz->name }}" {{ $tz->name === $currentTz ? 'selected' : '' }}>
                                        {{ $tz->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <!-- Recurrence Section -->
                    <div class="mt-6 p-4 bg-blue-50 border border-blue-100 rounded-xl">
                        <div class="flex items-center gap-3 mb-2">
                            <input type="checkbox" id="is_recurring" name="is_recurring" x-model="isRecurring"
                                class="w-5 h-5 text-blue-600 rounded focus:ring-blue-500 border-gray-300 cursor-pointer">
                            <label for="is_recurring" class="text-sm font-bold text-blue-900 select-none cursor-pointer">
                                Run this schedule continuously (Recursive)
                            </label>
                        </div>

                        <div x-show="isRecurring" x-transition>
                            <label class="block text-xs font-bold text-blue-400 uppercase mb-2 mt-4">Repeat Until Date
                                (Inclusive)</label>
                            <input type="date" name="repeat_until"
                                class="w-full px-4 py-3 bg-white border border-blue-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold text-blue-900">

                            <!-- Exclude Days -->
                            <div class="mt-4 p-4 bg-white/50 border border-blue-100 rounded-xl gap-4">
                                <label class="block text-xs font-bold text-blue-400 uppercase mb-3">Exclude Days
                                    (Optional)</label>
                                <div class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-7 gap-2">
                                    @php
                                        $days = [
                                            0 => 'Sun',
                                            1 => 'Mon',
                                            2 => 'Tue',
                                            3 => 'Wed',
                                            4 => 'Thu',
                                            5 => 'Fri',
                                            6 => 'Sat'
                                        ];
                                    @endphp
                                    @foreach($days as $value => $label)
                                        <label
                                            class="flex items-center gap-2 p-2 bg-white border border-blue-50 rounded-lg cursor-pointer hover:bg-blue-50 transition-colors">
                                            <input type="checkbox" name="exclude_days[]" value="{{ $value }}"
                                                class="rounded border-blue-300 text-blue-600 focus:ring-blue-500">
                                            <span class="text-[10px] font-bold text-blue-700 uppercase">{{ $label }}</span>
                                        </label>
                                    @endforeach
                                </div>
                                <p class="text-[9px] text-blue-400 font-bold uppercase mt-2">Selected days will be skipped
                                    during the recurrence period.</p>
                            </div>

                            <p class="text-[10px] text-blue-400 font-bold uppercase mt-2">
                                A separate duty schedule will be created for each day from the start date until this date.
                            </p>
                        </div>
                    </div>
                </div>

                <!-- Guard Individual Settings Section -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8">
                    <h2 class="text-lg font-bold text-gray-800 mb-6 flex items-center gap-2">
                        <svg class="w-5 h-5 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z">
                            </path>
                        </svg>
                        Guard Individual Pricing & Wages
                    </h2>

                    <div class="space-y-8">
                        <template x-for="(emp, empIndex) in selectedEmployeesData" :key="emp.id">
                            <div class="p-6 bg-gray-50 rounded-3xl border border-gray-100 relative overflow-hidden">
                                <!-- Watermark background color -->
                                <div
                                    class="absolute -right-10 -top-10 text-gray-200/50 -rotate-12 select-none pointer-events-none">
                                    <svg class="w-32 h-32" fill="currentColor" viewBox="0 0 24 24">
                                        <path
                                            d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z">
                                        </path>
                                    </svg>
                                </div>

                                <div class="relative z-10">
                                    <!-- Header -->
                                    <div class="flex items-center gap-4 mb-6">
                                        <img :src="emp.profile_picture"
                                            class="w-12 h-12 rounded-full object-cover ring-4 ring-white shadow-sm">
                                        <div class="flex-1">
                                            <div class="font-bold text-gray-900" x-text="emp.name"></div>
                                            <div class="flex items-center gap-2">
                                                <div class="text-[10px] text-gray-400 font-bold uppercase tracking-widest"
                                                    x-text="emp.employee_id"></div>
                                                <template x-if="emp.stat_holiday">
                                                    <span
                                                        class="px-2 py-0.5 bg-amber-100 text-amber-700 text-[9px] font-black rounded-lg uppercase tracking-tight border border-amber-200"
                                                        :title="'Stat Holiday: ' + emp.stat_holiday.name + ' (' + emp.stat_holiday.multiplier + 'x Multiplier)'">
                                                        Stat Holiday: <span x-text="emp.stat_holiday.name"></span>
                                                    </span>
                                                </template>
                                            </div>
                                        </div>
                                        <div class="bg-blue-50/50 p-2 rounded-lg mt-1 w-full"
                                            x-show="emp.unavailable_days && emp.unavailable_days.length > 0">
                                            <div class="text-[10px] font-bold text-amber-600 flex items-start gap-1">
                                                <svg class="w-3 h-3 flex-shrink-0" fill="none" stroke="currentColor"
                                                    viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                                    </path>
                                                </svg>
                                                <span>Unavailable: <span
                                                        x-text="emp.unavailable_days.join(', ')"></span></span>
                                            </div>
                                        </div>
                                        <div class="bg-emerald-50/50 p-2 rounded-lg mt-1 w-full"
                                            x-show="!emp.unavailable_days || emp.unavailable_days.length === 0">
                                            <div class="text-[10px] font-bold text-emerald-600 flex items-center gap-1">
                                                <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M5 13l4 4L19 7"></path>
                                                </svg>
                                                Available all days
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Settings Grid -->
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-6" x-show="calculateDuration() > 0">
                                    <!-- GAS Rate -->
                                    <div class="md:col-span-2">
                                        <label class="block text-[10px] font-bold text-gray-400 uppercase mb-2">GAS Rate
                                            (Optional)</label>
                                        <div class="relative">
                                            <span
                                                class="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 font-bold">$</span>
                                            <input type="number" step="0.01" :name="'employees[' + emp.id + '][gas_rate]'"
                                                x-model="emp.gas_rate"
                                                class="w-full pl-8 pr-4 py-3 bg-white border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                                        </div>
                                    </div>

                                    <!-- Wage Types for this employee -->
                                    <div class="md:col-span-2">
                                        <label
                                            class="block text-[10px] font-black text-slate-400 uppercase tracking-[0.2em] mb-4">Select
                                            Wage Types</label>
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <template x-for="(wt, wtIndex) in emp.wage_types" :key="wt.virtual_id || wt.id">
                                                <div class="p-6 bg-white border border-slate-100 rounded-[2rem] shadow-sm hover:shadow-md transition-all space-y-4"
                                                    :class="wt.selected ? 'ring-2 ring-blue-500/10 border-blue-500/20' : 'opacity-60'">
                                                    <div class="flex items-center justify-between">
                                                        <div class="flex items-center gap-3">
                                                            <input type="checkbox" x-model="wt.selected"
                                                                @change="recalculateAllocatedHours(emp)"
                                                                class="w-6 h-6 rounded-lg border-slate-300 text-blue-600 focus:ring-blue-500/20 transition-all cursor-pointer">
                                                            <div>
                                                                <p class="text-xs font-black text-slate-900 uppercase tracking-tight"
                                                                    x-text="wt.name"></p>
                                                                <template
                                                                    x-if="wt.virtual_id && wt.virtual_id.includes('_hol')">
                                                                    <span
                                                                        class="text-[8px] font-black px-1.5 py-0.5 bg-amber-100 text-amber-700 rounded-md uppercase tracking-tighter">Stat
                                                                        Multiplier Applied</span>
                                                                </template>
                                                                <p class="text-[10px] font-bold text-slate-400 uppercase tracking-widest mt-0.5"
                                                                    x-text="'ID: ' + wt.id"></p>
                                                            </div>
                                                        </div>
                                                        <div class="w-24 relative">
                                                            <span
                                                                class="absolute left-3 top-1/2 -translate-y-1/2 text-[10px] font-black text-slate-300">$</span>
                                                            <input type="number" step="0.01" x-model="wt.rate"
                                                                class="w-full pl-6 pr-3 py-2 bg-slate-50 border border-slate-100 rounded-xl text-xs font-black text-slate-900 focus:ring-4 focus:ring-blue-500/5 focus:border-blue-500 outline-none transition-all placeholder:text-slate-200"
                                                                placeholder="Rate">
                                                        </div>
                                                    </div>

                                                    <!-- Hour Input if multiple are selected OR multiple guards are present -->
                                                    <template
                                                        x-if="wt.selected && (emp.wage_types.filter(w => w.selected).length > 1 || selectedEmployeeIds.length > 1)">
                                                        <div
                                                            class="pt-4 border-t border-slate-50 flex items-center justify-between gap-4 animate-in slide-in-from-top-2 duration-300">
                                                            <label
                                                                class="text-[10px] font-black text-slate-400 uppercase">Hours
                                                                to Calculate</label>
                                                            <div class="relative w-28">
                                                                <input type="number" step="0.1" x-model="wt.allocated_hours"
                                                                    class="w-full px-4 py-2 bg-blue-50/50 border border-blue-100 rounded-xl text-xs font-black text-blue-900 focus:ring-4 focus:ring-blue-500/5 focus:border-blue-500 outline-none transition-all"
                                                                    placeholder="0.0">
                                                                <span
                                                                    class="absolute right-3 top-1/2 -translate-y-1/2 text-[10px] font-black text-blue-300">HRS</span>
                                                            </div>
                                                        </div>
                                                    </template>

                                                    <!-- Hidden inputs for JSON/Pivot storage -->
                                                    <template x-if="wt.selected">
                                                        <div>
                                                            <input type="hidden"
                                                                :name="'employees[' + emp.id + '][wage_types][' + wtIndex + '][id]'"
                                                                :value="wt.id">
                                                            <input type="hidden"
                                                                :name="'employees[' + emp.id + '][wage_types][' + wtIndex + '][rate]'"
                                                                :value="wt.rate">
                                                            <input type="hidden"
                                                                :name="'employees[' + emp.id + '][wage_types][' + wtIndex + '][name]'"
                                                                :value="wt.name">
                                                            <input type="hidden"
                                                                :name="'employees[' + emp.id + '][wage_types][' + wtIndex + '][allocated_hours]'"
                                                                :value="wt.allocated_hours">
                                                        </div>
                                                    </template>
                                                </div>
                                            </template>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Hidden input for the employee ID -->
                            <input type="hidden" name="employee_ids[]" :value="emp.id">
                    </div>
                </div>
                </template>

                <!-- Hidden inputs to ensure employee_ids are always submitted -->
                <template x-for="empId in selectedEmployeeIds" :key="empId">
                    <input type="hidden" name="employee_ids[]" :value="empId">
                </template>

                <div x-show="selectedEmployeeIds.length === 0"
                    class="p-12 text-center bg-gray-50 border-2 border-dashed border-gray-200 rounded-3xl">
                    <div class="text-gray-400 font-medium">No guards selected. Assign guards above to configure
                        their individual rates.</div>
                </div>
            </div>
    </div>

    <!-- Footer Remarks -->
    <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8">
        <div class="grid grid-cols-1 gap-6">
            <div>
                <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Customer Reference</label>
                <input type="text" name="customer_reference"
                    class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
            </div>
            <div>
                <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Comments / Special
                    Instructions</label>
                <textarea name="comments" rows="3"
                    class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-medium"></textarea>
            </div>
        </div>

        <!-- Hour Allocation Validation -->
        <!-- Hour Allocation Validation Removed -->

        <div class="mt-10 pt-8 border-t border-gray-100 flex gap-4">
            <button type="submit"
                class="px-10 py-4 bg-blue-600 text-white font-bold rounded-2xl hover:bg-blue-700 shadow-xl shadow-blue-500/25 transition-all active:scale-95 disabled:opacity-50 disabled:cursor-not-allowed disabled:hover:bg-blue-600"
                :disabled="(fromDatetime && toDatetime && new Date(toDatetime) <= new Date(fromDatetime)) || (isUserRole && calculateDuration() > maxShiftHours)">Create
                Duty Schedule</button>
            <button type="reset" @click="selectedEmployeeIds = []; fetchEmployeesInfo();"
                class="px-8 py-4 bg-gray-50 text-gray-500 font-bold rounded-2xl hover:bg-gray-100 transition-all">Reset
                Form</button>
        </div>
    </div>
    </div>

    <!-- Right Column: Site & Employee Preview -->
    <div class="space-y-6">
        <!-- Site Details Card -->
        <div x-show="siteInfo"
            class="bg-white rounded-[2.5rem] shadow-2xl shadow-gray-200/50 border border-gray-100 overflow-hidden sticky top-6 animate-in slide-in-from-right-10 duration-500">
            <div class="p-6 bg-slate-900 text-white relative overflow-hidden">
                <div class="absolute -right-4 -bottom-4 w-24 h-24 bg-white/5 rounded-full blur-2xl"></div>
                <h3 class="text-[10px] font-black uppercase tracking-[0.2em] opacity-70 mb-1">Site Deployment</h3>
                <div class="text-xl font-black tracking-tight" x-text="siteInfo.name"></div>
            </div>
            <div class="p-8 space-y-8">
                <div>
                    <label class="block text-[10px] font-black text-gray-400 uppercase tracking-widest mb-3">Deployment
                        Address</label>
                    <p class="text-sm font-bold text-gray-700 leading-relaxed flex items-start gap-3">
                        <svg class="w-5 h-5 text-blue-500 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor"
                            viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z">
                            </path>
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                        </svg>
                        <span x-text="siteInfo.address"></span>
                    </p>
                </div>

                <div class="grid grid-cols-2 gap-4">
                    <div class="p-4 bg-gray-50 rounded-2xl border border-gray-100">
                        <span
                            class="block text-[9px] font-black text-gray-400 uppercase tracking-widest mb-1">Identifier</span>
                        <span class="text-xs font-black text-gray-900" x-text="siteInfo.site_id"></span>
                    </div>
                    <div class="p-4 bg-gray-50 rounded-2xl border border-gray-100">
                        <span class="block text-[9px] font-black text-gray-400 uppercase tracking-widest mb-1">Billing
                            Rate</span>
                        <span class="text-xs font-black text-blue-600" x-text="'$' + siteInfo.rate + '/hr'"></span>
                    </div>
                </div>

                <!-- Loading State -->
                <div x-show="isFetching" class="space-y-3">
                    <template x-for="i in 2">
                        <div class="flex items-center gap-4 p-3 bg-gray-50 rounded-2xl animate-pulse">
                            <div class="w-10 h-10 bg-gray-200 rounded-xl"></div>
                            <div class="flex-1 space-y-2">
                                <div class="h-2.5 bg-gray-200 rounded-full w-24"></div>
                                <div class="h-2 bg-gray-200 rounded-full w-16"></div>
                            </div>
                        </div>
                    </template>
                </div>

                <!-- Economics Card -->
                <div class="bg-white rounded-[2rem] shadow-sm border border-slate-100 overflow-hidden"
                    x-show="financialSummary.revenue > 0 || financialSummary.payout > 0">
                    <div class="p-6 bg-slate-900">
                        <div class="flex items-center justify-between">
                            <h3 class="text-[10px] font-black text-slate-400 uppercase tracking-[0.2em]">Duty
                                Economics</h3>
                            <div class="px-2 py-1 bg-emerald-500/10 rounded-lg">
                                <span class="text-[9px] font-black text-emerald-500 uppercase tracking-wider"
                                    x-text="financialSummary.marginPercent + '% Margin'"></span>
                            </div>
                        </div>
                    </div>
                    <div class="p-6 space-y-6">
                        <!-- Revenue -->
                        <div class="flex items-center justify-between">
                            <div class="flex items-center gap-3">
                                <div class="w-8 h-8 bg-blue-50 rounded-xl flex items-center justify-center text-blue-600">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5"
                                            d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z">
                                        </path>
                                    </svg>
                                </div>
                                <span class="text-[11px] font-black text-slate-400 uppercase tracking-wider">Gross
                                    Revenue</span>
                            </div>
                            <span class="text-lg font-black text-slate-900" x-text="'$' + financialSummary.revenue"></span>
                        </div>

                        <div class="h-px bg-slate-50"></div>

                        <!-- Payout -->
                        <div class="flex items-center justify-between">
                            <div class="flex items-center gap-3">
                                <div class="w-8 h-8 bg-rose-50 rounded-xl flex items-center justify-center text-rose-600">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5"
                                            d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z">
                                        </path>
                                    </svg>
                                </div>
                                <span class="text-[11px] font-black text-slate-400 uppercase tracking-wider">Guard
                                    Payout</span>
                                <template
                                    x-if="selectedEmployeesData.some(e => e.holiday_split && e.holiday_split.holiday_hours > 0)">
                                    <span
                                        class="px-2 py-0.5 bg-amber-500/10 text-amber-500 text-[10px] font-black rounded-lg uppercase tracking-tight ml-2">Holiday
                                        Pay Included</span>
                                </template>
                            </div>
                            <span class="text-lg font-black text-slate-900" x-text="'$' + financialSummary.payout"></span>
                        </div>

                        <!-- Margin Card -->
                        <div class="p-6 bg-slate-50 rounded-[1.5rem] mt-4 flex items-center justify-between">
                            <div>
                                <p class="text-[9px] font-black text-slate-400 uppercase tracking-[0.1em] mb-1">Net
                                    Margin</p>
                                <p class="text-2xl font-black text-emerald-600" x-text="'$' + financialSummary.margin"></p>
                            </div>
                            <div class="w-12 h-12 bg-white rounded-2xl flex items-center justify-center shadow-sm">
                                <svg class="w-6 h-6 text-emerald-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5"
                                        d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6"></path>
                                </svg>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Assigned Employees Preview -->
                <div x-show="selectedEmployeesData.length > 0 && !isFetching">
                    <label class="block text-[10px] font-black text-gray-400 uppercase tracking-widest mb-4">Assigned
                        Force (<span x-text="selectedEmployeesData.length"></span>)</label>
                    <div class="space-y-3">
                        <template x-for="emp in selectedEmployeesData" :key="emp.id">
                            <div
                                class="flex items-center gap-4 p-3 bg-white border border-gray-100 rounded-2xl hover:border-blue-200 transition-all group">
                                <div class="relative">
                                    <img :src="emp.profile_picture"
                                        class="w-10 h-10 rounded-xl object-cover ring-2 ring-gray-100 group-hover:ring-blue-100 transition-all">
                                    <div x-show="emp.is_license_expired"
                                        class="absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full border-2 border-white shadow-sm ring-2 ring-red-100">
                                    </div>
                                </div>
                                <div class="flex-1 min-w-0">
                                    <div class="text-[11px] font-black text-gray-900 uppercase truncate" x-text="emp.name">
                                    </div>
                                    <div class="flex items-center gap-2">
                                        <span class="text-[9px] font-black text-blue-600 uppercase"
                                            x-text="'Est. Pay: $' + calculateGuardPayout(emp).toFixed(2)"></span>
                                        <span class="w-1 h-1 rounded-full bg-gray-300"></span>
                                        <template x-if="emp.holiday_split && emp.holiday_split.has_holiday">
                                            <span class="text-[9px] font-bold text-amber-600"
                                                x-text="'(Holiday: ' + emp.holiday_split.holiday_hours.toFixed(1) + 'h)'"></span>
                                        </template>
                                        <span class="text-[9px] font-black text-slate-400 uppercase"
                                            x-text="getDurationString()"></span>
                                    </div>
                                </div>
                                <button type="button" @click="toggleEmployee(emp.id)"
                                    class="text-gray-300 hover:text-red-500 p-1">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5"
                                            d="M6 18L18 6M6 6l12 12"></path>
                                    </svg>
                                </button>
                            </div>
                        </template>
                    </div>
                </div>
            </div>
        </div>

        <!-- Empty State Preview -->
        <div x-show="!siteInfo && selectedEmployeesData.length === 0"
            class="bg-white rounded-[2.5rem] border-2 border-dashed border-gray-200 p-12 text-center sticky top-6">
            <div class="w-16 h-16 bg-gray-50 rounded-3xl flex items-center justify-center mx-auto mb-4 text-gray-300">
                <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                        d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2">
                    </path>
                </svg>
            </div>
            <h3 class="text-xs font-black text-gray-400 uppercase tracking-widest">Real-time Deployment Preview</h3>
            <p class="text-[10px] text-gray-400 font-bold uppercase tracking-wider mt-1">Select a site and assign
                guards to see dynamic calculations.</p>
        </div>
    </div>
    </form>
    </div>

    @push('scripts')
        <script>
            function scheduleForm() {
                return {
                    selectedSiteId: '',
                    siteSearch: '',
                    siteSearchOpen: false,
                    allSites: @json($sites),
                    allEmployees: @json($employees),
                    employeeSearch: '',
                    selectedEmployeeIds: [],
                    selectedEmployeesData: [],
                    siteRate: '',
                    baseSiteRate: 0,
                    siteInfo: null,
                    fromDatetime: '{{ $defaultStart }}',
                    toDatetime: '{{ $defaultEnd }}',
                    scheduleDate: '{{ $defaultDate }}',
                    allConflicts: [],
                    isFetching: false,
                    isRecurring: false,
                    maxShiftHours: {{ $maxShiftHours }},
                    isUserRole: {{ auth()->user()->role === 'user' ? 'true' : 'false' }},
                    availableTourRoutes: [],
                    selectedTourRouteIds: [],

                    getDayName(dateStr) {
                        if (!dateStr) return '';
                        const date = new Date(dateStr);
                        // Adjust for timezone offset if needed, but usually YYYY-MM-DD is local
                        // To be safe with date input value (YYYY-MM-DD), appends T00:00 to ensure local parse
                        const dayNames = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
                        // getDay() returns 0 for Sunday, 1 for Monday...
                        // We need to map it. 0=Sun, 1=Mon...
                        // My array is Mon=0...
                        // Let's use standard:
                        const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
                        return days[new Date(dateStr + 'T00:00:00').getDay()];
                    },

                    getUnavailableConflicts() {
                        if (!this.scheduleDate) return [];
                        const currentDay = this.getDayName(this.scheduleDate);
                        return this.selectedEmployeesData.filter(emp =>
                            emp.unavailable_days && emp.unavailable_days.includes(currentDay)
                        );
                    },

                    calculateDuration() {
                        if (!this.fromDatetime || !this.toDatetime) return null;
                        const start = new Date(this.fromDatetime);
                        const end = new Date(this.toDatetime);
                        const diffMs = end - start;
                        if (diffMs <= 0) return 0;
                        return diffMs / (1000 * 60 * 60); // Return decimal hours
                    },

                    getFilteredSites() {
                        if (!this.siteSearch) return this.allSites;
                        const search = this.siteSearch.toLowerCase();
                        return this.allSites.filter(site =>
                            site.name.toLowerCase().includes(search) ||
                            site.site_id.toLowerCase().includes(search)
                        );
                    },

                    getFilteredEmployees() {
                        if (!this.employeeSearch) return this.allEmployees;
                        const search = this.employeeSearch.toLowerCase();
                        return this.allEmployees.filter(emp =>
                            emp.first_name.toLowerCase().includes(search) ||
                            emp.last_name.toLowerCase().includes(search) ||
                            (emp.employee_id && emp.employee_id.toLowerCase().includes(search))
                        );
                    },

                    selectSite(site) {
                        this.selectedSiteId = site.id;
                        this.siteSearch = `${site.site_id} - ${site.name}`;
                        this.siteSearchOpen = false;
                        this.selectedTourRouteIds = [];
                        this.fetchSiteInfo();
                        this.fetchTourRoutes(site.id);
                    },

                    async fetchTourRoutes(siteId) {
                        if (!siteId) {
                            this.availableTourRoutes = [];
                            return;
                        }
                        try {
                            const response = await fetch(`/sites/${siteId}/tour-routes`, {
                                headers: { 'Accept': 'application/json' }
                            });
                            const data = await response.json();
                            this.availableTourRoutes = data.tourRoutes || [];
                        } catch (e) {
                            console.error('Error fetching tour routes:', e);
                        }
                    },

                    getDurationString() {
                        const hrs = this.calculateDuration();
                        if (!hrs) return null;
                        const h = Math.floor(hrs);
                        const m = Math.round((hrs - h) * 60);
                        return `${h}h ${m}m`;
                    },

                    getTotalAllocatedHours() {
                        let total = 0;
                        this.selectedEmployeesData.forEach(emp => {
                            const selectedWages = emp.wage_types.filter(w => w.selected);
                            if (selectedWages.length === 0) return;

                            // If only one is selected and only one guard, it gets full duration
                            // BUT if there are multiple guards, we rely on the manual input if provided
                            // or split it? The user wants total <= duration.
                            if (selectedWages.length === 1 && this.selectedEmployeeIds.length === 1) {
                                total += this.calculateDuration();
                            } else {
                                selectedWages.forEach(w => {
                                    total += parseFloat(w.allocated_hours || 0);
                                });
                            }
                        });
                        return total;
                    },

                    isAllocationMismatch() {
                        const duration = this.calculateDuration();
                        if (duration <= 0) return false;
                        const total = this.getTotalAllocatedHours();
                        // Use a small epsilon for floating point comparison
                        return Math.abs(total - duration) > 0.01;
                    },

                    isOverAllocated() {
                        const duration = this.calculateDuration();
                        if (duration <= 0) return false;
                        return this.getTotalAllocatedHours() > (duration + 0.01);
                    },

                    // Calculate payout for a specific employee based on their selected wage types and hour allocations
                    calculateGuardPayout(emp) {
                        const totalHrs = this.calculateDuration();
                        if (totalHrs <= 0) return 0;

                        const selectedWages = emp.wage_types.filter(w => w.selected);
                        if (selectedWages.length === 0) return 0;

                        // If only one is selected and only one guard, it gets all hours automatically
                        if (selectedWages.length === 1 && this.selectedEmployeeIds.length === 1) {
                            return totalHrs * parseFloat(selectedWages[0].rate || 0);
                        }

                        // If multiple, sum up the allocated hours * rates
                        let total = 0;
                        selectedWages.forEach(w => {
                            total += (parseFloat(w.allocated_hours) || 0) * parseFloat(w.rate || 0);
                        });
                        return total;
                    },

                    get financialSummary() {
                        const hrs = this.calculateDuration();
                        const revenue = hrs * (parseFloat(this.siteRate) || 0);

                        let totalPayout = 0;
                        this.selectedEmployeesData.forEach(emp => {
                            totalPayout += this.calculateGuardPayout(emp);
                            // Add gas if any
                            totalPayout += (parseFloat(emp.gas_rate) || 0);
                        });

                        const margin = revenue - totalPayout;
                        const marginPercent = revenue > 0 ? (margin / revenue) * 100 : 0;

                        return {
                            revenue: revenue.toFixed(2),
                            payout: totalPayout.toFixed(2),
                            margin: margin.toFixed(2),
                            marginPercent: marginPercent.toFixed(1)
                        };
                    },

                    toggleEmployee(id) {
                        const sId = id.toString();
                        const index = this.selectedEmployeeIds.indexOf(sId);
                        if (index > -1) {
                            this.selectedEmployeeIds.splice(index, 1);
                        } else {
                            this.selectedEmployeeIds.push(sId);
                        }
                        this.updateSiteRate();
                        this.fetchEmployeesInfo();
                    },

                    updateSiteRate() {
                        const count = this.selectedEmployeeIds.length;
                        // If no guards selected, treat as 1 unit for the base rate display?
                        // "site rate appears" initially with 0 guards.
                        const multiplier = count > 0 ? count : 1;
                        // Calculate total rate: Base Rate * Multiplier
                        if (this.baseSiteRate) {
                            this.siteRate = (parseFloat(this.baseSiteRate) * multiplier).toFixed(2);
                        }
                    },

                    async fetchSiteInfo() {
                        if (!this.selectedSiteId) { this.siteInfo = null; return; }
                        try {
                            const response = await fetch(`/schedules/site-info/${this.selectedSiteId}`);
                            const data = await response.json();
                            this.siteInfo = data;
                            this.baseSiteRate = data.rate; // Store base rate
                            this.updateSiteRate(); // Update displayed rate
                        } catch (e) { console.error(e); }
                    },

                    // Recalculate allocated hours based on selection
                    recalculateAllocatedHours(emp) {
                        // 1. Determine available hours for periods
                        const duration = this.calculateDuration() || 0;
                        let regularDuration = duration;
                        let holidayDuration = 0;

                        if (emp.holiday_split && emp.holiday_split.has_holiday) {
                            regularDuration = emp.holiday_split.regular_hours || 0;
                            holidayDuration = emp.holiday_split.holiday_hours || 0;
                        }

                        // 2. Identify selected wage types
                        const selected = emp.wage_types.filter(w => w.selected);

                        // 3. Group by period_type
                        const regularTypes = selected.filter(w => w.period_type === 'regular');
                        const holidayTypes = selected.filter(w => w.period_type === 'holiday');

                        // 4. Distribute
                        if (regularTypes.length > 0) {
                            const hrs = regularDuration / regularTypes.length;
                            regularTypes.forEach(w => w.allocated_hours = hrs.toFixed(2));
                        }

                        if (holidayTypes.length > 0) {
                            const hrs = holidayDuration / holidayTypes.length;
                            holidayTypes.forEach(w => w.allocated_hours = hrs.toFixed(2));
                        }
                    },

                    async fetchEmployeesInfo() {
                        if (this.selectedEmployeeIds.length === 0) {
                            this.selectedEmployeesData = [];
                            this.allConflicts = [];
                            return;
                        }

                        this.isFetching = true;
                        try {
                            // Filter out selected IDs that are already in our data to avoid redundant fetches
                            const fetchPromises = this.selectedEmployeeIds.map(async (id) => {
                                let url = `/schedules/employee-info/${id}?`;
                                let params = new URLSearchParams();
                                if (this.fromDatetime) params.append('start', this.fromDatetime);
                                if (this.toDatetime) params.append('end', this.toDatetime);
                                if (this.scheduleDate) params.append('date', this.scheduleDate);

                                url += params.toString();
                                const res = await fetch(url);
                                if (!res.ok) {
                                    const errorText = await res.text();
                                    console.error(`Error ${res.status}: ${errorText}`);
                                    throw new Error(`Failed to fetch employee info (Status: ${res.status})`);
                                }
                                return await res.json();
                            });

                            const employeesData = await Promise.all(fetchPromises);

                            // Map existing data to preserve user input while adding new data
                            this.selectedEmployeesData = employeesData.map(newData => {
                                const existing = (this.selectedEmployeesData || []).find(e => e.id == newData.id);
                                const duration = this.calculateDuration() || 0;

                                let processedWageTypes = [];
                                (newData.wage_types || []).forEach((wt, index) => {
                                    const holidayHours = newData.holiday_split ? newData.holiday_split.holiday_hours : 0;
                                    const regularHours = newData.holiday_split ? newData.holiday_split.regular_hours : 0;
                                    const hasSplit = holidayHours > 0 && regularHours > 0;
                                    const isFullHoliday = holidayHours > 0 && regularHours <= 0;

                                    const nameLower = wt.name.toLowerCase();
                                    const isSin = nameLower.includes('sin');

                                    // Check if this wage type was already selected in the UI
                                    let isSelected = false;
                                    let allocatedHours = 0;

                                    if (existing && existing.wage_types) {
                                        // Finding by ID is best, handling potential virtual IDs or matches
                                        const prev = existing.wage_types.find(w => w.id == wt.id);
                                        if (prev) {
                                            isSelected = prev.selected;
                                            allocatedHours = prev.allocated_hours;
                                        }
                                    } else {
                                        // Default for new additions: select the first one (usually SIN)
                                        isSelected = index === 0;
                                    }

                                    if (hasSplit) {
                                        // Push Regular Part
                                        processedWageTypes.push({
                                            ...wt,
                                            virtual_id: wt.id + '_reg',
                                            name: wt.name + ' (Regular Part)',
                                            rate: wt.base_rate,
                                            period_type: 'regular',
                                            allocated_hours: allocatedHours,
                                            selected: isSelected
                                        });
                                        // Push Holiday Part
                                        processedWageTypes.push({
                                            ...wt,
                                            virtual_id: wt.id + '_hol',
                                            name: wt.name + ' (Stat Holiday: ' + newData.holiday_split.holiday_name + ')',
                                            rate: wt.holiday_rate,
                                            period_type: 'holiday',
                                            allocated_hours: allocatedHours,
                                            selected: isSelected
                                        });
                                    } else if (isFullHoliday) {
                                        // Full Holiday
                                        processedWageTypes.push({
                                            ...wt,
                                            virtual_id: wt.id,
                                            name: wt.name + ' (Stat Holiday: ' + newData.holiday_split.holiday_name + ')',
                                            rate: wt.holiday_rate,
                                            period_type: 'holiday',
                                            allocated_hours: allocatedHours,
                                            selected: isSelected
                                        });
                                    } else {
                                        // Regular Shift
                                        processedWageTypes.push({
                                            ...wt,
                                            virtual_id: wt.id,
                                            name: wt.name,
                                            rate: wt.base_rate,
                                            period_type: 'regular',
                                            allocated_hours: allocatedHours,
                                            selected: isSelected
                                        });
                                    }
                                });

                                const employeeObj = {
                                    ...newData,
                                    wage_rate: existing ? existing.wage_rate : newData.wage_rate || '',
                                    gas_rate: existing ? existing.gas_rate : '',
                                    wage_types: processedWageTypes
                                };

                                // Perform initial calculation
                                this.recalculateAllocatedHours(employeeObj);

                                return employeeObj;
                            });

                            // Collect conflicts
                            this.allConflicts = [];
                            employeesData.forEach(emp => {
                                (emp.conflicts || []).forEach(c => {
                                    this.allConflicts.push({ ...c, guard_name: emp.name });
                                });
                            });

                        } catch (e) {
                            console.error(e);
                        } finally {
                            this.isFetching = false;
                        }
                    }
                }
            }
        </script>
        <style>
            [x-cloak] {
                display: none !important;
            }

            .scrollbar-hide::-webkit-scrollbar {
                display: none;
            }
        </style>
    @endpush
@endsection