@extends('layouts.user')

@section('title', 'Edit Patroller Schedule')

@section('content')
    <div class="max-w-7xl mx-auto" x-data="patrollerScheduleForm()">
        <div class="mb-8 flex items-center justify-between">
            <div>
                <div class="flex items-center gap-2 text-xs text-slate-400 mb-2">
                    <a href="{{ route('patroller-schedules.index') }}"
                        class="hover:text-blue-600 transition-colors">Patroller Schedules</a>
                    <svg class="w-3 h-3 text-slate-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                    </svg>
                    <span class="text-slate-900 font-bold">Edit</span>
                </div>
                <h1 class="text-2xl font-black text-slate-900 tracking-tight">Edit Schedule:
                    {{ $patrollerSchedule->duty_number }}
                </h1>
            </div>
            <a href="{{ route('patroller-schedules.index') }}"
                class="bg-white border border-slate-200 text-slate-700 hover:bg-slate-50 px-4 py-2 rounded-lg text-sm font-bold transition-all shadow-sm flex items-center gap-2">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18">
                    </path>
                </svg>
                Back to List
            </a>
        </div>

        @if ($errors->any())
            <div class="mb-6 p-4 bg-red-50 border border-red-100 rounded-lg">
                <div class="flex items-center gap-2 text-red-700 font-bold mb-2">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    There were some errors with your submission
                </div>
                <ul class="list-disc list-inside text-sm text-red-600">
                    @foreach ($errors->all() as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
            </div>
        @endif

        <form action="{{ route('patroller-schedules.update', $patrollerSchedule) }}" method="POST" class="space-y-6">
            @csrf
            @method('PUT')

            <!-- Schedule Details -->
            <div class="bg-white rounded-xl shadow-sm border border-slate-200 p-6">
                <h3 class="text-lg font-bold text-slate-800 mb-4 flex items-center gap-2">
                    <span class="w-1 h-6 bg-blue-600 rounded-full"></span>
                    Schedule Details
                </h3>

                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    <!-- Duty Number (Read-only) -->
                    <div>
                        <label class="text-xs font-bold text-slate-500 uppercase mb-1 block">Duty Number</label>
                        <input type="text" value="{{ $patrollerSchedule->duty_number }}" disabled
                            class="w-full px-3 py-2 text-sm bg-slate-50 border border-slate-200 rounded-lg text-slate-500 cursor-not-allowed">
                    </div>

                    <!-- Scheduled Date -->
                    <div>
                        <label class="text-xs font-bold text-slate-500 uppercase mb-1 block">Scheduled Date <span
                                class="text-red-500">*</span></label>
                        <input type="date" name="scheduled_date" required x-model="formData.scheduled_date"
                            class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                    </div>

                    <!-- Select Route (Searchable) -->
                    <div class="flex flex-col relative"
                        x-data="{ open: false, search: '', selected: '{{ $patrollerSchedule->route->name }}', value: '{{ $patrollerSchedule->route_id }}' }"
                        @click.outside="open = false">
                        <label class="text-xs font-bold text-slate-500 uppercase mb-1 block">Select Route <span
                                class="text-red-500">*</span></label>
                        <input type="hidden" name="route_id" x-model="value" required>
                        <button type="button" @click="open = !open"
                            class="w-full text-sm border border-slate-200 rounded-lg focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 py-2.5 px-3 bg-white transition-all shadow-sm text-left flex justify-between items-center h-[42px]">
                            <span x-text="selected || 'Select Route'"
                                :class="{'text-slate-400': !selected, 'text-slate-700': selected}"></span>
                            <svg class="w-4 h-4 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7">
                                </path>
                            </svg>
                        </button>

                        <div x-show="open" x-cloak
                            class="absolute top-full left-0 w-full mt-1 bg-white border border-slate-200 rounded-lg shadow-xl z-50 max-h-60 overflow-hidden flex flex-col">
                            <div class="p-2 border-b border-slate-100">
                                <input type="text" x-model="search" placeholder="Search routes..."
                                    class="w-full text-xs border border-slate-200 rounded px-2 py-1.5 focus:outline-none focus:border-blue-500">
                            </div>
                            <ul class="overflow-y-auto flex-1 p-1">
                                @foreach($routes as $route)
                                    <li x-show="'{{ strtolower($route->name) }}'.includes(search.toLowerCase())"
                                        @click="selected = '{{ $route->name }}'; value = '{{ $route->id }}'; open = false; handleRouteSelect('{{ $route->id }}', '{{ $route->rate }}')"
                                        class="px-3 py-2 hover:bg-blue-50 hover:text-blue-700 rounded text-sm cursor-pointer text-slate-600 transition-colors">
                                        {{ $route->name }}
                                    </li>
                                @endforeach
                            </ul>
                        </div>
                    </div>

                    <!-- Route Rate -->
                    <div>
                        <label class="text-xs font-bold text-slate-500 uppercase mb-1 block">Route Rate <span
                                class="text-red-500">*</span></label>
                        <div class="relative">
                            <span class="absolute left-3 top-1/2 -translate-y-1/2 text-slate-400">$</span>
                            <input type="number" name="route_rate" step="0.01" required x-model="formData.route_rate"
                                @input="updateCalculations"
                                class="w-full pl-7 pr-3 py-2 text-sm border border-slate-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                        </div>
                    </div>
                </div>

                <!-- Route Details Display -->
                <div x-show="routeDetails" x-cloak x-transition
                    class="mt-6 bg-blue-50 border border-blue-200 rounded-lg p-4">
                    <h4 class="text-sm font-bold text-blue-900 mb-3 flex items-center gap-2">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z">
                            </path>
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M15 11a3 3 0 11-6 0 3 3 0 016 0z">
                            </path>
                        </svg>
                        Route Sites (<span x-text="routeDetails?.sites?.length || 0"></span>)
                    </h4>
                    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                        <template x-for="site in routeDetails?.sites || []" :key="site.id">
                            <div class="bg-white rounded-lg p-3 border border-blue-100 shadow-sm">
                                <div class="text-xs font-bold text-slate-800" x-text="site.name"></div>
                                <div class="text-[10px] text-slate-500 mt-1" x-text="site.address + ', ' + site.city"></div>
                                <div class="text-[10px] text-slate-400 mt-0.5" x-show="site.contact_person"
                                    x-text="site.contact_person + ' • ' + site.contact_phone"></div>
                            </div>
                        </template>
                    </div>
                </div>
            </div>

            <!-- Employee & Timing -->
            <div class="bg-white rounded-xl shadow-sm border border-slate-200 p-6">
                <h3 class="text-lg font-bold text-slate-800 mb-4 flex items-center gap-2">
                    <span class="w-1 h-6 bg-blue-600 rounded-full"></span>
                    Employee & Timing
                </h3>

                <!-- Time Selection -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6 pb-6 border-b border-slate-100">
                    <div>
                        <label class="text-xs font-bold text-slate-500 uppercase mb-1 block">From Time <span
                                class="text-red-500">*</span></label>
                        <input type="datetime-local" name="from_time" required x-model="formData.from_time"
                            @change="recheckAllEmployees"
                            class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                    </div>
                    <div>
                        <label class="text-xs font-bold text-slate-500 uppercase mb-1 block">To Time <span
                                class="text-red-500">*</span></label>
                        <input type="datetime-local" name="to_time" required x-model="formData.to_time"
                            @change="recheckAllEmployees"
                            class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all"
                            :class="{'border-red-300 bg-red-50': !isTimeValid}">
                        <div x-show="!isTimeValid" class="text-xs text-red-500 mt-1">
                            To Time must be after From Time
                        </div>
                    </div>
                </div>

                <!-- Max Shift Hours Alert Section -->
                <template x-if="isUserRole && calculateHours() > maxShiftHours">
                    <div
                        class="mb-6 p-4 bg-red-50 border-2 border-red-300 rounded-xl animate-in fade-in zoom-in duration-300">
                        <div class="flex items-start gap-4">
                            <div
                                class="w-10 h-10 bg-red-600 rounded-lg flex items-center justify-center text-white shadow-lg shadow-red-200 flex-shrink-0">
                                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                    </path>
                                </svg>
                            </div>
                            <div class="flex-1">
                                <h3 class="text-sm font-bold text-red-800 mb-1 uppercase tracking-wide">Shift Duration
                                    Error!</h3>
                                <p class="text-red-600 text-xs font-medium"
                                    x-text="'Total shift duration (' + formatDuration(calculateHours()) + ') exceeds the maximum allowed shift hours of ' + maxShiftHours + ' hours.'">
                                </p>
                            </div>
                        </div>
                    </div>
                </template>

                <!-- Total Job Hours Display -->
                <div x-show="formData.from_time && formData.to_time && isTimeValid" x-cloak
                    class="mb-6 pb-6 border-b border-slate-100">
                    <div class="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-4">
                        <div class="flex items-center justify-between">
                            <div class="flex items-center gap-3">
                                <div class="w-10 h-10 bg-blue-600 rounded-lg flex items-center justify-center">
                                    <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                    </svg>
                                </div>
                                <div>
                                    <div class="text-xs font-medium text-blue-600 uppercase tracking-wide">Total Job Hours
                                    </div>
                                    <div class="text-2xl font-bold text-blue-900"
                                        x-text="calculateHours().toFixed(2) + ' hours'"></div>
                                </div>
                            </div>
                            <div class="text-right">
                                <div class="text-xs text-blue-600 font-medium">Duration</div>
                                <div class="text-sm text-blue-800" x-text="formatDuration(calculateHours())"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Employees Selection -->
                <div class="mb-4">
                    <label class="text-xs font-bold text-slate-500 uppercase mb-2 block">Select Employee(s) <span
                            class="text-red-500">*</span></label>

                    <div class="flex gap-2 mb-4">
                        <div class="flex-1 relative" x-data="{ open: false, search: '' }" @click.outside="open = false">
                            <button type="button" @click="open = !open"
                                class="w-full text-sm border border-slate-200 rounded-lg focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 py-2.5 px-3 bg-white transition-all shadow-sm text-left flex justify-between items-center h-[42px]">
                                <span class="text-slate-400">Search and add employees...</span>
                                <svg class="w-4 h-4 text-slate-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M12 4v16m8-8H4">
                                    </path>
                                </svg>
                            </button>

                            <div x-show="open" x-cloak
                                class="absolute top-full left-0 w-full mt-1 bg-white border border-slate-200 rounded-lg shadow-xl z-50 max-h-60 overflow-hidden flex flex-col">
                                <div class="p-2 border-b border-slate-100">
                                    <input type="text" x-model="search" placeholder="Search by name or ID..."
                                        class="w-full text-xs border border-slate-200 rounded px-2 py-1.5 focus:outline-none focus:border-blue-500">
                                </div>
                                <ul class="overflow-y-auto flex-1 p-1">
                                    @foreach($employees as $emp)
                                        <li x-show="'{{ strtolower($emp->first_name . ' ' . $emp->last_name . ' ' . $emp->employee_id) }}'.includes(search.toLowerCase())"
                                            @click="addEmployee({
                                                                                                                                                                                    id: '{{ $emp->id }}',
                                                                                                                                                                                    name: '{{ $emp->first_name }} {{ $emp->last_name }}',
                                                                                                                                                                                    emp_id: '{{ $emp->employee_id ?? 'ID:' . $emp->id }}'
                                                                                                                                                                                }); open = false; search = ''"
                                            class="px-3 py-2 hover:bg-blue-50 hover:text-blue-700 rounded text-sm cursor-pointer text-slate-600 transition-colors flex justify-between items-center">
                                            <div class="flex flex-col">
                                                <span class="font-medium">{{ $emp->first_name }} {{ $emp->last_name }}</span>
                                                <span
                                                    class="text-xs text-slate-400">{{ $emp->employee_id ?? '#' . $emp->id }}</span>
                                            </div>
                                            <span
                                                class="text-xs bg-slate-100 text-slate-500 px-2 py-0.5 rounded">{{ $emp->employee_id ?? '#' . $emp->id }}</span>
                                        </li>
                                    @endforeach
                                </ul>
                            </div>
                        </div>

                        <button type="button" @click="findAvailableGuards()"
                            class="h-[42px] px-4 bg-emerald-600 text-white rounded-lg font-bold text-xs hover:bg-emerald-700 transition-all shadow-sm flex items-center gap-2 whitespace-nowrap">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            Quick Check
                        </button>
                    </div>

                    <!-- Selected Employees List -->
                    <div class="space-y-3">
                        <template x-for="(emp, index) in selectedEmployees" :key="emp.id">
                            <div class="bg-slate-50 border border-slate-200 rounded-lg p-4 relative">
                                <input type="hidden" name="employee_ids[]" :value="emp.id">

                                <div class="flex justify-between items-start mb-3">
                                    <div class="flex items-center gap-2">
                                        <div
                                            class="w-8 h-8 rounded-full bg-blue-100 text-blue-600 flex items-center justify-center text-xs font-bold">
                                            <span x-text="emp.name.charAt(0)"></span>
                                        </div>
                                        <div>
                                            <div class="text-sm font-bold text-slate-800" x-text="emp.name"></div>
                                            <div class="text-xs text-slate-500" x-text="emp.emp_id"></div>
                                            <!-- Availability Display -->
                                            <template x-if="emp.unavailable_days && emp.unavailable_days.length > 0">
                                                <div
                                                    class="text-[10px] font-bold text-emerald-600 mt-1 flex items-start gap-1">
                                                    <svg class="w-3 h-3 flex-shrink-0" fill="none" stroke="currentColor"
                                                        viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round"
                                                            stroke-width="2" d="M5 13l4 4L19 7"></path>
                                                    </svg>
                                                    <span>Available on: <span
                                                            x-text="getAvailableDays(emp.unavailable_days).join(', ')"></span></span>
                                                </div>
                                            </template>
                                            <template x-if="!emp.unavailable_days || emp.unavailable_days.length === 0">
                                                <div
                                                    class="text-[10px] font-bold text-emerald-600 mt-1 flex items-center gap-1">
                                                    <svg class="w-3 h-3" fill="none" stroke="currentColor"
                                                        viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round"
                                                            stroke-width="2" d="M5 13l4 4L19 7"></path>
                                                    </svg>
                                                    Available on all days
                                                </div>
                                            </template>
                                        </div>
                                    </div>
                                    <button type="button" @click="removeEmployee(index)"
                                        class="text-slate-400 hover:text-red-500">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M6 18L18 6M6 6l12 12"></path>
                                        </svg>
                                    </button>
                                </div>

                                <!-- Validation Warnings -->
                                <div x-show="employeeValidations[emp.id]" class="mb-3 space-y-1">
                                    <template x-if="employeeValidations[emp.id]?.has_errors">
                                        <template x-for="error in employeeValidations[emp.id]?.errors || []"
                                            :key="error.type">
                                            <div
                                                class="flex items-center gap-2 text-xs bg-red-50 border border-red-200 rounded px-2 py-1.5 text-red-700">
                                                <svg class="w-3 h-3 flex-shrink-0" fill="none" stroke="currentColor"
                                                    viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                                </svg>
                                                <span x-text="error.message"></span>
                                            </div>
                                        </template>
                                    </template>
                                    <template x-if="employeeValidations[emp.id]?.has_warnings">
                                        <template x-for="warning in employeeValidations[emp.id]?.warnings || []"
                                            :key="warning.type">
                                            <div
                                                class="flex items-center gap-2 text-xs bg-yellow-50 border border-yellow-200 rounded px-2 py-1.5 text-yellow-700">
                                                <svg class="w-3 h-3 flex-shrink-0" fill="none" stroke="currentColor"
                                                    viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                        d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                                    </path>
                                                </svg>
                                                <span x-text="warning.message"></span>
                                            </div>
                                        </template>
                                    </template>
                                </div>

                                <div class="grid grid-cols-1 sm:grid-cols-2 gap-3">
                                    <!-- Wage Types (Multi-select) -->
                                    <div class="flex flex-col relative" x-data="{ open: false }"
                                        @click.outside="open = false">
                                        <label class="text-[10px] font-bold text-slate-500 uppercase mb-1">Wage
                                            Type</label>
                                        <div @click="open = !open"
                                            class="w-full min-h-[34px] px-2 py-1.5 bg-white border border-slate-200 rounded flex flex-wrap gap-1 cursor-pointer">
                                            <template x-if="emp.wage_types.length === 0">
                                                <span class="text-xs text-slate-400">Select Types</span>
                                            </template>
                                            <template x-for="wtId in emp.wage_types" :key="wtId">
                                                <span
                                                    class="text-[10px] bg-blue-50 text-blue-600 border border-blue-100 px-1.5 rounded flex items-center gap-1">
                                                    <span x-text="getWageTypeName(wtId)"></span>
                                                    <span @click.stop="removeWageType(index, wtId)"
                                                        class="hover:text-blue-800">&times;</span>
                                                </span>
                                            </template>
                                            <!-- Hidden inputs for form submission -->
                                            <template x-for="wtId in emp.wage_types" :key="wtId">
                                                <input type="hidden" :name="'wage_types_' + emp.id + '[]'" :value="wtId">
                                            </template>
                                        </div>

                                        <div x-show="open" x-cloak
                                            class="absolute top-full left-0 w-full mt-1 bg-white border border-slate-200 rounded shadow-lg z-10 max-h-40 overflow-y-auto">
                                            @foreach($wageTypes as $wt)
                                                <div @click="toggleWageType(index, '{{ $wt->id }}')"
                                                    class="px-3 py-2 hover:bg-slate-50 text-xs cursor-pointer flex items-center gap-2">
                                                    <div class="w-3 h-3 border border-slate-300 rounded flex items-center justify-center transition-colors"
                                                        :class="{'bg-blue-600 border-blue-600': emp.wage_types.includes('{{ $wt->id }}')}">
                                                        <svg x-show="emp.wage_types.includes('{{ $wt->id }}')"
                                                            class="w-2 h-2 text-white" fill="none" stroke="currentColor"
                                                            viewBox="0 0 24 24">
                                                            <path stroke-linecap="round" stroke-linejoin="round"
                                                                stroke-width="4" d="M5 13l4 4L19 7"></path>
                                                        </svg>
                                                    </div>
                                                    {{ $wt->name }}
                                                </div>
                                            @endforeach
                                        </div>
                                    </div>
                                </div>

                                <!-- Wage Type Details (Rate + Hours to Calculate) -->
                                <div x-show="emp.wage_types.length > 0" class="mt-3 space-y-3">
                                    <div class="text-[10px] font-bold text-slate-500 uppercase mb-2">Assigned Force
                                        Capabilities (Wage Types)</div>
                                    <template x-for="wtId in emp.wage_types" :key="wtId">
                                        <div class="bg-blue-50/50 border border-blue-100 rounded-lg p-3">
                                            <div class="flex items-center justify-between mb-2">
                                                <span class="text-xs font-bold text-blue-700 uppercase"
                                                    x-text="getWageTypeName(wtId)"></span>
                                                <span class="text-[10px] text-blue-600" x-text="'ID: ' + wtId"></span>
                                            </div>
                                            <div class="grid grid-cols-2 gap-2">
                                                <div>
                                                    <label class="text-[10px] font-medium text-slate-600 mb-1 block">Rate
                                                        ($/hr)</label>
                                                    <div class="relative">
                                                        <span
                                                            class="absolute left-2 top-1/2 -translate-y-1/2 text-slate-400 text-xs">$</span>
                                                        <input type="number" :name="'wage_rate_' + emp.id + '_' + wtId"
                                                            step="0.01" x-model="emp.wageRates[wtId]"
                                                            @input="updateCalculations"
                                                            class="w-full pl-5 pr-2 py-1.5 text-xs border border-slate-200 rounded focus:outline-none focus:border-blue-500 bg-white">
                                                    </div>
                                                </div>
                                                <div>
                                                    <label class="text-[10px] font-medium text-slate-600 mb-1 block">Hours
                                                        to Calculate</label>
                                                    <input type="number" :name="'allocated_hours_' + emp.id + '_' + wtId"
                                                        step="0.01" x-model="emp.allocatedHours[wtId]"
                                                        @input="updateCalculations" placeholder="5.00"
                                                        class="w-full px-2 py-1.5 text-xs border border-slate-200 rounded focus:outline-none focus:border-blue-500 bg-white">
                                                </div>
                                            </div>
                                        </div>
                                    </template>
                                </div>

                                <!-- Per-Employee Expenses -->
                                <div class="grid grid-cols-3 gap-2 mt-3">
                                    <div>
                                        <label class="text-[10px] font-bold text-slate-500 uppercase mb-1 block">Gas</label>
                                        <input type="number" :name="'gas_rate_' + emp.id" step="0.01" x-model="emp.gas_rate"
                                            @input="updateCalculations"
                                            class="w-full px-2 py-1.5 text-xs border border-slate-200 rounded focus:outline-none focus:border-blue-500">
                                    </div>
                                    <div>
                                        <label
                                            class="text-[10px] font-bold text-slate-500 uppercase mb-1 block">Allowance</label>
                                        <input type="number" :name="'allowance_' + emp.id" step="0.01"
                                            x-model="emp.allowance" @input="updateCalculations"
                                            class="w-full px-2 py-1.5 text-xs border border-slate-200 rounded focus:outline-none focus:border-blue-500">
                                    </div>
                                    <div>
                                        <label
                                            class="text-[10px] font-bold text-slate-500 uppercase mb-1 block">Other</label>
                                        <input type="number" :name="'other_expense_' + emp.id" step="0.01"
                                            x-model="emp.other_expense" @input="updateCalculations"
                                            class="w-full px-2 py-1.5 text-xs border border-slate-200 rounded focus:outline-none focus:border-blue-500">
                                    </div>
                                </div>
                            </div>
                        </template>

                        <div x-show="selectedEmployees.length === 0"
                            class="text-center py-6 bg-slate-50 border border-dashed border-slate-300 rounded-lg text-slate-400 text-sm">
                            No employees selected. Search above to add.
                        </div>
                    </div>
                </div>
            </div>

            <!-- Real-Time Calculations -->
            <div x-show="selectedEmployees.length > 0" x-cloak
                class="bg-gradient-to-br from-slate-50 to-blue-50 rounded-xl shadow-sm border border-slate-200 p-6">
                <h3 class="text-lg font-bold text-slate-800 mb-4 flex items-center gap-2">
                    <svg class="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M9 7h6m0 10v-3m-3 3h.01M9 17h.01M9 14h.01M12 14h.01M15 11h.01M12 11h.01M9 11h.01M7 21h10a2 2 0 002-2V5a2 2 0 00-2-2H7a2 2 0 00-2 2v14a2 2 0 002 2z">
                        </path>
                    </svg>
                    Financial Summary
                </h3>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div class="bg-white rounded-lg p-4 border border-slate-200 shadow-sm">
                        <div class="text-xs text-slate-500 mb-1 font-medium">Total Expense</div>
                        <div class="text-2xl font-bold text-red-600" x-text="'$' + totalExpense.toFixed(2)"></div>
                        <div class="text-[10px] text-slate-400 mt-1" x-text="selectedEmployees.length + ' employee(s)'">
                        </div>
                    </div>
                    <div class="bg-white rounded-lg p-4 border border-slate-200 shadow-sm">
                        <div class="text-xs text-slate-500 mb-1 font-medium">Total Income</div>
                        <div class="text-2xl font-bold text-green-600" x-text="'$' + totalIncome.toFixed(2)"></div>
                        <div class="text-[10px] text-slate-400 mt-1" x-text="calculateHours().toFixed(2) + ' hours'">
                        </div>
                    </div>
                    <div class="bg-white rounded-lg p-4 border border-slate-200 shadow-sm">
                        <div class="text-xs text-slate-500 mb-1 font-medium">Margin</div>
                        <div class="text-2xl font-bold" :class="margin >= 0 ? 'text-green-600' : 'text-red-600'"
                            x-text="'$' + margin.toFixed(2)"></div>
                        <div class="text-[10px] text-slate-400 mt-1" x-text="margin >= 0 ? 'Profit' : 'Loss'"></div>
                    </div>
                </div>
            </div>

            <!-- Other Info -->
            <div class="bg-white rounded-xl shadow-sm border border-slate-200 p-6">
                <h3 class="text-lg font-bold text-slate-800 mb-4 flex items-center gap-2">
                    <span class="w-1 h-6 bg-blue-600 rounded-full"></span>
                    Additional Information
                </h3>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="text-xs font-bold text-slate-500 uppercase mb-1 block">Customer Reference</label>
                        <input type="text" name="customer_reference" x-model="formData.customer_reference"
                            class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                    </div>
                    <div>
                        <label class="text-xs font-bold text-slate-500 uppercase mb-1 block">Comments</label>
                        <textarea name="comments" rows="3" x-model="formData.comments"
                            class="w-full px-3 py-2 text-sm border border-slate-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all"></textarea>
                    </div>
                </div>
            </div>

            <!-- Form Actions -->
            <div class="flex items-center justify-end gap-3 pt-4">
                <a href="{{ route('patroller-schedules.index') }}"
                    class="px-6 py-2.5 rounded-lg border border-slate-200 text-slate-600 font-bold text-sm hover:bg-slate-50 transition-all">
                    Cancel
                </a>
                <button type="submit" :disabled="!isValid"
                    class="px-6 py-2.5 bg-blue-600 text-white rounded-lg font-bold text-sm hover:bg-blue-700 transition-all shadow-md disabled:opacity-50 disabled:cursor-not-allowed flex items-center gap-2">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                    </svg>
                    Update Schedule
                </button>
            </div>
        </form>
        <!-- Available Guards Modal -->
        <template x-teleport="body">
            <div class="relative z-[100]" aria-labelledby="modal-title" role="dialog" aria-modal="true"
                x-show="showAvailableGuardsModal" x-cloak>

                <!-- Background backdrop -->
                <div x-show="showAvailableGuardsModal" x-transition:enter="ease-out duration-300"
                    x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100"
                    x-transition:leave="ease-in duration-200" x-transition:leave-start="opacity-100"
                    x-transition:leave-end="opacity-0"
                    class="fixed inset-0 bg-slate-900/75 transition-opacity backdrop-blur-sm"></div>

                <div class="fixed inset-0 z-10 w-screen overflow-y-auto">
                    <div class="flex min-h-full items-end justify-center p-4 text-center sm:items-center sm:p-0">

                        <!-- Modal panel -->
                        <div x-show="showAvailableGuardsModal" @click.away="showAvailableGuardsModal = false"
                            x-transition:enter="ease-out duration-300"
                            x-transition:enter-start="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95"
                            x-transition:enter-end="opacity-100 translate-y-0 sm:scale-100"
                            x-transition:leave="ease-in duration-200"
                            x-transition:leave-start="opacity-100 translate-y-0 sm:scale-100"
                            x-transition:leave-end="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95"
                            class="relative transform overflow-hidden rounded-2xl bg-white text-left shadow-xl transition-all sm:my-8 sm:w-full sm:max-w-lg">

                            <!-- Modal Header -->
                            <div class="bg-white px-4 pt-5 pb-4 sm:p-6 border-b border-slate-100">
                                <div class="flex items-center justify-between">
                                    <div class="flex items-center gap-3">
                                        <div
                                            class="mx-auto flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-xl bg-emerald-100 sm:mx-0">
                                            <svg class="h-6 w-6 text-emerald-600" fill="none" viewBox="0 0 24 24"
                                                stroke-width="1.5" stroke="currentColor" aria-hidden="true">
                                                <path stroke-linecap="round" stroke-linejoin="round"
                                                    d="M9 12.75L11.25 15 15 9.75M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                            </svg>
                                        </div>
                                        <div class="text-left">
                                            <h3 class="text-lg font-bold leading-6 text-slate-800" id="modal-title">
                                                Available Guards</h3>
                                            <p class="text-xs text-slate-500 mt-0.5">System checked availability & conflicts
                                            </p>
                                        </div>
                                    </div>
                                    <button @click="showAvailableGuardsModal = false" type="button"
                                        class="rounded-md bg-white text-slate-400 hover:text-slate-500 focus:outline-none">
                                        <span class="sr-only">Close</span>
                                        <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke-width="1.5"
                                            stroke="currentColor" aria-hidden="true">
                                            <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" />
                                        </svg>
                                    </button>
                                </div>
                            </div>

                            <!-- Modal Body -->
                            <div class="px-4 py-4 sm:p-6 max-h-[60vh] overflow-y-auto custom-scrollbar">
                                <div x-show="searchingGuards" class="py-12 text-center">
                                    <div
                                        class="inline-block animate-spin rounded-full h-8 w-8 border-4 border-emerald-500 border-t-transparent mb-4">
                                    </div>
                                    <p class="text-sm text-slate-600 font-medium">Analyzing guard schedules...</p>
                                </div>

                                <div x-show="!searchingGuards">
                                    <template x-if="availableGuards.length === 0">
                                        <div
                                            class="py-8 text-center bg-slate-50 rounded-xl border border-dashed border-slate-200">
                                            <svg class="w-10 h-10 text-slate-300 mx-auto mb-2" fill="none"
                                                stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                                </path>
                                            </svg>
                                            <p class="text-sm text-slate-500 font-medium">No available guards found.</p>
                                        </div>
                                    </template>

                                    <div class="space-y-3">
                                        <template x-for="guard in availableGuards" :key="guard.id">
                                            <div
                                                class="flex items-center justify-between p-3 bg-white border border-slate-200 rounded-xl hover:border-emerald-500 hover:shadow-md transition-all group">
                                                <div class="flex items-center gap-3">
                                                    <div class="relative flex-shrink-0">
                                                        <img :src="guard.photo_url"
                                                            class="w-10 h-10 rounded-full object-cover border-2 border-slate-100 shadow-sm" />
                                                        <div
                                                            class="absolute -bottom-0.5 -right-0.5 w-3 h-3 bg-emerald-500 border-2 border-white rounded-full">
                                                        </div>
                                                    </div>
                                                    <div class="min-w-0">
                                                        <div class="text-sm font-bold text-slate-800 truncate"
                                                            x-text="guard.name"></div>
                                                        <div class="text-[10px] font-medium text-slate-500"
                                                            x-text="'ID: ' + guard.emp_id"></div>
                                                    </div>
                                                </div>
                                                <button type="button" @click.stop="selectAvailableGuard(guard)"
                                                    :disabled="isGuardSelected(guard)"
                                                    class="flex-shrink-0 px-4 py-1.5 rounded-lg text-xs font-bold transition-all shadow-sm"
                                                    :class="isGuardSelected(guard)
                                                            ? 'bg-slate-100 text-slate-400 cursor-not-allowed border border-slate-200'
                                                            : 'bg-emerald-50 text-emerald-600 border border-emerald-100 hover:bg-emerald-600 hover:text-white hover:border-emerald-600'">
                                                    <span x-text="isGuardSelected(guard) ? 'Added' : 'Select'"></span>
                                                </button>
                                            </div>
                                        </template>
                                    </div>
                                </div>
                            </div>

                            <!-- Modal Footer -->
                            <div
                                class="bg-slate-50 px-4 py-3 sm:flex sm:flex-row-reverse sm:px-6 border-t border-slate-100">
                                <button type="button" @click="showAvailableGuardsModal = false"
                                    class="mt-3 inline-flex w-full justify-center rounded-lg bg-white px-3 py-2 text-sm font-semibold text-slate-900 shadow-sm ring-1 ring-inset ring-slate-300 hover:bg-slate-50 sm:mt-0 sm:w-auto">Close</button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </template>
    </div>

    <script>
        function patrollerScheduleForm() {
            return {
                formData: {
                    scheduled_date: '{{ $patrollerSchedule->scheduled_date->format("Y-m-d") }}',
                    route_id: '{{ $patrollerSchedule->route_id }}',
                    route_rate: '{{ $patrollerSchedule->route_rate }}',
                    from_time: '{{ $patrollerSchedule->from_time->format("Y-m-d\TH:i") }}',
                    to_time: '{{ $patrollerSchedule->to_time->format("Y-m-d\TH:i") }}',
                    customer_reference: '{{ $patrollerSchedule->customer_reference }}',
                    comments: '{{ $patrollerSchedule->comments }}'
                },
                availableGuards: [],
                searchingGuards: false,
                showAvailableGuardsModal: false,
                selectedEmployees: [
                    @foreach($patrollerSchedule->employees as $emp)
                                                                                                                    @php
                                                                                                                        $wageTypesData = json_decode($emp->pivot->wage_types ?? '[]', true);
                                                                                                                        $empWageTypeIds = [];
                                                                                                                        $wageRates = [];
                                                                                                                        $allocatedHours = [];

                                                                                                                        if (is_array($wageTypesData) && count($wageTypesData) > 0) {
                                                                                                                            foreach ($wageTypesData as $wt) {
                                                                                                                                if (isset($wt['id'])) {
                                                                                                                                    $empWageTypeIds[] = (string) $wt['id'];
                                                                                                                                    $wageRates[(string) $wt['id']] = $wt['rate'] ?? 0;
                                                                                                                                    $allocatedHours[(string) $wt['id']] = $wt['allocated_hours'] ?? 0;
                                                                                                                                }
                                                                                                                            }
                                                                                                                        }
                                                                                                                    @endphp
                                                                                                                    {
                            id: '{{ $emp->id }}',
                            name: '{{ $emp->first_name }} {{ $emp->last_name }}',
                            emp_id: '{{ $emp->employee_id ?? "ID:" . $emp->id }}',
                            wage_types: @json($empWageTypeIds),
                            wageTypeDetails: [],
                            wageRates: @json($wageRates),
                            allocatedHours: @json($allocatedHours),
                            gas_rate: {{ $emp->pivot->gas_rate ?? 0 }},
                            allowance: {{ $emp->pivot->allowance ?? 0 }},
                            other_expense: {{ $emp->pivot->other_expense ?? 0 }},
                            unavailable_days: []
                        },
                    @endforeach
                                                                    ],
                wageTypes: @json($wageTypes),
                routeDetails: null,
                employeeValidations: {},
                maxShiftHours: {{ $maxShiftHours }},
                isUserRole: {{ auth()->user()->role === 'user' ? 'true' : 'false' }},

                init() {
                    // Fetch route details for the current route
                    this.fetchRouteDetails('{{ $patrollerSchedule->route_id }}');

                    // Fetch details for each employee
                    this.selectedEmployees.forEach(emp => {
                        this.fetchEmployeeWageTypes(emp.id);
                        this.checkEmployeeAvailability(emp.id);
                    });
                },

                async handleRouteSelect(routeId, defaultRate) {
                    this.formData.route_id = routeId;
                    if (defaultRate && !this.formData.route_rate) {
                        this.formData.route_rate = defaultRate;
                    }

                    // Fetch route details
                    await this.fetchRouteDetails(routeId);

                    // Recheck all employees for banned sites
                    this.recheckAllEmployees();

                    this.updateCalculations();
                },

                async fetchRouteDetails(routeId) {
                    try {
                        const response = await fetch(`/patroller-schedules/api/route-details/${routeId}`);
                        const data = await response.json();
                        this.routeDetails = data;
                    } catch (error) {
                        console.error('Error fetching route details:', error);
                    }
                },

                async findAvailableGuards() {
                    if (!this.formData.scheduled_date || !this.formData.from_time || !this.formData.to_time) {
                        alert('Please select Date, From Time and To Time first.');
                        return;
                    }

                    console.log('Starting findAvailableGuards search...');
                    this.searchingGuards = true;
                    this.showAvailableGuardsModal = true;
                    this.availableGuards = [];

                    try {
                        const response = await fetch('/patroller-schedules/api/search-available-employees', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content
                            },
                            body: JSON.stringify({
                                scheduled_date: this.formData.scheduled_date,
                                from_time: this.formData.from_time,
                                to_time: this.formData.to_time
                            })
                        });

                        if (!response.ok) {
                            const errorData = await response.json().catch(() => ({}));
                            throw new Error(errorData.error || 'Failed to fetch available guards');
                        }

                        const rawGuards = await response.json();
                        this.availableGuards = rawGuards.map(g => ({
                            ...g,
                            id: String(g.id),
                            added: this.selectedEmployees.some(e => String(e.id) === String(g.id))
                        }));
                        console.log('Found available guards:', this.availableGuards.length);
                    } catch (error) {
                        console.error('Error finding available guards:', error);
                        alert('Error: ' + error.message);
                    } finally {
                        this.searchingGuards = false;
                    }
                },

                selectAvailableGuard(guard) {
                    if (this.isGuardSelected(guard)) return;

                    try {
                        this.addEmployee({
                            id: String(guard.id),
                            name: guard.name,
                            emp_id: guard.emp_id
                        });

                        const idx = this.availableGuards.findIndex(g => String(g.id) === String(guard.id));
                        if (idx !== -1) {
                            this.availableGuards[idx].added = true;
                        }
                    } catch (e) {
                        console.error('Error selecting guard:', e);
                    }
                },

                isGuardSelected(guard) {
                    return guard.added || this.selectedEmployees.some(e => String(e.id) === String(guard.id));
                },

                async addEmployee(employee) {
                    const empIdVal = String(employee.id);
                    // prevent duplicate adds
                    if (this.selectedEmployees.find(e => String(e.id) === empIdVal)) {
                        return;
                    }

                    this.selectedEmployees.push({
                        ...employee,
                        id: empIdVal,
                        wage_types: [],
                        wageTypeDetails: [],
                        wageRates: {},
                        allocatedHours: {},
                        gas_rate: 0,
                        allowance: 0,
                        other_expense: 0
                    });

                    // Fetch employee wage types
                    await this.fetchEmployeeWageTypes(empIdVal);

                    // Check employee availability
                    await this.checkEmployeeAvailability(employee.id);

                    this.updateCalculations();
                },

                async fetchEmployeeWageTypes(employeeId) {
                    try {
                        const response = await fetch(`/patroller-schedules/api/employee-wage-types/${employeeId}`);
                        const data = await response.json();

                        const emp = this.selectedEmployees.find(e => e.id == employeeId);
                        if (emp && data.wage_types) {
                            // Store all wage type details for reference
                            emp.wageTypeDetails = data.wage_types;
                            emp.unavailable_days = data.employee.unavailable_days || [];
                            this.updateCalculations();

                            // If employee doesn't have wage types yet (new addition), auto-select SIN
                            if (!emp.wage_types || emp.wage_types.length === 0) {
                                const sinWageType = data.wage_types.find(wt =>
                                    wt.name.toLowerCase().includes('sin')
                                );

                                const defaultWageType = sinWageType || data.wage_types[0];

                                if (defaultWageType) {
                                    emp.wage_types = [defaultWageType.id.toString()];
                                    emp.wageRates = {};
                                    emp.wageRates[defaultWageType.id.toString()] = parseFloat(defaultWageType.rate) || 0;
                                    emp.allocatedHours = {};
                                    emp.allocatedHours[defaultWageType.id.toString()] = '';
                                }
                            }
                        }
                    } catch (error) {
                        console.error('Error fetching employee wage types:', error);
                    }
                },

                async checkEmployeeAvailability(employeeId) {
                    if (!this.formData.route_id) {
                        return;
                    }

                    try {
                        const requestData = {
                            employee_id: employeeId,
                            route_id: this.formData.route_id,
                            from_time: this.formData.from_time || null,
                            to_time: this.formData.to_time || null
                        };

                        const response = await fetch('/patroller-schedules/api/check-employee-availability', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            body: JSON.stringify(requestData)
                        });

                        if (!response.ok) {
                            return;
                        }

                        const data = await response.json();
                        this.employeeValidations[employeeId] = data;

                        if (data.employee && data.employee.unavailable_days) {
                            const emp = this.selectedEmployees.find(e => e.id == employeeId);
                            if (emp) {
                                emp.unavailable_days = data.employee.unavailable_days;
                                this.updateCalculations();
                            }
                        }
                    } catch (error) {
                        console.error('Error checking employee availability:', error);
                    }
                },

                async recheckAllEmployees() {
                    for (const emp of this.selectedEmployees) {
                        await this.checkEmployeeAvailability(emp.id);
                    }
                },

                removeEmployee(index) {
                    const emp = this.selectedEmployees[index];
                    delete this.employeeValidations[emp.id];
                    this.selectedEmployees.splice(index, 1);
                    this.updateCalculations();
                },

                toggleWageType(empIndex, wageTypeId) {
                    const emp = this.selectedEmployees[empIndex];
                    if (emp.wage_types.includes(wageTypeId)) {
                        emp.wage_types = emp.wage_types.filter(id => id !== wageTypeId);
                        delete emp.wageRates[wageTypeId];
                        delete emp.allocatedHours[wageTypeId];
                    } else {
                        emp.wage_types.push(wageTypeId);
                        const wageTypeDetail = emp.wageTypeDetails.find(wt => wt.id.toString() === wageTypeId.toString());
                        if (wageTypeDetail) {
                            emp.wageRates[wageTypeId] = parseFloat(wageTypeDetail.rate) || 0;
                        } else {
                            emp.wageRates[wageTypeId] = 0;
                        }
                        emp.allocatedHours[wageTypeId] = '';
                    }
                    this.updateCalculations();
                },

                removeWageType(empIndex, wageTypeId) {
                    const emp = this.selectedEmployees[empIndex];
                    emp.wage_types = emp.wage_types.filter(id => id !== wageTypeId);
                    delete emp.wageRates[wageTypeId];
                    delete emp.allocatedHours[wageTypeId];
                    this.updateCalculations();
                },

                getWageTypeName(id) {
                    const type = this.wageTypes.find(wt => wt.id == id);
                    return type ? type.name : 'Unknown';
                },

                getAvailableDays(unavailableDays) {
                    const allDays = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
                    if (!unavailableDays) return allDays;
                    return allDays.filter(day => !unavailableDays.includes(day));
                },

                updateCalculations() {
                    this.$nextTick(() => {
                        this.selectedEmployees = [...this.selectedEmployees];
                    });
                },

                calculateHours() {
                    if (!this.formData.from_time || !this.formData.to_time) return 0;
                    const from = new Date(this.formData.from_time);
                    const to = new Date(this.formData.to_time);
                    return (to - from) / (1000 * 60 * 60);
                },

                formatDuration(hours) {
                    if (!hours || hours <= 0) return '0h 0m';
                    const h = Math.floor(hours);
                    const m = Math.round((hours - h) * 60);
                    return `${h}h ${m}m`;
                },

                get totalExpense() {
                    let total = 0;

                    this.selectedEmployees.forEach(emp => {
                        if (emp.wageRates && emp.allocatedHours) {
                            Object.keys(emp.wageRates).forEach(wtId => {
                                const rate = parseFloat(emp.wageRates[wtId] || 0);
                                const hours = parseFloat(emp.allocatedHours[wtId] || 0);
                                total += rate * hours;
                            });
                        }
                        total += parseFloat(emp.gas_rate || 0);
                        total += parseFloat(emp.allowance || 0);
                        total += parseFloat(emp.other_expense || 0);
                    });
                    return total;
                },

                get totalIncome() {
                    if (!this.formData.route_rate || !this.formData.from_time || !this.formData.to_time) {
                        return 0;
                    }
                    const hours = this.calculateHours();
                    return hours * parseFloat(this.formData.route_rate || 0);
                },

                get margin() {
                    return this.totalIncome - this.totalExpense;
                },

                get isTimeValid() {
                    if (!this.formData.from_time || !this.formData.to_time) return true;
                    return new Date(this.formData.to_time) > new Date(this.formData.from_time);
                },

                get isValid() {
                    return this.formData.scheduled_date &&
                        this.formData.route_id &&
                        this.formData.route_rate &&
                        this.isTimeValid &&
                        this.selectedEmployees.length > 0 &&
                        !(this.isUserRole && this.calculateHours() > this.maxShiftHours);
                }
            }
        }
    </script>
@endsection