@extends('layouts.meeting')

@section('title', $meeting->title)

@section('content')
    <div x-data="meetingRoom" class="h-screen flex flex-col overflow-hidden">
        <!-- Top Header -->
        <header class="h-10 px-6 glass flex items-center justify-between z-20">
            <div class="flex items-center gap-4">
                <div class="flex items-center gap-2">
                    <div
                        class="w-7 h-7 bg-blue-600 rounded-lg flex items-center justify-center font-bold text-white shadow-lg text-xs">
                        M</div>
                    <div>
                        <h1 class="text-xs font-bold tracking-tight leading-none">{{ $meeting->title }}</h1>
                        <p class="text-[7px] text-gray-500 uppercase tracking-widest font-black mt-0.5">Room:
                            {{ $meeting->uuid }}
                        </p>
                    </div>
                </div>
            </div>

            <div class="flex items-center gap-3">
                <div class="flex -space-x-1 mr-2">
                    <template x-for="p in participants" :key="p.id">
                        <div class="w-6 h-6 rounded-full border border-slate-900 bg-blue-500 flex items-center justify-center text-[8px] font-bold"
                            :title="p.name">
                            <span x-text="p.name.charAt(0)"></span>
                        </div>
                    </template>
                </div>

                <button @click="toggleSidebar('chat')" class="p-2 hover:bg-white/10 rounded-lg transition-colors relative">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z">
                        </path>
                    </svg>
                    <div x-show="newMessages" class="absolute top-1 right-1 w-2 h-2 bg-blue-500 rounded-full"></div>
                </button>
                <button @click="toggleSidebar('participants')"
                    class="p-2 hover:bg-white/10 rounded-lg transition-colors relative">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z">
                        </path>
                    </svg>
                    <div x-show="waitingRoom.length > 0"
                        class="absolute -top-1 -right-1 min-w-[18px] h-[18px] bg-red-500 text-white text-[9px] font-black rounded-full flex items-center justify-center px-1 border-2 border-slate-900 shadow-lg"
                        x-text="waitingRoom.length"></div>
                </button>
            </div>
        </header>

        <div class="flex-1 flex overflow-hidden relative">
            <!-- Main Video Grid -->
            <div class="flex-1 bg-slate-950 overflow-y-auto p-4 meeting-grid"
                :style="`grid-template-columns: repeat(${gridCols}, 1fr);`" id="video-grid">

                <!-- Local Video -->
                <div class="participant-card active-speaker" id="local-video-container">
                    <video id="local-video" class="video-feed" autoplay muted playsinline></video>
                    <div class="participant-name">You ({{ Auth::user()->name }})</div>

                    <div x-show="!micActive" class="absolute top-4 right-4 bg-red-500/80 p-1.5 rounded-full">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M19 11a7 7 0 01-7 7m0 0a7 7 0 01-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z">
                            </path>
                            <line x1="1" y1="1" x2="23" y2="23" stroke="currentColor" stroke-width="2" />
                        </svg>
                    </div>
                </div>

                <!-- Remote Videos -->
                <template x-for="p in participants" :key="p.id">
                    <div class="participant-card">
                        <!-- Placeholder for video feed until signaling is fully active -->
                        <div class="w-full h-full bg-slate-800 flex flex-col items-center justify-center gap-4">
                            <div
                                class="w-20 h-20 rounded-full bg-blue-600/20 text-blue-500 flex items-center justify-center text-3xl font-black border border-blue-500/30">
                                <span x-text="p.name.charAt(0)"></span>
                            </div>
                            <div class="text-xs font-bold text-blue-400/50 tracking-widest uppercase">Connecting Feed...
                            </div>
                        </div>
                        <div class="participant-name" x-text="p.name"></div>
                    </div>
                </template>
            </div>

            <!-- Right Sidebar (Chat/Participants) -->
            <aside x-show="sidebarOpen" x-transition:enter="transition ease-out duration-300 transform"
                x-transition:enter-start="translate-x-full" x-transition:enter-end="translate-x-0"
                x-transition:leave="transition ease-in duration-200 transform" x-transition:leave-start="translate-x-0"
                x-transition:leave-end="translate-x-full"
                class="w-full md:w-64 lg:w-72 glass h-full z-10 flex flex-col border-l border-white/5">

                <div class="p-3 px-4 border-b border-white/5 flex items-center justify-between bg-slate-900/50">
                    <h2 class="font-black uppercase tracking-[0.15em] text-[9px] text-blue-400"
                        x-text="sidebarTab === 'chat' ? 'Discussion Forum' : (hostMode ? 'Approval Queue' : 'Participants')">
                    </h2>
                    <button @click="sidebarOpen = false" class="p-1 hover:bg-white/10 rounded-md">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                            </path>
                        </svg>
                    </button>
                </div>

                <!-- Chat Tab -->
                <div x-show="sidebarTab === 'chat'" class="flex-1 flex flex-col overflow-hidden">
                    <div class="flex-1 overflow-y-auto p-3 space-y-3" id="chat-container">
                        <template x-for="msg in messages" :key="msg.id">
                            <div
                                :class="msg.user_id == {{ Auth::id() }} ? 'flex flex-col items-end' : 'flex flex-col items-start'">
                                <div class="flex items-center gap-2 mb-1">
                                    <span class="text-[10px] font-bold text-gray-500" x-text="msg.user.name"></span>
                                    <span class="text-[8px] text-gray-600" x-text="formatTime(msg.created_at)"></span>
                                </div>

                                <div :class="msg.user_id == {{ Auth::id() }} ? 'bg-blue-600 text-white rounded-2xl rounded-tr-none' : 'bg-slate-800 text-gray-200 rounded-2xl rounded-tl-none'"
                                    class="px-3 py-1.5 text-sm shadow-sm max-w-[88%]">
                                    <template x-if="msg.type === 'text'">
                                        <p x-text="msg.message"></p>
                                    </template>
                                    <template x-if="msg.type === 'image'">
                                        <div class="space-y-2">
                                            <img :src="'/storage/' + msg.file_path"
                                                class="rounded-lg max-h-48 w-auto cursor-pointer"
                                                @click="window.open('/storage/' + msg.file_path)">
                                            <p x-show="msg.message" x-text="msg.message" class="text-xs"></p>
                                        </div>
                                    </template>
                                    <template x-if="msg.type === 'file'">
                                        <a :href="'/storage/' + msg.file_path" target="_blank"
                                            class="flex items-center gap-2 hover:underline">
                                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z">
                                                </path>
                                            </svg>
                                            <span x-text="msg.file_path.split('/').pop()" class="truncate text-xs"></span>
                                        </a>
                                    </template>
                                </div>
                            </div>
                        </template>
                    </div>

                    <div class="p-3 bg-slate-900/50 border-t border-white/5">
                        <div class="flex items-end gap-2 bg-slate-800 rounded-2xl p-2 border border-white/10">
                            <label class="p-2 hover:bg-white/10 rounded-xl cursor-pointer">
                                <input type="file" @change="handleFileUpload" class="hidden"
                                    accept="image/*,.pdf,.doc,.docx">
                                <svg class="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13">
                                    </path>
                                </svg>
                            </label>
                            <textarea x-model="newMessage" @keydown.enter.prevent="sendMessage()"
                                class="flex-1 bg-transparent border-none focus:ring-0 text-sm py-2 resize-none max-h-32"
                                placeholder="Type a message..." rows="1"></textarea>
                            <button @click="sendMessage()"
                                class="p-2 bg-blue-600 hover:bg-blue-700 rounded-xl transition-colors shadow-lg shadow-blue-500/20">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M5 13l4 4L19 7">
                                    </path>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Participants Tab -->
                <div x-show="sidebarTab === 'participants'" class="flex-1 overflow-y-auto">
                    <!-- Host Waiting List -->
                    <template x-if="hostMode && waitingRoom.length > 0">
                        <div class="p-4 space-y-3">
                            <h3
                                class="text-[10px] font-bold text-blue-500 uppercase tracking-widest border-b border-blue-500/20 pb-2">
                                Waiting Requests</h3>
                            <template x-for="req in waitingRoom" :key="req.user.id">
                                <div
                                    class="bg-blue-600/10 border border-blue-500/20 p-3 rounded-xl flex items-center justify-between">
                                    <div class="flex items-center gap-3">
                                        <div class="w-8 h-8 rounded-full bg-blue-500 flex items-center justify-center font-bold text-xs"
                                            x-text="req.user.name.charAt(0)"></div>
                                        <div class="text-xs font-bold" x-text="req.user.name"></div>
                                    </div>
                                    <div class="flex gap-2">
                                        <button @click="approveUser(req.user.id)"
                                            class="p-1.5 bg-blue-600 hover:bg-blue-700 rounded-lg shadow-lg">
                                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M5 13l4 4L19 7"></path>
                                            </svg>
                                        </button>
                                        <button class="p-1.5 bg-red-600 hover:bg-red-700 rounded-lg">
                                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M6 18L18 6M6 6l12 12"></path>
                                            </svg>
                                        </button>
                                    </div>
                                </div>
                            </template>
                        </div>
                    </template>

                    <div class="p-3 space-y-1">
                        <h3 class="text-[9px] font-black text-gray-500 uppercase tracking-widest pb-2 px-1">Meeting Members
                            (<span x-text="participants.length + 1"></span>)</h3>

                        <!-- Current User (You) -->
                        <div
                            class="flex items-center justify-between p-2 bg-blue-500/5 rounded-lg border border-blue-500/10 mb-1">
                            <div class="flex items-center gap-3">
                                <div
                                    class="w-8 h-8 rounded-full bg-blue-600 flex items-center justify-center font-bold text-xs text-white">
                                    {{ Auth::user()->name[0] }}
                                </div>
                                <div class="text-xs font-bold text-blue-400">You ({{ Auth::user()->name }})</div>
                            </div>
                            <div class="flex gap-2">
                                <div class="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                            </div>
                        </div>

                        <!-- Other participants -->
                        <template x-for="p in participants" :key="p.id">
                            <div class="flex items-center justify-between p-2 hover:bg-white/5 rounded-lg">
                                <div class="flex items-center gap-3">
                                    <div class="w-8 h-8 rounded-full bg-slate-700 flex items-center justify-center font-bold text-xs"
                                        x-text="p.name.charAt(0)"></div>
                                    <div class="text-xs font-medium" x-text="p.name"></div>
                                </div>
                                <div class="flex gap-2">
                                    <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 11a7 7 0 01-7 7m0 0a7 7 0 01-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z">
                                        </path>
                                    </svg>
                                </div>
                            </div>
                        </template>
                    </div>
                </div>
            </aside>
        </div>

        <!-- Floating Controls -->
        <div class="h-10 glass border-t border-white/5 flex items-center justify-center gap-4 relative z-20">
            <button @click="toggleMic()" :class="micActive ? 'control-btn active' : 'control-btn danger'" class="shadow-xl">
                <svg x-show="micActive" class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M19 11a7 7 0 01-7 7m0 0a7 7 0 01-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z">
                    </path>
                </svg>
                <svg x-show="!micActive" class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M19 11a7 7 0 01-7 7m0 0a7 7 0 01-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z">
                    </path>
                    <line x1="1" y1="1" x2="23" y2="23" stroke="currentColor" stroke-width="2" />
                </svg>
            </button>
            <button @click="toggleCamera()" :class="cameraActive ? 'control-btn active' : 'control-btn danger'"
                class="shadow-xl">
                <svg x-show="cameraActive" class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z">
                    </path>
                </svg>
                <svg x-show="!cameraActive" class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z">
                    </path>
                    <line x1="1" y1="1" x2="23" y2="23" stroke="currentColor" stroke-width="2" />
                </svg>
            </button>
            <button @click="toggleScreenShare()" :class="screenSharing ? 'control-btn active' : 'control-btn'"
                class="shadow-xl">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M9.75 17L9 20l-1 1h8l-1-1-.75-3M3 13h18M5 17h14a2 2 0 002-2V5a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z">
                    </path>
                </svg>
            </button>

            <form action="{{ route('meetings.end', $meeting->uuid) }}" method="POST">
                @csrf
                <button type="submit" class="control-btn danger shadow-xl" title="Leave/End Meeting">
                    <svg class="w-5 h-5 rotate-[135deg]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z">
                        </path>
                    </svg>
                </button>
            </form>
        </div>

        <!-- Notification Toast -->
        <div x-show="toast.show" x-cloak x-transition:enter="transition ease-out duration-300"
            x-transition:enter-start="opacity-0 translate-y-4" x-transition:enter-end="opacity-100 translate-y-0"
            class="fixed bottom-24 left-6 z-50 bg-blue-600 text-white px-6 py-3 rounded-2xl shadow-2xl flex items-center gap-4">
            <span x-text="toast.message"></span>
            <button @click="toast.show = false" class="hover:text-white/80"><svg class="w-4 h-4" fill="none"
                    stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg></button>
        </div>
    </div>
@endsection

@push('scripts')
    <script>
            document.addEventListener('alpin        e:init', () => {
                Alpine.data('meetingRoom', () => ({
                    meetingUuid: '{{ $meeting->uuid }}',
                    hostMode: {{ $meeting->creator_id === Auth::id() ? 'true' : 'false' }},
                    participants: [],
                    messages: @json($meeting->messages),
                    waitingRoom: @json($pendingParticipants ?? []),
                    newMessage: '',
                    sidebarOpen: false,
                    sidebarTab: 'chat',
                    newMessages: false,
                    micActive: true,
                    cameraActive: true,
                    screenSharing: false,
                    localStream: null,
                    toast: { show: false, message: '' },
                    gridCols: 1,

                    init() {
                        this.setupWebcam();
                        this.listenToChannels();
                        this.updateGrid();
                    },

                    async setupWebcam() {
                        try {
                            this.localStream = await navigator.mediaDevices.getUserMedia({
                                video: true,
                                audio: true
                            });
                            document.getElementById('local-video').srcObject = this.localStream;
                        } catch (err) {
                            console.error("Error accessing camera:", err);
                        }
                    },

                    listenToChannels() {
                        console.log("Connecting to Presence Channel: meeting." + this.meetingUuid);
                        const currentUserId = String({{ Auth::id() }});

                        // Presence Channel for Meeting
                        window.Echo.join(`meeting.${this.meetingUuid}`)
                            .here((users) => {
                                console.log("Participants here:", users);
                                this.participants = users.filter(u => String(u.id) !== currentUserId);
                                this.updateGrid();
                            })
                            .joining((user) => {
                                console.log("User joined:", user);
                                if (String(user.id) !== currentUserId) {
                                    if (!this.participants.find(p => String(p.id) === String(user.id))) {
                                        this.participants = [...this.participants, user];
                                        this.showToast(`${user.name} joined`);
                                        this.updateGrid();
                                    }
                                }
                            })
                            .leaving((user) => {
                                console.log("User left:", user);
                                this.participants = this.participants.filter(u => String(u.id) !== String(user.id));
                                this.updateGrid();
                            })
                            .listen('.message.sent', (data) => {
                                this.messages = [...this.messages, data.message];
                                if (!this.sidebarOpen || this.sidebarTab !== 'chat') {
                                    this.newMessages = true;
                                }
                                this.$nextTick(() => {
                                    const container = document.getElementById('chat-container');
                                    if (container) container.scrollTop = container.scrollHeight;
                                });
                            })
                            .listen('.meeting.ended', () => {
                                alert('The host has ended this meeting.');
                                window.location.href = '{{ route("meetings.index") }}';
                            });

                        // Waiting Room Channel (for host)
                        if (this.hostMode) {
                            window.Echo.private(`waiting-room.${this.meetingUuid}`)
                                .listen('.join.requested', (data) => {
                                    this.waitingRoom.push(data);
                                    this.showToast(`${data.user.name} is waiting to join.`);
                                    this.sidebarOpen = true;
                                    this.sidebarTab = 'participants';
                                });
                        }

                        // User's private channel for approvals
                        @if(Auth::check())
                            window.Echo.private('user.{{ Auth::id() }}')
                                .listen('.join.approved', (data) => {
                                    if (data.meeting.uuid === this.meetingUuid) {
                                        window.location.reload();
                                    }
                                });
                        @endif
                                                            },

                    async sendMessage() {
                        if (!this.newMessage.trim()) return;

                        const content = this.newMessage;
                        this.newMessage = '';

                        try {
                            const response = await fetch(`{{ route('meetings.message', $meeting->uuid) }}`, {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                },
                                body: JSON.stringify({ message: content })
                            });
                            const data = await response.json();
                            this.messages.push(data.message);
                            this.$nextTick(() => {
                                const container = document.getElementById('chat-container');
                                container.scrollTop = container.scrollHeight;
                            });
                        } catch (err) {
                            console.error("Failed to send message:", err);
                        }
                    },

                    async handleFileUpload(e) {
                        const file = e.target.files[0];
                        if (!file) return;

                        const formData = new FormData();
                        formData.append('file', file);

                        try {
                            const response = await fetch(`{{ route('meetings.message', $meeting->uuid) }}`, {
                                method: 'POST',
                                headers: {
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                },
                                body: formData
                            });
                            const data = await response.json();
                            this.messages.push(data.message);
                            this.showToast('File uploaded successfully');
                        } catch (err) {
                            console.error("Upload failed:", err);
                        }
                    },

                    async approveUser(userId) {
                        try {
                            await fetch(`{{ route('meetings.approve', $meeting->uuid) }}`, {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                },
                                body: JSON.stringify({ user_id: userId })
                            });
                            this.waitingRoom = this.waitingRoom.filter(r => r.user.id !== userId);
                            this.showToast('Participant approved');
                        } catch (err) {
                            console.error("Approval failed:", err);
                        }
                    },

                    toggleSidebar(tab) {
                        if (this.sidebarOpen && this.sidebarTab === tab) {
                            this.sidebarOpen = false;
                        } else {
                            this.sidebarOpen = true;
                            this.sidebarTab = tab;
                            if (tab === 'chat') this.newMessages = false;
                        }
                    },

                    toggleMic() {
                        this.micActive = !this.micActive;
                        const audioTrack = this.localStream.getAudioTracks()[0];
                        if (audioTrack) audioTrack.enabled = this.micActive;
                    },

                    toggleCamera() {
                        this.cameraActive = !this.cameraActive;
                        const videoTrack = this.localStream.getVideoTracks()[0];
                        if (videoTrack) videoTrack.enabled = this.cameraActive;
                    },

                    async toggleScreenShare() {
                        if (!this.screenSharing) {
                            try {
                                const screenStream = await navigator.mediaDevices.getDisplayMedia({ video: true });
                                // Replacing the video track with screen track (logic placeholder)
                                this.screenSharing = true;
                                // In a full WebRTC impl, you'd replace the track for each peer
                                screenStream.getTracks()[0].onended = () => {
                                    this.screenSharing = false;
                                };
                            } catch (err) {
                                console.error("Screen share error:", err);
                            }
                        } else {
                            this.screenSharing = false;
                            // Revert to camera (logic placeholder)
                        }
                    },

                    updateGrid() {
                        const count = this.participants.length + 1; // +1 for local
                        if (count <= 1) this.gridCols = 1;
                        else if (count <= 4) this.gridCols = 2;
                        else this.gridCols = 3;
                    },

                    formatTime(dateStr) {
                        const date = new Date(dateStr);
                        return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
                    },

                    showToast(msg) {
                        this.toast.message = msg;
                        this.toast.show = true;
                        setTimeout(() => { this.toast.show = false; }, 5000);
                    }
                }));
            });
        </script>
@endpush