@extends('layouts.user')

@section('title', 'Employee List')

@section('content')
    @if(session('success'))
        <div
            class="mb-6 mx-6 p-4 bg-emerald-50 border border-emerald-100 text-emerald-700 rounded-2xl flex items-center gap-3 shadow-sm animate-fade-in-down">
            <div class="w-8 h-8 bg-emerald-500 rounded-full flex items-center justify-center text-white shrink-0">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M5 13l4 4L19 7"></path>
                </svg>
            </div>
            <p class="text-xs font-bold uppercase tracking-wider">{{ session('success') }}</p>
        </div>
    @endif

    @if(session('error'))
        <div
            class="mb-6 mx-6 p-4 bg-red-50 border border-red-100 text-red-700 rounded-2xl flex items-center gap-3 shadow-sm animate-fade-in-down">
            <div class="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center text-white shrink-0">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </div>
            <p class="text-xs font-bold uppercase tracking-wider">{{ session('error') }}</p>
        </div>
    @endif

    <!-- Import Loader Overlay -->
    <div id="import-loader"
        class="fixed inset-0 bg-slate-900/50 backdrop-blur-sm z-[9999] hidden items-center justify-center">
        <div class="bg-white p-8 rounded-3xl shadow-2xl flex flex-col items-center gap-4 max-w-xs w-full">
            <div class="relative">
                <div class="w-16 h-16 border-4 border-blue-50/30 border-t-blue-600 rounded-full animate-spin"></div>
                <div class="absolute inset-0 flex items-center justify-center">
                    <svg class="w-6 h-6 text-blue-600 animate-pulse" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12">
                        </path>
                    </svg>
                </div>
            </div>
            <div class="text-center">
                <h3 class="text-lg font-bold text-slate-800">Importing Employees</h3>
                <p class="text-sm text-slate-500 mt-1">Please wait while we process the CSV file...</p>
            </div>
        </div>
    </div>

    <div class="flex flex-col h-full">
        <!-- Toolbar & Filters -->
        <div class="bg-white border-b border-gray-100 px-6 py-4">
            <form id="search-form" action="{{ route('employees.index') }}" method="GET"
                class="flex flex-col md:flex-row md:items-center justify-between gap-4">
                <div class="flex flex-wrap items-center gap-2">
                    <a href="{{ route('employees.create') }}"
                        class="px-4 py-2 bg-blue-600 text-white font-bold rounded-xl hover:bg-blue-700 transition-all shadow-md shadow-blue-500/20 flex items-center gap-2 text-xs uppercase tracking-wider">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                        </svg>
                        Add Employee
                    </a>

                    <button type="button" onclick="openBulkUpdateModal()" id="bulk-update-btn" style="display: none;"
                        class="px-4 py-2 bg-purple-600 text-white font-bold rounded-xl hover:bg-purple-700 transition-all shadow-md shadow-purple-500/20 flex items-center gap-2 text-xs uppercase tracking-wider">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z">
                            </path>
                        </svg>
                        Bulk Update <span id="selected-count"
                            class="bg-purple-800 text-white text-[10px] px-1.5 py-0.5 rounded-md ml-1">0</span>
                    </button>

                    <button type="button" onclick="openCopyModal()" id="copy-btn" style="display: none;"
                        class="px-4 py-2 bg-amber-600 text-white font-bold rounded-xl hover:bg-amber-700 transition-all shadow-md shadow-amber-500/20 flex items-center gap-2 text-xs uppercase tracking-wider">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z">
                            </path>
                        </svg>
                        Copy <span id="copy-count"
                            class="bg-amber-800 text-white text-[10px] px-1.5 py-0.5 rounded-md ml-1">0</span>
                    </button>

                    <a href="{{ route('employees.export-excel', request()->all()) }}" target="_blank"
                        class="px-4 py-2 bg-green-600 text-white font-bold rounded-xl hover:bg-green-700 transition-all shadow-md shadow-green-500/20 flex items-center gap-2 text-xs uppercase tracking-wider">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                        </svg>
                        Export Excel
                    </a>

                    <a href="{{ route('employees.export-pdf', request()->all()) }}" target="_blank"
                        class="px-4 py-2 bg-red-600 text-white font-bold rounded-xl hover:bg-red-700 transition-all shadow-md shadow-red-500/20 flex items-center gap-2 text-xs uppercase tracking-wider">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                        </svg>
                        Export PDF
                    </a>

                    <label for="csv-upload"
                        class="px-4 py-2 bg-slate-800 text-white font-bold rounded-xl hover:bg-slate-900 transition-all shadow-md flex items-center gap-2 text-xs uppercase tracking-wider cursor-pointer">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12">
                            </path>
                        </svg>
                        Import CSV
                    </label>

                    <a href="{{ route('employees.import.settings') }}"
                        class="px-4 py-2 bg-indigo-600 text-white font-bold rounded-xl hover:bg-indigo-700 transition-all shadow-md shadow-indigo-500/20 flex items-center gap-2 text-xs uppercase tracking-wider">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z">
                            </path>
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M15 12a3 3 0 11-6 0 3 3 0 016 0z">
                            </path>
                        </svg>
                        Import Settings
                    </a>
                    <span class="text-xs text-slate-500 font-medium ml-2">Total Employees: {{ $employees->total() }}</span>

                    <div class="h-8 w-[1px] bg-gray-100 mx-2 hidden md:block"></div>
                    <div class="flex items-center gap-3">
                        <select name="department_id" id="department-filter"
                            class="px-4 py-2 bg-gray-50 border border-gray-100 rounded-xl text-xs font-bold text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500/20 cursor-pointer">
                            <option value="">All Departments</option>
                            @foreach($departments as $dept)
                                <option value="{{ $dept->id }}" {{ request('department_id') == $dept->id ? 'selected' : '' }}>
                                    {{ $dept->name }}
                                </option>
                            @endforeach
                        </select>
                        <select name="job_role_id" id="job-role-filter"
                            class="px-4 py-2 bg-gray-50 border border-gray-100 rounded-xl text-xs font-bold text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500/20 cursor-pointer">
                            <option value="">All Job Roles</option>
                            @foreach($jobRoles as $role)
                                <option value="{{ $role->id }}" {{ request('job_role_id') == $role->id ? 'selected' : '' }}>
                                    {{ $role->name }}
                                </option>
                            @endforeach
                        </select>
                        <select name="status" id="status-filter"
                            class="px-4 py-2 bg-gray-50 border border-gray-100 rounded-xl text-xs font-bold text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500/20 cursor-pointer">
                            <option value="">All Statuses</option>
                            <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>Active Only</option>
                            <option value="inactive" {{ request('status') === 'inactive' ? 'selected' : '' }}>Inactive Only
                            </option>
                        </select>
                    </div>
                </div>

                <div class="relative max-w-sm w-full group">
                    <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                        <svg class="h-4 w-4 text-gray-400 group-focus-within:text-blue-500 transition-colors" fill="none"
                            stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                        </svg>
                    </div>
                    <input type="text" name="search" id="search-input" value="{{ request('search') }}"
                        placeholder="Search (Name, ID, Phone)..."
                        class="block w-full pl-10 pr-10 py-2.5 bg-gray-50 border border-gray-100 rounded-2xl text-xs focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:bg-white transition-all">
                    
                    <a href="{{ route('employees.index') }}" id="clear-search"
                        class="absolute inset-y-0 right-0 pr-3 flex items-center text-gray-400 hover:text-red-500 transition-colors {{ (request('search') || request('department_id') || request('job_role_id') || request('status')) ? '' : 'hidden' }}"
                        title="Clear Filters">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                            </path>
                        </svg>
                    </a>
                </div>
            </form>
        </div>

        <!-- Hidden Import Form -->
        <form id="csv-import-form" action="{{ route('employees.import-csv') }}" method="POST" enctype="multipart/form-data"
            class="hidden">
            @csrf
            <input type="file" id="csv-upload" name="csv_file" onchange="showLoaderAndSubmit()">
        </form>

        <!-- Table Container -->
        <div id="table-wrapper" class="flex-1 overflow-auto bg-white">
            @include('employees._table')
        </div>
    </div>

    <!-- Bulk Update Modal -->
    <div id="bulk-update-modal"
        class="fixed inset-0 bg-slate-900/50 backdrop-blur-sm z-[9999] hidden items-center justify-center">
        <div class="bg-white w-full max-w-6xl h-[90vh] rounded-2xl shadow-2xl overflow-hidden flex flex-col">
            <div class="flex justify-between items-center p-6 border-b border-gray-100 bg-gray-50">
                <div>
                    <h3 class="text-xl font-bold text-gray-900">Bulk Update Employees</h3>
                    <p class="text-xs text-gray-500 mt-1">Edit values for individual employees below.</p>
                </div>
                <button type="button" onclick="closeBulkUpdateModal()" class="text-gray-400 hover:text-gray-600">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                        </path>
                    </svg>
                </button>
            </div>

            <form action="{{ route('employees.bulk-update') }}" method="POST" class="flex-1 overflow-hidden flex flex-col">
                @csrf
                <div class="flex-1 overflow-y-auto p-6">
                    <table class="min-w-full divide-y divide-gray-200 border-separate border-spacing-0">
                        <thead class="bg-gray-50">
                            <tr>
                                <th
                                    class="px-3 py-2 text-left text-xs font-bold text-gray-500 uppercase tracking-wider sticky top-0 bg-gray-50 z-10">
                                    Employee</th>
                                <th
                                    class="px-3 py-2 text-left text-xs font-bold text-gray-500 uppercase tracking-wider sticky top-0 bg-gray-50 z-10 w-40">
                                    Job Role
                                    <div class="flex items-center gap-1 mt-1">
                                        <button type="button" onclick="copySelectToAll('job_role_id', this)"
                                            class="text-[10px] text-purple-600 hover:text-purple-800 font-bold hover:underline cursor-pointer">
                                            Apply to All
                                        </button>
                                    </div>
                                </th>
                                <th
                                    class="px-3 py-2 text-left text-xs font-bold text-gray-500 uppercase tracking-wider sticky top-0 bg-gray-50 z-10 w-[280px]">
                                    Wage Configuration
                                    <div class="flex items-center gap-1 mt-1">
                                        <button type="button" onclick="copyWagesToAll(this)"
                                            class="text-[10px] text-purple-600 hover:text-purple-800 font-bold hover:underline cursor-pointer">
                                            Apply to All
                                        </button>
                                    </div>
                                </th>
                                <th
                                    class="px-3 py-2 text-left text-xs font-bold text-gray-500 uppercase tracking-wider sticky top-0 bg-gray-50 z-10">
                                    License #</th>
                                <th
                                    class="px-3 py-2 text-left text-xs font-bold text-gray-500 uppercase tracking-wider sticky top-0 bg-gray-50 z-10">
                                    Expiry</th>
                            </tr>
                        </thead>
                        <tbody id="bulk-update-tbody" class="bg-white divide-y divide-gray-100">
                            <!-- Rows will be injected by JS -->
                        </tbody>
                    </table>
                </div>

                <div class="p-6 border-t border-gray-100 bg-gray-50 flex justify-end gap-3 z-20">
                    <button type="button" onclick="closeBulkUpdateModal()"
                        class="px-4 py-2 bg-white text-gray-700 font-bold rounded-xl border border-gray-200 hover:bg-gray-50 transition-all text-sm">Cancel</button>
                    <button type="submit"
                        class="px-6 py-2 bg-purple-600 text-white font-bold rounded-xl hover:bg-purple-700 transition-all shadow-md shadow-purple-500/20 text-sm">Save
                        Changes</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Copy Modal -->
    <div id="copy-modal" class="fixed inset-0 bg-slate-900/50 backdrop-blur-sm z-[9999] hidden items-center justify-center">
        <div class="bg-white p-8 rounded-3xl shadow-2xl flex flex-col gap-6 max-w-md w-full">
            <div>
                <h3 class="text-xl font-bold text-gray-900">Copy Employees</h3>
                <p class="text-sm text-gray-500 mt-1">Select the target company to copy <span id="copy-modal-count"
                        class="font-bold text-gray-800">0</span> employees to.</p>
            </div>

            <form action="{{ route('employees.copy') }}" method="POST" class="flex flex-col gap-4">
                @csrf
                <div id="copy-hidden-inputs"></div>

                <div>
                    <label class="block text-xs font-bold text-gray-700 uppercase tracking-wider mb-2">Target
                        Company</label>
                    <select name="target_company_id" required
                        class="w-full bg-gray-50 border border-gray-200 rounded-xl px-4 py-3 text-sm focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                        <option value="">Select Company</option>
                        @foreach($targetCompanies as $company)
                            <option value="{{ $company->id }}">{{ $company->name }}</option>
                        @endforeach
                    </select>
                </div>

                <div class="flex justify-end gap-3 mt-2">
                    <button type="button" onclick="closeCopyModal()"
                        class="px-5 py-2.5 bg-white text-gray-700 font-bold rounded-xl border border-gray-200 hover:bg-gray-50 transition-all text-sm">Cancel</button>
                    <button type="submit"
                        class="px-5 py-2.5 bg-blue-600 text-white font-bold rounded-xl hover:bg-blue-700 transition-all shadow-md shadow-blue-500/20 text-sm">Copy
                        Employees</button>
                </div>
            </form>
        </div>
    </div>

    @push('scripts')
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                const searchInput = document.getElementById('search-input');
                const deptFilter = document.getElementById('department-filter');
                const roleFilter = document.getElementById('job-role-filter');
                const statusFilter = document.getElementById('status-filter');
                const tableWrapper = document.getElementById('table-wrapper');
                const clearSearch = document.getElementById('clear-search');
                let debounceTimer;

                function fetchEmployees(url = null) {
                    const formData = new FormData(document.getElementById('search-form'));
                    const params = new URLSearchParams(formData);
                    
                    if (!url) {
                        url = "{{ route('employees.index') }}?" + params.toString();
                    } else {
                        const targetUrl = new URL(url);
                        params.forEach((value, key) => {
                            if (!targetUrl.searchParams.has(key)) {
                                targetUrl.searchParams.append(key, value);
                            }
                        });
                        url = targetUrl.toString();
                    }

                    window.history.pushState({}, '', url);

                    fetch(url, {
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest'
                        }
                    })
                    .then(response => response.text())
                    .then(html => {
                        tableWrapper.innerHTML = html;
                        attachPaginationListeners();
                        updateBulkButton(); // Refresh buttons if they were active
                    })
                    .catch(error => console.error('Error fetching employees:', error));
                }

                function attachPaginationListeners() {
                    const paginationLinks = tableWrapper.querySelectorAll('.table-pagination a, .ajax-sort');
                    paginationLinks.forEach(link => {
                        link.addEventListener('click', function(e) {
                            e.preventDefault();
                            fetchEmployees(this.getAttribute('href'));
                            window.scrollTo({ top: 0, behavior: 'smooth' });
                        });
                    });
                }

                function handleInput() {
                    if (searchInput.value || deptFilter.value || roleFilter.value || statusFilter.value) {
                        clearSearch.classList.remove('hidden');
                    } else {
                        clearSearch.classList.add('hidden');
                    }
                }

                searchInput.addEventListener('input', function() {
                    handleInput();
                    clearTimeout(debounceTimer);
                    debounceTimer = setTimeout(() => {
                        fetchEmployees();
                    }, 500);
                });

                [deptFilter, roleFilter, statusFilter].forEach(filter => {
                    filter.addEventListener('change', function() {
                        handleInput();
                        fetchEmployees();
                    });
                });

                clearSearch.addEventListener('click', function(e) {
                    e.preventDefault();
                    searchInput.value = '';
                    deptFilter.value = '';
                    roleFilter.value = '';
                    statusFilter.value = '';
                    this.classList.add('hidden');
                    fetchEmployees();
                });

                window.addEventListener('popstate', function() {
                    location.reload();
                });

                attachPaginationListeners();
            });

            function showLoaderAndSubmit() { const fileInput = document.getElementById('csv-upload'); if (fileInput.files.length > 0) { document.getElementById('import-loader').classList.remove('hidden'); document.getElementById('import-loader').classList.add('flex'); document.getElementById('csv-import-form').submit(); } }
            function toggleSelectAll() { const selectAll = document.getElementById('select-all'); const checkboxes = document.querySelectorAll('.employee-checkbox'); checkboxes.forEach(cb => cb.checked = selectAll.checked); updateBulkButton(); }
            function updateBulkButton() {
                const checked = document.querySelectorAll('.employee-checkbox:checked').length; const btn = document.getElementById('bulk-update-btn'); const copyBtn = document.getElementById('copy-btn'); const countSpan = document.getElementById('selected-count'); const copyCountSpan = document.getElementById('copy-count');
                if (countSpan) countSpan.innerText = checked; if (copyCountSpan) copyCountSpan.innerText = checked;
                if (checked > 0) { btn.classList.remove('hidden'); btn.style.display = 'flex'; if (copyBtn) { copyBtn.classList.remove('hidden'); copyBtn.style.display = 'flex'; } } else { btn.classList.add('hidden'); btn.style.display = 'none'; if (copyBtn) { copyBtn.classList.add('hidden'); copyBtn.style.display = 'none'; } }
            }
            function openCopyModal() {
                const checked = document.querySelectorAll('.employee-checkbox:checked'); if (checked.length === 0) return;
                document.getElementById('copy-modal-count').innerText = checked.length; const container = document.getElementById('copy-hidden-inputs'); container.innerHTML = '';
                checked.forEach(cb => { const input = document.createElement('input'); input.type = 'hidden'; input.name = 'employee_ids[]'; input.value = cb.value; container.appendChild(input); });
                document.getElementById('copy-modal').classList.remove('hidden'); document.getElementById('copy-modal').classList.add('flex');
            }
            function closeCopyModal() { document.getElementById('copy-modal').classList.add('hidden'); document.getElementById('copy-modal').classList.remove('flex'); }
            function copySelectToAll(field, btn) {
                const selects = document.querySelectorAll(`select[name*="[${field}]"]`);
                if (selects.length === 0) return;
                const firstValue = selects[0].value;
                selects.forEach(select => { select.value = firstValue; });
                if (btn) {
                    const originalText = btn.innerText;
                    btn.innerText = 'Applied!';
                    const bgC = btn.style.color;
                    btn.style.color = 'green';
                    setTimeout(() => { btn.innerText = originalText; btn.style.color = bgC; }, 1000);
                }
            }

            function addWageRow(empId, btn) {
                const container = document.getElementById(`wages-container-${empId}`);
                const index = container.children.length;
                const wageTypes = @json($wageTypes);
                let wageTypesOptions = '<option value="">Select Type</option>';
                wageTypes.forEach(w => { wageTypesOptions += `<option value="${w.id}">${w.name}</option>`; });
                const row = document.createElement('div');
                row.className = 'flex gap-2 mb-2 wage-row';
                row.innerHTML = `
                    <select name="employees[${empId}][wages][${index}][type_id]" class="w-32 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none">
                        ${wageTypesOptions}
                    </select>
                    <input type="number" step="0.01" name="employees[${empId}][wages][${index}][rate]" class="w-24 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none" placeholder="Rate">
                    <button type="button" onclick="this.parentElement.remove()" class="text-red-400 hover:text-red-600 px-1">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                    </button>
                `;
                container.appendChild(row);
            }

            function copyWagesToAll(btn) {
                const firstContainer = document.querySelector('[id^="wages-container-"]');
                if (!firstContainer) return;
                const rows = firstContainer.querySelectorAll('.wage-row');
                const templateData = [];
                rows.forEach(row => {
                    const select = row.querySelector('select');
                    const input = row.querySelector('input');
                    if (select.value && input.value) { templateData.push({ type: select.value, rate: input.value }); }
                });
                if (templateData.length === 0) return;
                const allContainers = document.querySelectorAll('[id^="wages-container-"]');
                allContainers.forEach((container, idx) => {
                    if (idx === 0) return;
                    container.innerHTML = '';
                    const empId = container.id.replace('wages-container-', '');
                    templateData.forEach((data, i) => {
                        const wageTypes = @json($wageTypes);
                        let wageTypesOptions = '<option value="">Select Type</option>';
                        wageTypes.forEach(w => {
                            const selected = w.id == data.type ? 'selected' : '';
                            wageTypesOptions += `<option value="${w.id}" ${selected}>${w.name}</option>`;
                        });
                        const row = document.createElement('div');
                        row.className = 'flex gap-2 mb-2 wage-row';
                        row.innerHTML = `
                            <select name="employees[${empId}][wages][${i}][type_id]" class="w-32 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none">
                                ${wageTypesOptions}
                            </select>
                            <input type="number" step="0.01" name="employees[${empId}][wages][${i}][rate]" value="${data.rate}" class="w-24 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none" placeholder="Rate">
                            <button type="button" onclick="this.parentElement.remove()" class="text-red-400 hover:text-red-600 px-1">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                            </button>
                        `;
                        container.appendChild(row);
                    });
                });
                if (btn) {
                    const originalText = btn.innerText;
                    btn.innerText = 'Applied!';
                    const bgC = btn.style.color;
                    btn.style.color = 'green';
                    setTimeout(() => { btn.innerText = originalText; btn.style.color = bgC; }, 1000);
                }
            }

            function openBulkUpdateModal() {
                const checked = document.querySelectorAll('.employee-checkbox:checked');
                const tbody = document.getElementById('bulk-update-tbody');
                tbody.innerHTML = '';
                const jobRoles = @json($jobRoles);
                const wageTypes = @json($wageTypes);

                checked.forEach(cb => {
                    const id = cb.value;
                    const name = cb.dataset.name;
                    const currentJobRole = cb.dataset.jobRole;
                    const licenseNum = cb.dataset.licenseNumber || '';
                    const licenseExp = cb.dataset.licenseExpiry || '';
                    const wagesJson = cb.dataset.wages || '[]';
                    let wages = [];
                    try { wages = JSON.parse(wagesJson); } catch (e) { console.error('Error parsing wages JSON', e); }

                    let rowJobRolesOptions = '<option value="">Select Job Role</option>';
                    jobRoles.forEach(r => {
                        const selected = (currentJobRole && r.id == currentJobRole) ? 'selected' : '';
                        rowJobRolesOptions += `<option value="${r.id}" ${selected}>${r.name}</option>`;
                    });

                    let wagesHtml = `<div id="wages-container-${id}">`;
                    if (wages.length > 0) {
                        wages.forEach((wage, index) => {
                            let wageTypesOptions = '<option value="">Select Type</option>';
                            wageTypes.forEach(w => {
                                const selected = w.id == wage.type_id ? 'selected' : '';
                                wageTypesOptions += `<option value="${w.id}" ${selected}>${w.name}</option>`;
                            });
                            wagesHtml += `
                                <div class="flex gap-2 mb-2 wage-row">
                                    <select name="employees[${id}][wages][${index}][type_id]" class="w-32 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none">
                                        ${wageTypesOptions}
                                    </select>
                                    <input type="number" step="0.01" name="employees[${id}][wages][${index}][rate]" value="${wage.rate}" class="w-24 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none" placeholder="Rate">
                                    <button type="button" onclick="this.parentElement.remove()" class="text-red-400 hover:text-red-600 px-1">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
                                    </button>
                                </div>
                            `;
                        });
                    }
                    wagesHtml += `</div>
                        <button type="button" onclick="addWageRow(${id}, this)" class="text-[10px] text-blue-600 font-bold hover:underline flex items-center gap-1">
                            <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path></svg>
                            Add Another Wage
                        </button>`;

                    const row = `
                        <tr class="hover:bg-gray-50 group align-top">
                            <td class="px-3 py-3 whitespace-nowrap">
                                <span class="text-xs font-bold text-gray-700">${name}</span>
                                <span class="block text-[9px] text-gray-400">ID: ${id}</span>
                            </td>
                            <td class="px-3 py-3 whitespace-nowrap">
                                <select name="employees[${id}][job_role_id]" class="w-32 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none">
                                    ${rowJobRolesOptions}
                                </select>
                            </td>
                            <td class="px-3 py-3 whitespace-nowrap">
                                ${wagesHtml}
                            </td>
                            <td class="px-3 py-3 whitespace-nowrap">
                                <input type="text" name="employees[${id}][license_number]" value="${licenseNum}" class="w-32 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none" placeholder="License #">
                            </td>
                            <td class="px-3 py-3 whitespace-nowrap">
                                <input type="date" name="employees[${id}][license_expiry]" value="${licenseExp}" class="w-32 bg-white border border-gray-200 rounded-lg px-2 py-1.5 text-xs focus:ring-1 focus:ring-blue-500 outline-none">
                            </td>
                        </tr>
                    `;
                    tbody.innerHTML += row;
                });

                document.getElementById('bulk-update-modal').classList.remove('hidden');
                document.getElementById('bulk-update-modal').classList.add('flex');
            }

            function closeBulkUpdateModal() {
                document.getElementById('bulk-update-modal').classList.add('hidden');
                document.getElementById('bulk-update-modal').classList.remove('flex');
            }

            async function toggleEmployeeStatus(url, button) {
                const circle = button.querySelector('span');
                const isActive = button.getAttribute('aria-checked') === 'true';

                try {
                    button.setAttribute('aria-checked', !isActive);
                    if (!isActive) {
                        button.classList.remove('bg-gray-200'); button.classList.add('bg-blue-600');
                        circle.classList.remove('translate-x-0'); circle.classList.add('translate-x-4');
                    } else {
                        button.classList.remove('bg-blue-600'); button.classList.add('bg-gray-200');
                        circle.classList.remove('translate-x-4'); circle.classList.add('translate-x-0');
                    }
                    const response = await fetch(url, {
                        method: 'PATCH',
                        headers: { 'X-CSRF-TOKEN': '{{ csrf_token() }}', 'Content-Type': 'application/json', 'Accept': 'application/json' }
                    });
                    if (!response.ok) throw new Error('Network response was not ok');
                } catch (error) {
                    console.error('Error toggling status:', error);
                    button.setAttribute('aria-checked', isActive);
                    if (isActive) {
                        button.classList.remove('bg-gray-200'); button.classList.add('bg-blue-600');
                        circle.classList.remove('translate-x-0'); circle.classList.add('translate-x-4');
                    } else {
                        button.classList.remove('bg-blue-600'); button.classList.add('bg-gray-200');
                        circle.classList.remove('translate-x-4'); circle.classList.add('translate-x-0');
                    }
                    alert('Failed to update status. Please try again.');
                }
            }

            function copyToClipboard(text) {
                if (!text) return;
                navigator.clipboard.writeText(text).then(() => { alert("Email copied to clipboard: " + text); }).catch(err => { console.error('Failed to copy: ', err); });
            }
        </script>
    @endpush
@endsection