@extends('layouts.admin')

@section('title', 'Add Zone')
@section('page_title', 'Add Zone')

@section('content')
    <div class="max-w-4xl mx-auto">
        <div class="mb-6">
            <a href="{{ route('admin.zones.index') }}"
                class="flex items-center gap-2 text-gray-500 hover:text-blue-600 transition-colors w-fit">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18">
                    </path>
                </svg>
                Back to Zones
            </a>
        </div>

        <div class="bg-white rounded-2xl border border-gray-100 shadow-sm p-8">
            <h1 class="text-2xl font-bold text-gray-900 mb-6">Add New Zone</h1>

            <form action="{{ route('admin.zones.store') }}" method="POST" class="space-y-6">
                @csrf

                <!-- Name & ID -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">Zone Name <span
                                class="text-red-500">*</span></label>
                        <input type="text" name="name" value="{{ old('name') }}" required
                            class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                        @error('name') <p class="mt-1 text-xs text-red-500">{{ $message }}</p> @enderror
                    </div>
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">ID (Optional)</label>
                        <input type="text" name="external_id" value="{{ old('external_id') }}"
                            class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                    </div>
                </div>

                <!-- Details -->
                <div>
                    <label class="block text-sm font-bold text-gray-700 mb-2">Details</label>
                    <textarea name="details" rows="3"
                        class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">{{ old('details') }}</textarea>
                </div>

                <!-- Language & Template -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">Preferred Language</label>
                        <!-- Custom Searchable Dropdown for Language -->
                        <div class="relative custom-select" id="language-select">
                            <input type="hidden" name="preferred_language" value="{{ old('preferred_language') }}">
                            <button type="button"
                                class="select-trigger w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all text-left flex justify-between items-center text-gray-700">
                                <span class="selected-text">{{ old('preferred_language', 'Select Language') }}</span>
                                <svg class="w-4 h-4 text-gray-400 transition-transform" fill="none" stroke="currentColor"
                                    viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M19 9l-7 7-7-7"></path>
                                </svg>
                            </button>
                            <div
                                class="options-container absolute z-10 w-full mt-1 bg-white border border-gray-100 rounded-xl shadow-xl hidden overflow-hidden">
                                <div class="p-2 border-b border-gray-100">
                                    <input type="text"
                                        class="search-input w-full px-3 py-2 bg-gray-50 border border-gray-200 rounded-lg text-sm focus:outline-none focus:border-blue-500"
                                        placeholder="Search language...">
                                </div>
                                <div class="options-list max-h-60 overflow-y-auto p-1">
                                    <div class="option px-3 py-2 hover:bg-blue-50 hover:text-blue-600 rounded-lg cursor-pointer text-sm"
                                        data-value="">None</div>
                                    @foreach($languages as $lang)
                                        <div class="option px-3 py-2 hover:bg-blue-50 hover:text-blue-600 rounded-lg cursor-pointer text-sm"
                                            data-value="{{ $lang }}">{{ $lang }}</div>
                                    @endforeach
                                </div>
                            </div>
                        </div>
                    </div>
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">Zone Template</label>
                        <input type="text" name="zone_template" value="{{ old('zone_template') }}"
                            class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                    </div>
                </div>

                <div class="h-px bg-gray-100 my-4"></div>
                <h3 class="text-lg font-bold text-gray-900">Address Information</h3>
                <p class="text-sm text-gray-500 mb-4">If this is a patrol zone, you can set the address of where the patrol
                    vehicles normally reside.</p>

                <!-- Address -->
                <div>
                    <label class="block text-sm font-bold text-gray-700 mb-2">Address</label>
                    <textarea name="address" rows="2"
                        class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">{{ old('address') }}</textarea>
                </div>

                <!-- City & Zip -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">City</label>
                        <input type="text" name="city" value="{{ old('city') }}"
                            class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                    </div>
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">ZIP / Postal Code</label>
                        <input type="text" name="zip_code" value="{{ old('zip_code') }}"
                            class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all">
                    </div>
                </div>

                <!-- Country & State -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">Country</label>
                        <!-- Custom Searchable Dropdown for Country -->
                        <div class="relative custom-select" id="country-select">
                            <input type="hidden" name="country_id" id="country_id" value="{{ old('country_id') }}">
                            <button type="button"
                                class="select-trigger w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all text-left flex justify-between items-center text-gray-700">
                                <span
                                    class="selected-text">{{ old('country_id') ? $countries->firstWhere('id', old('country_id'))->name : 'Select Country' }}</span>
                                <svg class="w-4 h-4 text-gray-400 transition-transform" fill="none" stroke="currentColor"
                                    viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M19 9l-7 7-7-7"></path>
                                </svg>
                            </button>
                            <div
                                class="options-container absolute z-10 w-full mt-1 bg-white border border-gray-100 rounded-xl shadow-xl hidden overflow-hidden">
                                <div class="p-2 border-b border-gray-100">
                                    <input type="text"
                                        class="search-input w-full px-3 py-2 bg-gray-50 border border-gray-200 rounded-lg text-sm focus:outline-none focus:border-blue-500"
                                        placeholder="Search country...">
                                </div>
                                <div class="options-list max-h-60 overflow-y-auto p-1">
                                    <div class="option px-3 py-2 hover:bg-blue-50 hover:text-blue-600 rounded-lg cursor-pointer text-sm"
                                        data-value="">Select Country</div>
                                    @foreach($countries as $country)
                                        <div class="option px-3 py-2 hover:bg-blue-50 hover:text-blue-600 rounded-lg cursor-pointer text-sm"
                                            data-value="{{ $country->id }}">{{ $country->name }}</div>
                                    @endforeach
                                </div>
                            </div>
                        </div>
                    </div>
                    <div>
                        <label class="block text-sm font-bold text-gray-700 mb-2">State / Province</label>
                        <!-- Custom Searchable Dropdown for State -->
                        <div class="relative custom-select" id="state-select">
                            <input type="hidden" name="state_id" id="state_id" value="{{ old('state_id') }}">
                            <button type="button"
                                class="select-trigger w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all text-left flex justify-between items-center text-gray-700">
                                <span class="selected-text" id="state-selected-text">Select State</span>
                                <svg class="w-4 h-4 text-gray-400 transition-transform" fill="none" stroke="currentColor"
                                    viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M19 9l-7 7-7-7"></path>
                                </svg>
                            </button>
                            <div
                                class="options-container absolute z-10 w-full mt-1 bg-white border border-gray-100 rounded-xl shadow-xl hidden overflow-hidden">
                                <div class="p-2 border-b border-gray-100">
                                    <input type="text"
                                        class="search-input w-full px-3 py-2 bg-gray-50 border border-gray-200 rounded-lg text-sm focus:outline-none focus:border-blue-500"
                                        placeholder="Search state...">
                                </div>
                                <div class="options-list max-h-60 overflow-y-auto p-1" id="state-options-list">
                                    <div class="p-3 text-sm text-gray-400 text-center">Select a country first</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="flex items-center justify-end gap-3 pt-6">
                    <a href="{{ route('admin.zones.index') }}"
                        class="px-6 py-3 rounded-xl border border-gray-200 text-gray-700 font-bold hover:bg-gray-50 transition-all">Cancel</a>
                    <button type="submit"
                        class="px-6 py-3 rounded-xl bg-blue-600 text-white font-bold hover:bg-blue-700 transition-all shadow-lg shadow-blue-200">
                        Create Zone
                    </button>
                </div>
            </form>
        </div>
    </div>

    @push('scripts')
        <script>
            document.addEventListener('DOMContentLoaded', function () {
                // Dropdown Logic
                const dropdowns = document.querySelectorAll('.custom-select');

                dropdowns.forEach(dropdown => {
                    const trigger = dropdown.querySelector('.select-trigger');
                    const container = dropdown.querySelector('.options-container');
                    const searchInput = dropdown.querySelector('.search-input');
                    const optionsList = dropdown.querySelector('.options-list');
                    const hiddenInput = dropdown.querySelector('input[type="hidden"]');
                    const selectedText = dropdown.querySelector('.selected-text');

                    // Toggle dropdown
                    trigger.addEventListener('click', (e) => {
                        e.stopPropagation(); // Prevent closing immediately
                        closeAllDropdowns(dropdown);
                        container.classList.toggle('hidden');
                        if (!container.classList.contains('hidden')) {
                            searchInput.focus();
                        }
                    });

                    // Filter options
                    searchInput.addEventListener('input', (e) => {
                        const term = e.target.value.toLowerCase();
                        const options = optionsList.querySelectorAll('.option');
                        options.forEach(option => {
                            const text = option.textContent.toLowerCase();
                            if (text.includes(term)) {
                                option.style.display = 'block';
                            } else {
                                option.style.display = 'none';
                            }
                        });
                    });

                    // Select option
                    optionsList.addEventListener('click', (e) => {
                        if (e.target.classList.contains('option')) {
                            const value = e.target.dataset.value;
                            const text = e.target.textContent;

                            hiddenInput.value = value;
                            selectedText.textContent = text;
                            container.classList.add('hidden');
                            searchInput.value = ''; // Reset search

                            // Trigger change event for listeners (like country change)
                            hiddenInput.dispatchEvent(new Event('change'));
                        }
                    });

                    // Close when clicking outside
                    document.addEventListener('click', (e) => {
                        if (!dropdown.contains(e.target)) {
                            container.classList.add('hidden');
                        }
                    });
                });

                function closeAllDropdowns(except = null) {
                    dropdowns.forEach(d => {
                        if (d !== except) {
                            d.querySelector('.options-container').classList.add('hidden');
                        }
                    });
                }

                // Dependent Dropdown Logic (Country -> State)
                const countryInput = document.getElementById('country_id');
                const stateList = document.getElementById('state-options-list');
                const stateHiddenInput = document.getElementById('state_id');
                const stateSelectedText = document.getElementById('state-selected-text');

                countryInput.addEventListener('change', function () {
                    const countryId = this.value;
                    console.log('Country changed to:', countryId);

                    // Reset state selection
                    stateHiddenInput.value = '';
                    stateSelectedText.textContent = 'Select State';
                    stateList.innerHTML = '<div class="p-3 text-sm text-gray-400 text-center">Loading states...</div>';

                    if (!countryId) {
                        stateList.innerHTML = '<div class="p-3 text-sm text-gray-400 text-center">Select a country first</div>';
                        return;
                    }

                    // Fetch states
                    fetch(`{{ url('admin/zones/get-states') }}/${countryId}`)
                        .then(response => response.json())
                        .then(states => {
                            if (states.length === 0) {
                                stateList.innerHTML = '<div class="p-3 text-sm text-gray-400 text-center">No states found</div>';
                            } else {
                                let html = '<div class="option px-3 py-2 hover:bg-blue-50 hover:text-blue-600 rounded-lg cursor-pointer text-sm" data-value="">Select State</div>';
                                states.forEach(state => {
                                    html += `<div class="option px-3 py-2 hover:bg-blue-50 hover:text-blue-600 rounded-lg cursor-pointer text-sm" data-value="${state.id}">${state.name}</div>`;
                                });
                                stateList.innerHTML = html;
                            }
                        })
                        .catch(error => {
                            console.error('Error fetching states:', error);
                            stateList.innerHTML = '<div class="p-3 text-sm text-red-400 text-center">Error loading states</div>';
                        });
                });

                // Trigger logic on load if Old value present (Handle later if needed, simple logic for now)
                if (countryInput.value) {
                    // Manually fetch states if loaded with value, but better to handle strictly via change or pre-load in blade.
                    // For now, let's just trigger change if it has value, but avoiding double load if blade pre-rendered states logic (which we didn't do, we rely on JS).
                    countryInput.dispatchEvent(new Event('change'));
                    // Wait, dispatching change will wipe the OLD state value if we don't preserve it. 
                    // Better to load states and THEN set value.
                    // Since this is a new "Create" form, old value only exists on validation error.
                    // Handling old state value on validation error is tricky with async fetch.
                    // Simplest way: render states in PHP if old country exists? 
                    // Or just re-fetch and set value in JS.
                }
            });
        </script>
    @endpush
@endsection