@extends('layouts.admin')

@section('title', 'System Users')
@section('page_title', 'Users')

@section('content')
    <div class="mb-6 flex flex-col md:flex-row md:items-center justify-between gap-4">
        <div>
            <h1 class="text-2xl font-bold text-gray-800">System Users</h1>
            <p class="text-sm text-gray-500 mt-1">Manage system administrators and back-office staff access.</p>
        </div>
        <div class="flex flex-wrap items-center gap-2">
            <form action="{{ route('admin.users.index') }}" method="GET" class="relative group">
                <input type="text" name="search" value="{{ $search ?? '' }}" placeholder="Search users..."
                    class="pl-9 pr-4 py-2 bg-white border border-gray-200 rounded-md w-64 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all text-sm">
                <svg class="w-4 h-4 absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 group-focus-within:text-blue-500 transition-colors"
                    fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                </svg>
                @if($search)
                    <a href="{{ route('admin.users.index') }}"
                        class="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 hover:text-gray-600">
                        <svg class="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                            </path>
                        </svg>
                    </a>
                @endif
            </form>

            <a href="{{ route('admin.users.export-pdf') }}"
                class="p-2 bg-white border border-gray-200 text-gray-700 rounded-md hover:bg-gray-50 transition-all shadow-sm group"
                title="Export PDF">
                <svg class="w-4 h-4 text-red-500 group-hover:scale-110 transition-transform" fill="none"
                    stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z">
                    </path>
                </svg>
            </a>

            <a href="{{ route('admin.users.create') }}"
                class="inline-flex items-center gap-2 bg-blue-600 text-white px-4 py-2 rounded-md font-bold text-sm hover:bg-blue-700 transition-all shadow-sm">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6">
                    </path>
                </svg>
                Add User
            </a>
        </div>
    </div>

    @if(session('success'))
        <div class="mb-6 p-4 bg-green-50 border border-green-100 text-green-700 rounded-lg flex items-center gap-3">
            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
            </svg>
            {{ session('success') }}
        </div>
    @endif

    <div class="bg-white rounded-lg border border-gray-100 shadow-sm overflow-hidden">
        <table class="w-full text-left border-collapse">
            <thead class="bg-gray-50/50 border-b border-gray-100">
                <tr>
                    <th class="px-6 py-4 text-xs font-semibold text-gray-500 uppercase tracking-wider">User</th>
                    <th class="px-6 py-4 text-xs font-semibold text-gray-500 uppercase tracking-wider">Status</th>
                    <th class="px-6 py-4 text-xs font-semibold text-gray-500 uppercase tracking-wider">Companies</th>
                    <th class="px-6 py-4 text-xs font-semibold text-gray-500 uppercase tracking-wider">Joined At</th>
                    <th class="px-6 py-4 text-xs font-semibold text-gray-500 uppercase tracking-wider text-right">Actions
                    </th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-50">
                @forelse($users as $user)
                    <tr class="hover:bg-gray-50/50 transition-colors">
                        <td class="px-6 py-4">
                            <div class="flex items-center gap-3">
                                <div
                                    class="w-8 h-8 rounded-full bg-blue-50 flex items-center justify-center text-blue-600 font-bold border border-blue-100 text-xs">
                                    {{ strtoupper(substr($user->name, 0, 1)) }}
                                </div>
                                <div>
                                    <div class="text-sm font-bold text-gray-800">{{ $user->name }}</div>
                                    <div class="flex items-center gap-2">
                                        <span class="text-xs text-gray-500">{{ $user->email }}</span>
                                        <span class="w-1 h-1 rounded-full bg-gray-300"></span>
                                        <span
                                            class="text-[10px] font-bold text-blue-600 uppercase tracking-widest">{{ $user->roles->pluck('name')->implode(', ') ?: 'No Role' }}</span>
                                    </div>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4">
                            <button type="button"
                                onclick="toggleStatus('{{ route('admin.users.toggle-status', $user) }}', this)"
                                class="relative inline-flex h-5 w-9 flex-shrink-0 cursor-pointer rounded-full border-2 border-transparent transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-blue-600 focus:ring-offset-2 {{ $user->active ? 'bg-blue-600' : 'bg-gray-200' }}"
                                role="switch" aria-checked="{{ $user->active ? 'true' : 'false' }}">
                                <span aria-hidden="true"
                                    class="pointer-events-none inline-block h-4 w-4 transform rounded-full bg-white shadow ring-0 transition duration-200 ease-in-out {{ $user->active ? 'translate-x-4' : 'translate-x-0' }}"></span>
                            </button>
                        </td>
                        <td class="px-6 py-4">
                            <span
                                class="text-sm font-bold {{ $user->companies->count() > 0 ? 'text-blue-600' : 'text-gray-400' }}">
                                {{ $user->companies->count() }} {{ Str::plural('Company', $user->companies->count()) }}
                            </span>
                        </td>
                        <td class="px-6 py-4 text-sm text-gray-500 font-medium">{{ $user->created_at->format('M d, Y') }}</td>
                        <td class="px-6 py-4 text-right">
                            <div class="flex items-center justify-end gap-2">
                                <button type="button"
                                    onclick="openCompaniesModal({{ $user->id }}, '{{ $user->name }}', {{ json_encode($user->companies->pluck('id')) }})"
                                    class="p-1.5 text-gray-400 hover:text-emerald-600 hover:bg-emerald-50 rounded-md transition-all"
                                    title="Allow Companies">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4">
                                        </path>
                                    </svg>
                                </button>
                                <button type="button" onclick="openPasswordModal({{ $user->id }}, '{{ $user->name }}')"
                                    class="p-1.5 text-gray-400 hover:text-amber-600 hover:bg-amber-50 rounded-md transition-all"
                                    title="Change Password">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M15 7a2 2 0 012 2m4 0a6 6 0 01-7.743 5.743L11 17H9v2H7v2H4a1 1 0 01-1-1v-2.586a1 1 0 01.293-.707l5.964-5.964A6 6 0 1121 9z">
                                        </path>
                                    </svg>
                                </button>
                                <a href="{{ route('admin.users.edit', $user) }}"
                                    class="p-1.5 text-gray-400 hover:text-blue-600 hover:bg-blue-50 rounded-md transition-all">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z">
                                        </path>
                                    </svg>
                                </a>
                                <button type="button"
                                    onclick="openDeleteModal('{{ route('admin.users.destroy', $user) }}', '{{ $user->name }}')"
                                    class="p-1.5 text-gray-400 hover:text-red-600 hover:bg-red-50 rounded-md transition-all"
                                    title="Delete">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16">
                                        </path>
                                    </svg>
                                </button>
                            </div>
                        </td>
                    </tr>
                @empty
                    <tr>
                        <td colspan="5" class="px-6 py-12 text-center text-gray-500 text-sm">
                            No employees found. Click "Add Use" to create one.
                        </td>
                    </tr>
                @endforelse
            </tbody>
        </table>
        @if($users->hasPages())
            <div class="px-6 py-4 border-t border-gray-100">
                {{ $users->links() }}
            </div>
        @endif
    </div>

    <!-- Allowed Companies Modal -->
    <div id="companiesModal" class="fixed inset-0 z-50 hidden">
        <!-- Backdrop -->
        <div class="absolute inset-0 bg-slate-900/40 backdrop-blur-sm"></div>

        <!-- Modal Content -->
        <div class="absolute inset-0 flex items-center justify-center p-6">
            <div class="bg-white rounded-lg shadow-xl w-full max-w-lg transform transition-all scale-95 opacity-0 duration-200"
                id="companiesModalContent">
                <form id="companiesForm" method="POST">
                    @csrf
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-6">
                            <div>
                                <h3 class="text-lg font-bold text-gray-800" id="modalUserName">Allow Companies</h3>
                                <p class="text-gray-500 text-xs mt-1">Select companies this user can access.</p>
                            </div>
                            <button type="button" onclick="closeCompaniesModal()"
                                class="p-2 text-gray-400 hover:text-gray-900 hover:bg-gray-100 rounded-md transition-all">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M6 18L18 6M6 6l12 12"></path>
                                </svg>
                            </button>
                        </div>

                        <div class="max-h-96 overflow-y-auto pr-2 space-y-2 mb-6 custom-scrollbar">
                            @foreach($companies as $company)
                                <label
                                    class="flex items-center gap-3 p-3 rounded-md border border-gray-100 hover:bg-gray-50 cursor-pointer transition-all group">
                                    <input type="checkbox" name="companies[]" value="{{ $company->id }}"
                                        class="w-4 h-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500 transition-all company-checkbox">
                                    <div class="flex items-center gap-3">
                                        @if($company->logo)
                                            <img src="{{ asset('storage/' . $company->logo) }}" alt=""
                                                class="w-8 h-8 rounded-md object-cover border border-gray-100">
                                        @else
                                            <div
                                                class="w-8 h-8 rounded-md bg-gray-100 flex items-center justify-center text-xs font-bold text-gray-400">
                                                {{ strtoupper(substr($company->name, 0, 1)) }}
                                            </div>
                                        @endif
                                        <div>
                                            <div class="text-sm font-bold text-gray-800">{{ $company->name }}</div>
                                            <div class="text-xs text-gray-500">{{ $company->timeZone->name ?? 'N/A' }}</div>
                                        </div>
                                    </div>
                                </label>
                            @endforeach
                        </div>

                        <div class="flex items-center gap-3 pt-4 border-t border-gray-50">
                            <button type="button" onclick="closeCompaniesModal()"
                                class="flex-1 px-4 py-2 rounded-md border border-gray-200 text-gray-700 font-bold hover:bg-gray-50 transition-all text-sm">
                                Cancel
                            </button>
                            <button type="submit"
                                class="flex-1 px-4 py-2 rounded-md bg-emerald-600 text-white font-bold hover:bg-emerald-700 transition-all shadow-sm text-sm">
                                Save Selections
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Change Password Modal -->
    <div id="passwordModal" class="fixed inset-0 z-50 hidden">
        <!-- Backdrop -->
        <div class="absolute inset-0 bg-slate-900/40 backdrop-blur-sm"></div>

        <!-- Modal Content -->
        <div class="absolute inset-0 flex items-center justify-center p-6">
            <div class="bg-white rounded-lg shadow-xl w-full max-w-md transform transition-all scale-95 opacity-0 duration-200"
                id="passwordModalContent">
                <form id="passwordForm" method="POST">
                    @csrf
                    @method('PATCH')
                    <div class="p-6">
                        <div class="flex items-center justify-between mb-6">
                            <div>
                                <h3 class="text-lg font-bold text-gray-800">Change Password</h3>
                                <p class="text-gray-500 text-xs mt-1" id="passwordModalUser">Update user credentials.</p>
                            </div>
                            <button type="button" onclick="closePasswordModal()"
                                class="p-2 text-gray-400 hover:text-gray-900 hover:bg-gray-100 rounded-md transition-all">
                                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M6 18L18 6M6 6l12 12"></path>
                                </svg>
                            </button>
                        </div>

                        <div class="space-y-4 mb-6">
                            <div>
                                <label for="new_password"
                                    class="block text-xs font-semibold text-gray-500 uppercase mb-2">New
                                    Password</label>
                                <input type="password" id="new_password" name="password" required
                                    class="w-full px-3 py-2 rounded-md border border-gray-200 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all text-gray-900 text-sm">
                            </div>
                            <div>
                                <label for="password_confirmation"
                                    class="block text-xs font-semibold text-gray-500 uppercase mb-2">Confirm New
                                    Password</label>
                                <input type="password" id="password_confirmation" name="password_confirmation" required
                                    class="w-full px-3 py-2 rounded-md border border-gray-200 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all text-gray-900 text-sm">
                            </div>
                        </div>

                        <div class="flex items-center gap-3">
                            <button type="button" onclick="closePasswordModal()"
                                class="flex-1 px-4 py-2 rounded-md border border-gray-200 text-gray-700 font-bold hover:bg-gray-50 transition-all text-sm">
                                Cancel
                            </button>
                            <button type="submit"
                                class="flex-1 px-4 py-2 rounded-md bg-amber-600 text-white font-bold hover:bg-amber-700 transition-all shadow-sm text-sm">
                                Update Password
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <style>
        .custom-scrollbar::-webkit-scrollbar {
            width: 4px;
        }

        .custom-scrollbar::-webkit-scrollbar-track {
            background: transparent;
        }

        .custom-scrollbar::-webkit-scrollbar-thumb {
            background: #E2E8F0;
            border-radius: 10px;
        }

        .custom-scrollbar::-webkit-scrollbar-thumb:hover {
            background: #CBD5E1;
        }
    </style>

    <script>
        function openCompaniesModal(userId, userName, currentCompanies) {
            const modal = document.getElementById('companiesModal');
            const content = document.getElementById('companiesModalContent');
            const form = document.getElementById('companiesForm');
            const nameDisplay = document.getElementById('modalUserName');
            const checkboxes = document.querySelectorAll('.company-checkbox');

            form.action = `/admin/users/${userId}/sync-companies`;
            nameDisplay.innerText = `Allowed Companies for ${userName}`;

            // Reset and check currents
            checkboxes.forEach(cb => {
                cb.checked = currentCompanies.includes(parseInt(cb.value));
            });

            modal.classList.remove('hidden');
            setTimeout(() => {
                content.classList.remove('scale-95', 'opacity-0');
                content.classList.add('scale-100', 'opacity-100');
            }, 10);
        }

        function closeCompaniesModal() {
            const modal = document.getElementById('companiesModal');
            const content = document.getElementById('companiesModalContent');

            content.classList.remove('scale-100', 'opacity-100');
            content.classList.add('scale-95', 'opacity-0');

            setTimeout(() => {
                modal.classList.add('hidden');
            }, 200);
        }

        function openPasswordModal(userId, userName) {
            const modal = document.getElementById('passwordModal');
            const content = document.getElementById('passwordModalContent');
            const form = document.getElementById('passwordForm');
            const nameDisplay = document.getElementById('passwordModalUser');

            form.action = `/admin/users/${userId}/password`;
            nameDisplay.innerText = `Update credentials for ${userName}`;

            modal.classList.remove('hidden');
            setTimeout(() => {
                content.classList.remove('scale-95', 'opacity-0');
                content.classList.add('scale-100', 'opacity-100');
            }, 10);
        }

        function closePasswordModal() {
            const modal = document.getElementById('passwordModal');
            const content = document.getElementById('passwordModalContent');

            content.classList.remove('scale-100', 'opacity-100');
            content.classList.add('scale-95', 'opacity-0');

            setTimeout(() => {
                modal.classList.add('hidden');
            }, 200);
        }
    </script>
@endsection