@extends('layouts.admin')

@section('title', 'All Employees')
@section('page_title', 'All Employees')

@section('content')
    <div class="mb-6 flex flex-col md:flex-row md:items-center justify-between gap-4">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">All Employees</h1>
            <p class="text-gray-500 text-sm">Manage all employees across all companies.</p>
        </div>
        <div class="flex flex-wrap items-center gap-3">
             <a href="{{ route('admin.employees.export-pdf', request()->all()) }}" target="_blank" class="px-3 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition font-medium flex items-center gap-2 text-xs">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                Export PDF
            </a>
            <form id="search-form" action="{{ route('admin.employees.index') }}" method="GET" class="flex flex-wrap items-center gap-3">
                <!-- Company Filter -->
                <select name="company_id" id="company-filter"
                    class="bg-white border border-gray-200 rounded-xl px-4 py-2 text-sm font-medium text-gray-700 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all shadow-sm cursor-pointer">
                    <option value="">All Companies</option>
                    @foreach($companies as $company)
                        <option value="{{ $company->id }}" {{ request('company_id') == $company->id ? 'selected' : '' }}>
                            {{ $company->name }} (Emp: {{ $company->employees_count }}, Sites: {{ $company->sites_count }})
                        </option>
                    @endforeach
                </select>

                <!-- Search -->
                <div class="relative group">
                    <input type="text" name="search" id="search-input" value="{{ request('search') }}"
                        placeholder="Search (Name, Phone, ID)..."
                        class="pl-10 pr-4 py-2.5 bg-white border border-gray-200 rounded-xl w-72 focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all shadow-sm">
                    <svg class="w-5 h-5 absolute left-3.5 top-1/2 -translate-y-1/2 text-gray-400 group-focus-within:text-blue-500 transition-colors"
                        fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
                    </svg>
                    <a href="{{ route('admin.employees.index') }}" id="clear-search"
                        class="absolute right-3 top-1/2 -translate-y-1/2 text-gray-400 hover:text-gray-600 {{ request('search') ? '' : 'hidden' }}">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12">
                            </path>
                        </svg>
                    </a>
                </div>

                <button type="submit" class="hidden">Search</button>
            </form>
        </div>
    </div>

    @if(request('company_id'))
        @php
            $selectedCompany = $companies->firstWhere('id', request('company_id'));
        @endphp
        @if($selectedCompany)
            <div class="mb-10 grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div
                    class="bg-gradient-to-br from-blue-500 to-blue-600 p-6 rounded-3xl text-white shadow-lg shadow-blue-200/50 transition-all hover:scale-[1.02]">
                    <div class="flex items-center gap-4">
                        <div class="w-12 h-12 bg-white/20 rounded-xl flex items-center justify-center backdrop-blur-sm">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z">
                                </path>
                            </svg>
                        </div>
                        <div>
                            <p class="text-[10px] font-bold uppercase tracking-widest opacity-80 leading-tight">Company Employees
                            </p>
                            <h3 class="text-3xl font-black tracking-tight mt-1">{{ $selectedCompany->employees_count }}</h3>
                        </div>
                    </div>
                </div>
                <div
                    class="bg-gradient-to-br from-indigo-500 to-purple-600 p-6 rounded-3xl text-white shadow-lg shadow-indigo-200/50 transition-all hover:scale-[1.02]">
                    <div class="flex items-center gap-4">
                        <div class="w-12 h-12 bg-white/20 rounded-xl flex items-center justify-center backdrop-blur-sm">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4">
                                </path>
                            </svg>
                        </div>
                        <div>
                            <p class="text-[10px] font-bold uppercase tracking-widest opacity-80 leading-tight">Company Sites</p>
                            <h3 class="text-3xl font-black tracking-tight mt-1">{{ $selectedCompany->sites_count }}</h3>
                        </div>
                    </div>
                </div>
            </div>
        @endif
    @endif

    @if(session('success'))
        <div class="mb-6 p-4 bg-emerald-50 border border-emerald-100 text-emerald-700 rounded-xl flex items-center gap-3">
            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
            </svg>
            {{ session('success') }}
        </div>
    @endif

    <div id="table-wrapper">
        @include('admin.employees._table')
    </div>
@endsection

@push('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const searchInput = document.getElementById('search-input');
        const companyFilter = document.getElementById('company-filter');
        const tableWrapper = document.getElementById('table-wrapper');
        const clearSearch = document.getElementById('clear-search');
        let debounceTimer;

        function fetchEmployees(url = null) {
            const formData = new FormData(document.getElementById('search-form'));
            const params = new URLSearchParams(formData);
            
            if (!url) {
                url = "{{ route('admin.employees.index') }}?" + params.toString();
            } else {
                const targetUrl = new URL(url);
                params.forEach((value, key) => {
                    if (!targetUrl.searchParams.has(key)) {
                        targetUrl.searchParams.append(key, value);
                    }
                });
                url = targetUrl.toString();
            }

            window.history.pushState({}, '', url);

            fetch(url, {
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.text())
            .then(html => {
                tableWrapper.innerHTML = html;
                attachPaginationListeners();
            })
            .catch(error => console.error('Error fetching employees:', error));
        }

        function attachPaginationListeners() {
            const paginationLinks = tableWrapper.querySelectorAll('.table-pagination a, .ajax-sort');
            paginationLinks.forEach(link => {
                link.addEventListener('click', function(e) {
                    e.preventDefault();
                    fetchEmployees(this.getAttribute('href'));
                    window.scrollTo({ top: 0, behavior: 'smooth' });
                });
            });
        }

        if (searchInput) {
            searchInput.addEventListener('input', function() {
                if (this.value) {
                    clearSearch.classList.remove('hidden');
                } else {
                    clearSearch.classList.add('hidden');
                }

                clearTimeout(debounceTimer);
                debounceTimer = setTimeout(() => {
                    fetchEmployees();
                }, 500);
            });
        }

        if (companyFilter) {
            companyFilter.addEventListener('change', function() {
                fetchEmployees();
            });
        }

        if (clearSearch) {
            clearSearch.addEventListener('click', function(e) {
                e.preventDefault();
                searchInput.value = '';
                this.classList.add('hidden');
                fetchEmployees();
            });
        }

        window.addEventListener('popstate', function() {
            location.reload();
        });

        attachPaginationListeners();
    });
</script>
@endpush