<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;

class UserCsvSeeder extends Seeder
{
    public function run()
    {
        $file = public_path('users-4.csv');

        if (!file_exists($file)) {
            $this->command->error("File not found: $file");
            return;
        }

        $data = array_map('str_getcsv', file($file));
        $headers = array_map('trim', $data[0]);
        unset($data[0]); // Remove headers

        // Map CSV headers to database columns
        // CSV: Email,Employee ID,First Name,Last Name,Gender,Date of Birth,Employment Date,Phone Number,Permanent Address Line 1,Corresponding Address Line 1,SIN Number,Company ID,upload_license,upload_id,License Number,License Expiry Date,Has Car,wage_type,cash,sin,contract

        $employees = [];
        $now = now();
        $password = Hash::make('12345678'); // Default password

        foreach ($data as $row) {
            // Combine row with headers to make it associative
            if (count($row) !== count($headers)) {
                // Handle potential mismatch or empty lines
                continue;
            }

            $rowData = array_combine($headers, $row);

            // Parse dates
            try {
                $dob = !empty($rowData['Date of Birth']) ? Carbon::createFromFormat('d-m-Y', $rowData['Date of Birth'])->format('Y-m-d') : null;
            } catch (\Exception $e) {
                $dob = null;
            }

            try {
                $employDate = !empty($rowData['Employment Date']) ? Carbon::createFromFormat('d-m-Y', $rowData['Employment Date'])->format('Y-m-d') : null;
            } catch (\Exception $e) {
                $employDate = null;
            }

            try {
                $licenseExpiry = !empty($rowData['License Expiry Date']) ? Carbon::createFromFormat('d-m-Y', $rowData['License Expiry Date'])->format('Y-m-d') : null;
            } catch (\Exception $e) {
                $licenseExpiry = null;
            }

            // Map has_car
            $hasCar = strtolower(trim($rowData['Has Car'])) === 'yes' ? 1 : 0;

            // Map Gender
            $gender = ucfirst(strtolower(trim($rowData['Gender'])));

            $employees[] = [
                'email' => $rowData['Email'],
                'employee_id' => $rowData['Employee ID'],
                'first_name' => $rowData['First Name'],
                'last_name' => $rowData['Last Name'],
                'gender' => $gender,
                'birthday' => $dob,
                'employment_date' => $employDate,
                'phone_number' => $rowData['Phone Number'],
                'permanent_address_line_1' => trim($rowData['Permanent Address Line 1'], ' "'),
                'corresponding_address_line_1' => trim($rowData['Corresponding Address Line 1'], ' "'),
                'sin_number' => $rowData['SIN Number'] === 'NULL' ? null : $rowData['SIN Number'],
                'company_id' => (int) $rowData['Company ID'],
                'file_license' => $rowData['upload_license'],
                'license_number' => $rowData['License Number'],
                'license_expiry' => $licenseExpiry,
                'has_car' => $hasCar,
                'file_contract' => $rowData['contract'] ?? null,
                // Default/Required values
                'password' => $password,
                'active' => 1,
                'created_at' => $now,
                'updated_at' => $now,
            ];
        }

        // Insert in chunks
        $chunks = array_chunk($employees, 50);
        foreach ($chunks as $chunk) {
            DB::table('employees')->upsert(
                $chunk,
                ['employee_id', 'email'], // Unique constraints to match for update
                [
                    'first_name',
                    'last_name',
                    'gender',
                    'birthday',
                    'employment_date',
                    'phone_number',
                    'permanent_address_line_1',
                    'corresponding_address_line_1',
                    'sin_number',
                    'company_id',
                    'file_license',
                    'license_number',
                    'license_expiry',
                    'has_car',
                    'file_contract',
                    'updated_at'
                ]
            );
        }

        $this->command->info('Imported ' . count($employees) . ' employees.');
    }
}
