<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Schedule;
use App\Models\Employee;
use App\Models\Site;
use Carbon\Carbon;

class TestScheduleSeeder extends Seeder
{
    public function run()
    {
        $employees = Employee::all();
        $sites = Site::all();

        if ($employees->isEmpty() || $sites->isEmpty()) {
            $this->command->info('Please seed employees and sites first.');
            return;
        }

        // 1. Upcoming (5)
        for ($i = 0; $i < 5; $i++) {
            $this->createSchedule($sites->random(), $employees->random(), 'upcoming');
        }

        // 2. Missed (5)
        for ($i = 0; $i < 5; $i++) {
            $this->createSchedule($sites->random(), $employees->random(), 'missed');
        }

        // 3. Completed (5)
        for ($i = 0; $i < 5; $i++) {
            $this->createSchedule($sites->random(), $employees->random(), 'completed');
        }

        // 4. In Progress (5)
        for ($i = 0; $i < 5; $i++) {
            $this->createSchedule($sites->random(), $employees->random(), 'in_progress');
        }
    }

    private function createSchedule($site, $employee, $type)
    {
        $now = Carbon::now();
        $start = $now->copy();
        $end = $now->copy();
        $pivotData = [];

        switch ($type) {
            case 'upcoming':
                $start = $now->copy()->addDays(rand(1, 5))->hour(rand(8, 20));
                $end = $start->copy()->addHours(8);
                // No pivot data needed (defaults are null)
                break;

            case 'missed':
                $start = $now->copy()->subDays(rand(2, 5))->hour(rand(8, 20));
                $end = $start->copy()->addHours(8);
                // No checkin data
                break;

            case 'completed':
                $start = $now->copy()->subDays(rand(2, 10))->hour(rand(8, 20));
                $end = $start->copy()->addHours(8);
                $pivotData = [
                    'actual_start_at' => $start->copy()->addMinutes(rand(-5, 10)), // Checked in around start
                    'actual_end_at' => $end->copy()->addMinutes(rand(-5, 30)),   // Checked out around end
                ];
                break;

            case 'in_progress':
                $start = $now->copy()->subHours(4); // Started 4 hours ago
                $end = $start->copy()->addHours(8); // Ends in 4 hours
                $pivotData = [
                    'actual_start_at' => $start->copy()->addMinutes(rand(0, 5)),
                    'actual_end_at' => null,
                ];
                break;
        }

        $schedule = Schedule::create([
            'company_id' => $site->company_id,
            'site_id' => $site->id,
            'duty_number' => 'TEST-' . rand(1000, 9999),
            'schedule_date' => $start->toDateString(),
            'from_datetime' => $start,
            'to_datetime' => $end,
            'site_rate' => 25.00, // Default mock rate
        ]);

        $schedule->employees()->attach($employee->id, $pivotData);
    }
}
