<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Company;
use App\Models\Site;
use App\Models\Route;
use Illuminate\Support\Str;

class DummySitesAndRoutesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $companies = Company::all();

        // Get default country and state IDs (Canada/Ontario as default)
        $defaultCountryId = \App\Models\Country::where('name', 'Canada')->first()->id ?? 1;
        $defaultStateId = \App\Models\State::where('name', 'Ontario')->first()->id ?? 1;

        foreach ($companies as $company) {
            $this->command->info("Creating dummy sites and routes for company: {$company->name}");

            // Create 10 dummy sites for each company
            for ($i = 1; $i <= 10; $i++) {
                $site = Site::create([
                    'company_id' => $company->id,
                    'site_id' => 'SITE-' . strtoupper(Str::random(6)),
                    'name' => "Site Location {$i}",
                    'account_type' => $i % 2 == 0 ? 'Multi-Site client' : 'Regular Client',
                    'address_line_1' => "{$i}00 Main Street",
                    'address_line_2' => "Suite " . rand(100, 999),
                    'city' => $this->getRandomCity(),
                    'state_id' => $defaultStateId,
                    'country_id' => $defaultCountryId,
                    'zip_code' => $this->generatePostalCode(),
                    'latitude' => (string) (43 + (rand(0, 100) / 100)),
                    'longitude' => (string) (-79 - (rand(0, 100) / 100)),
                    'contact_first_name' => "Contact",
                    'contact_last_name' => "Person {$i}",
                    'contact_phone_main' => $this->generatePhoneNumber(),
                    'contact_email' => "site{$i}@" . strtolower(str_replace(' ', '', $company->name)) . ".com",
                    'rate' => rand(25, 50),
                    'active' => true,
                ]);

                $this->command->info("  - Created site: {$site->name}");
            }

            // Create 10 dummy routes for each company
            for ($i = 1; $i <= 10; $i++) {
                $route = Route::create([
                    'company_id' => $company->id,
                    'route_code' => 'ROUTE-' . strtoupper(Str::random(6)),
                    'name' => "Route {$i} - " . $this->getRandomRouteName(),
                    'description' => "Patrol route {$i} covering multiple locations in the area",
                    'rate' => rand(30, 60),
                    'active' => true,
                ]);

                $this->command->info("  - Created route: {$route->name}");
            }

            $this->command->info("Completed for company: {$company->name}\n");
        }

        $this->command->info("All dummy sites and routes created successfully!");
    }

    /**
     * Get a random city name
     */
    private function getRandomCity(): string
    {
        $cities = [
            'Toronto',
            'Vancouver',
            'Montreal',
            'Calgary',
            'Edmonton',
            'Ottawa',
            'Winnipeg',
            'Quebec City',
            'Hamilton',
            'Kitchener',
            'London',
            'Victoria',
            'Halifax',
            'Oshawa',
            'Windsor'
        ];

        return $cities[array_rand($cities)];
    }

    /**
     * Get a random route name
     */
    private function getRandomRouteName(): string
    {
        $names = [
            'Downtown Loop',
            'Industrial Park',
            'Residential Area',
            'Business District',
            'Waterfront',
            'North Sector',
            'South Sector',
            'East Zone',
            'West Zone',
            'Central Area',
            'Suburban Route',
            'Commercial Zone',
            'Tech Park',
            'University District',
            'Shopping District'
        ];

        return $names[array_rand($names)];
    }

    /**
     * Generate a random phone number
     */
    private function generatePhoneNumber(): string
    {
        return sprintf(
            '(%03d) %03d-%04d',
            rand(200, 999),
            rand(200, 999),
            rand(1000, 9999)
        );
    }

    /**
     * Generate a random Canadian postal code
     */
    private function generatePostalCode(): string
    {
        $letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $digits = '0123456789';

        return sprintf(
            '%s%s%s %s%s%s',
            $letters[rand(0, 25)],
            $digits[rand(0, 9)],
            $letters[rand(0, 25)],
            $digits[rand(0, 9)],
            $letters[rand(0, 25)],
            $digits[rand(0, 9)]
        );
    }
}
