<?php

namespace Database\Seeders;

use App\Models\Company;
use App\Models\Site;
use App\Models\Employee;
use App\Models\JobRole;
use App\Models\Department;
use App\Models\Zone;
use App\Models\Country;
use App\Models\State;
use App\Models\IdSetting;
use Illuminate\Database\Seeder;
use Faker\Factory as Faker;
use Illuminate\Support\Str;

class DummyDataSeeder extends Seeder
{
    public function run(): void
    {
        $faker = Faker::create();
        $companies = Company::all();

        if ($companies->isEmpty()) {
            $this->command->info('No companies found. Create at least one company first.');
            return;
        }

        $country = Country::first() ?? Country::create(['name' => 'Canada', 'code' => 'CA', 'active' => true]);
        $state = State::first() ?? State::create(['name' => 'Ontario', 'code' => 'ON', 'country_id' => $country->id, 'active' => true]);
        $jobRole = JobRole::first() ?? JobRole::create(['name' => 'Security Guard', 'active' => true]);
        $department = Department::first() ?? Department::create(['name' => 'Operations', 'active' => true]);
        $zone = Zone::first() ?? Zone::create([
            'name' => 'Toronto East',
            'country_id' => $country->id,
            'state_id' => $state->id,
            'active' => true
        ]);

        foreach ($companies as $company) {
            $this->command->info("Seeding data for company: {$company->name}");

            // Ensure ID Settings exist
            $idSetting = IdSetting::firstOrCreate(
                ['company_id' => $company->id],
                [
                    'emp_prefix' => 'E',
                    'emp_next_number' => 1,
                    'site_prefix' => 'S',
                    'site_next_number' => 1,
                    'route_prefix' => 'R',
                    'route_next_number' => 1,
                    'duty_prefix' => 'D',
                    'duty_next_number' => 1,
                    'incident_prefix' => 'I',
                    'incident_next_number' => 1,
                ]
            );

            // 1. Create 10 Sites
            for ($i = 0; $i < 10; $i++) {
                $siteId = $idSetting->site_prefix . str_pad($idSetting->site_next_number, 5, '0', STR_PAD_LEFT);

                Site::create([
                    'company_id' => $company->id,
                    'account_type' => $faker->randomElement(['Regular Client', 'Multi-Site client']),
                    'name' => $faker->company . " Site",
                    'rate' => $faker->randomFloat(2, 15, 50),
                    'site_id' => $siteId,
                    'language' => 'English',
                    'contact_first_name' => $faker->firstName,
                    'contact_last_name' => $faker->lastName,
                    'contact_phone_main' => $faker->phoneNumber,
                    'contact_email' => $faker->safeEmail,
                    'address_line_1' => $faker->streetAddress,
                    'country_id' => $country->id,
                    'state_id' => $state->id,
                    'city' => $faker->city,
                    'zip_code' => $faker->postcode,
                    'latitude' => $faker->latitude,
                    'longitude' => $faker->longitude,
                    'active' => true,
                ]);

                $idSetting->increment('site_next_number');
            }

            // 2. Create 10 Employees
            for ($i = 0; $i < 10; $i++) {
                $empId = $idSetting->emp_prefix . str_pad($idSetting->emp_next_number, 5, '0', STR_PAD_LEFT);

                Employee::create([
                    'company_id' => $company->id,
                    'employee_id' => $empId,
                    'first_name' => $faker->firstName,
                    'last_name' => $faker->lastName,
                    'gender' => $faker->randomElement(['Male', 'Female']),
                    'email' => $faker->unique()->safeEmail,
                    'phone_number' => $faker->phoneNumber,
                    'job_role_id' => $jobRole->id,
                    'department_id' => $department->id,
                    'zone_id' => $zone->id,
                    'license_number' => 'LIC' . strtoupper(Str::random(8)),
                    'license_expiry' => $faker->dateTimeBetween('+1 year', '+5 years'),
                    'file_license' => 'dummy/license.pdf',
                    'file_contract' => 'dummy/contract.pdf',
                    'employment_date' => $faker->dateTimeBetween('-2 years', 'now'),
                    'permanent_address_line_1' => $faker->streetAddress,
                    'permanent_city' => $faker->city,
                    'permanent_state_id' => $state->id,
                    'permanent_country_id' => $country->id,
                    'permanent_zip_code' => $faker->postcode,
                    'active' => true,
                ]);

                $idSetting->increment('emp_next_number');
            }
        }
    }
}
