<?php

use Illuminate\Foundation\Application;
use Illuminate\Foundation\Configuration\Exceptions;
use Illuminate\Foundation\Configuration\Middleware;

return Application::configure(basePath: dirname(__DIR__))
    ->withRouting(
        channels: __DIR__ . '/../routes/channels.php',
        web: __DIR__ . '/../routes/web.php',
        api: __DIR__ . '/../routes/api.php',
        commands: __DIR__ . '/../routes/console.php',
        health: '/up',
    )
    ->withMiddleware(function (Middleware $middleware): void {
        $middleware->alias([
            'admin' => \App\Http\Middleware\EnsureUserIsAdmin::class,
            '2fa' => \App\Http\Middleware\TwoFactorMiddleware::class,
        ]);
        $middleware->web(append: [
            \Illuminate\Session\Middleware\AuthenticateSession::class,
        ]);
        $middleware->validateCsrfTokens(except: [
            'api/webhooks/mailgun',
        ]);
    })
    ->withExceptions(function (Exceptions $exceptions): void {
        $exceptions->reportable(function (\Throwable $e) {
            try {
                // Ignore 404s and validation errors if desired, but for now log everything critical
                if ($e instanceof \Symfony\Component\HttpKernel\Exception\NotFoundHttpException) {
                    return;
                }

                // Avoid infinite loop if DB fails
                if (str_contains($e->getMessage(), 'system_logs')) {
                    return;
                }

                \App\Models\SystemLog::create([
                    'level' => 'error',
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString(),
                    'url' => request()->fullUrl(),
                    'method' => request()->method(),
                    'ip' => request()->ip(),
                    'user_id' => auth()->id(),
                    'context' => [],
                ]);
            } catch (\Throwable $loggingError) {
                // Fallback to standard log if DB logging fails
                \Illuminate\Support\Facades\Log::error('Failed to log to DB: ' . $loggingError->getMessage());
            }
        });
    })->create();
