<?php

namespace App\Traits;

use Carbon\Carbon;

trait PatrollerJobTransformer
{
    /**
     * Transform a single patroller job for API response
     */
    protected function transformPatrollerJob($job, $employee, $now = null)
    {
        if (!$now)
            $now = Carbon::now();

        $pivot = $job->employees->where('id', $employee->id)->first()->pivot ?? null;
        $status = $this->getPatrollerCalculatedStatus($job, $now);

        // Individual Status: If this specific employee has finished, status is completed.
        // If they have started but not finished, status is active.
        if ($pivot) {
            // If the global job is in progress, keep the individual status as 'active'
            // so it stays in the Active tab for everyone involved.
            if ($job->job_status === 'in_progress') {
                $status = 'active';
            } elseif (!empty($pivot->actual_end_at)) {
                $status = 'completed';
            } elseif (!empty($pivot->actual_start_at)) {
                $status = 'active';
            }
        }

        return [
            'id' => $job->id,
            'duty_number' => $job->duty_number,
            'route_name' => $job->route->name ?? 'N/A',
            'sites_count' => $job->route->sites->count() ?? 0,
            'status' => $status,
            'status_color' => $this->getPatrollerStatusColor($status),
            'from_time' => $job->from_time->format('Y-m-d H:i:s'),
            'to_time' => $job->to_time->format('Y-m-d H:i:s'),
            'scheduled_date' => $job->from_time->format('M d, Y'),
            'scheduled_time_range' => $job->from_time->format('H:i') . ' - ' . $job->to_time->format('H:i'),
            'duration' => $job->from_time->diff($job->to_time)->format('%h.%I') . 'h',
            'actual_start_at' => $pivot->actual_start_at ?? null,
            'actual_end_at' => $pivot->actual_end_at ?? null,
            'job_started_at' => $job->job_started_at,
            'job_ended_at' => $job->job_ended_at,
            'company' => [
                'id' => $job->company->id ?? null,
                'name' => $job->company->name ?? 'N/A',
                'logo' => isset($job->company->logo_url) ? $job->company->logo_url : null,
            ],
            'route' => $this->transformRouteForStatus($job, $employee, $status),
            'companions' => $job->employees->where('id', '!=', $employee->id)->map(function ($comp) {
                return [
                    'id' => $comp->id,
                    'name' => $comp->first_name . ' ' . $comp->last_name,
                    'profile_picture_url' => $comp->profile_picture_url,
                ];
            })->values(),
            'can_start' => empty($pivot->actual_start_at) &&
                ($job->job_status === 'pending' || $job->job_status === 'in_progress') &&
                $job->from_time->subMinutes(60)->isPast() &&
                $job->to_time->isFuture(),
            'can_end' => $status === 'active' &&
                !empty($pivot->actual_start_at) &&
                empty($pivot->actual_end_at) &&
                !$job->jobSites()->where('employee_id', $employee->id)->where('status', 'checked_in')->exists(),
            'can_cancel' => empty($pivot->actual_start_at) && $pivot->cancellation_status !== 'approved' && $pivot->cancellation_status !== 'pending' && $job->from_time->isFuture(),
            'cancellation_details' => [
                'status' => $pivot->cancellation_status ?? null,
                'reason' => $pivot->cancellation_reason ?? null,
                'requested_at' => $pivot->cancellation_requested_at ?? null,
            ],
        ];
    }

    protected function getPatrollerCalculatedStatus($job, $now)
    {
        if ($job->job_status === 'in_progress')
            return 'active';
        if (in_array($job->job_status, ['completed', 'auto_ended']))
            return 'completed';

        if ($job->from_time->isToday() && $job->to_time->isFuture() && $job->job_status === 'pending') {
            return 'today';
        }

        if ($job->from_time->isPast() && $job->job_status === 'pending') {
            // Check if any sites were visited
            if ($job->jobSites()->where('status', '!=', 'pending')->exists()) {
                return 'completed'; // Partially completed or finished without ending
            }
            return 'missed';
        }

        if ($job->job_status === 'pending' && $job->from_time->isFuture()) {
            return 'upcoming';
        }

        return $job->job_status;
    }

    protected function getPatrollerStatusColor($status)
    {
        return match ($status) {
            'active' => '#2563EB',    // Blue-600
            'completed' => '#16A34A', // Green-600
            'missed' => '#DC2626',    // Red-600
            'today' => '#D97706',     // Amber-600
            'upcoming' => '#D97706',  // Amber-600
            default => '#9CA3AF',     // Gray-400
        };
    }

    protected function transformRouteForStatus($job, $employee, $status)
    {
        $route = [
            'id' => $job->route->id ?? null,
            'name' => $job->route->name ?? 'N/A',
            'sites' => [],
        ];

        if (!isset($job->route->sites)) {
            return $route;
        }

        // 1. Check if user is actively checked into ANY site
        $currentCheckIn = $job->jobSites()->where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->first();

        // 2. If checked in, ONLY return that site (this forces the UI to focus on it)
        if ($currentCheckIn) {
            $site = $job->route->sites->where('id', $currentCheckIn->site_id)->first();
            if ($site) {
                $route['sites'][] = [
                    'id' => $site->id,
                    'name' => $site->name,
                    'address' => $site->address,
                    'lat' => $site->lat,
                    'lng' => $site->lng,
                    'status' => 'checked_in', // Explicit status for UI action buttons
                    'site_visit_id' => $currentCheckIn->id,
                    'issue_ticket' => true,
                ];
            }
            return $route;
        }

        // 3. If NOT checked in, return ALL sites with their individual statuses
        $pivot = $job->employees->where('id', $employee->id)->first()->pivot ?? null;
        $individualStarted = !empty($pivot->actual_start_at);
        $individualFinished = !empty($pivot->actual_end_at);

        $route['sites'] = $job->route->sites->map(function ($site) use ($job, $employee, $status, $individualStarted, $individualFinished) {
            // Find the latest status for this site
            $visit = $job->jobSites()->where('site_id', $site->id)
                ->where('employee_id', $employee->id)
                ->latest()
                ->first();

            $siteStatus = $visit ? $visit->status : 'pending';

            return [
                'id' => $site->id,
                'name' => $site->name,
                'address' => $site->address,
                'lat' => $site->lat,
                'lng' => $site->lng,
                'status' => $siteStatus,
                'can_checkin' => $status === 'active' && $individualStarted && !$individualFinished && ($siteStatus === 'pending' || $siteStatus === 'checked_out'),
                'site_visit_id' => $visit ? $visit->id : null,
            ];
        })->values();

        return $route;
    }
}
