<?php

namespace App\Traits;

use App\Models\ActivityLog;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Eloquent\Model;

trait LogsActivity
{
    public static function bootLogsActivity()
    {
        static::created(function ($model) {
            self::logAction($model, 'Created');
        });

        static::updated(function ($model) {
            self::logAction($model, 'Updated');
        });

        static::deleted(function ($model) {
            self::logAction($model, 'Deleted');
        });
    }

    protected static function logAction(Model $model, $action)
    {
        // Ignore if no user is logged in (e.g. seeders) unless required
        if (!Auth::check() && !Auth::guard('employee')->check()) {
            // allow logging even if unauthenticated? Maybe not.
            // But let's check
        }

        $userId = null;
        $actorInfo = '';

        if (Auth::guard('web')->check()) {
            $userId = Auth::guard('web')->id();
        } elseif (Auth::guard('employee')->check()) {
            $employee = Auth::guard('employee')->user();
            // Assuming we cannot store employee ID in user_id because of FK constraints
            // We append to description
            $actorInfo = " (by Employee: {$employee->first_name} {$employee->last_name})";
        }

        $details = $action . ' ' . class_basename($model) . $actorInfo;

        $properties = [];
        if ($action === 'Updated') {
            // Capture changes
            $changes = $model->getChanges();
            // Remove updated_at since it always changes
            unset($changes['updated_at']);

            if (empty($changes)) {
                return; // No meaningful changes
            }

            // Handle specific "password" case for cleaner log
            if (array_key_exists('password', $changes)) {
                $details = 'Changed Password for ' . class_basename($model) . $actorInfo;
                $changes['password'] = '********'; // Mask it
            }

            $properties = [
                'attributes' => $changes,
                // We could store old values using getOriginal() but getChanges() is sufficient for now
            ];

            // Append Name/Title for context if available
            $name = $model->name ?? ($model->first_name . ' ' . $model->last_name) ?? $model->title ?? $model->duty_number ?? null;
            if ($name) {
                $details .= ": $name";
            }
        } elseif ($action === 'Created' || $action === 'Deleted') {
            $name = $model->name ?? ($model->first_name . ' ' . $model->last_name) ?? $model->title ?? $model->duty_number ?? null;
            if ($name) {
                $details .= ": $name";
            }
        }

        ActivityLog::create([
            'user_id' => $userId,
            'action' => strtolower($action),
            'subject_type' => get_class($model),
            'subject_id' => $model->id,
            'description' => $details,
            'properties' => $properties,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
    }
}
