<?php

namespace App\Traits;

use Carbon\Carbon;

trait CalculatesEarnings
{
    /**
     * Calculate the earnings for a specific guard schedule.
     * Logic matches EmployeePayoutController for consistency.
     */
    protected function calculateGuardEarnings($schedule)
    {
        $pivot = $schedule->pivot;

        if (empty($pivot->actual_start_at) || empty($pivot->actual_end_at)) {
            return 0;
        }

        $start = Carbon::parse($pivot->actual_start_at);
        $end = Carbon::parse($pivot->actual_end_at);
        $actualHours = abs($end->diffInMinutes($start)) / 60;

        $rate = $pivot->wage_rate ?? 0;
        $calculatedPay = 0;

        // If base rate is 0, we use tiered wage component logic (wage_types)
        if ($rate <= 0) {
            $wageTypes = is_string($pivot->wage_types) ? json_decode($pivot->wage_types, true) : $pivot->wage_types;

            if (is_array($wageTypes) && count($wageTypes) > 0) {
                $remainingActualHours = $actualHours;

                foreach ($wageTypes as $wt) {
                    if ($remainingActualHours <= 0)
                        break;

                    $allocated = $wt['allocated_hours'] ?? 0;
                    $componentRate = $wt['rate'] ?? 0;

                    // Distribute actual hours into allocated buckets
                    $hoursForThisComponent = min($remainingActualHours, $allocated);
                    $calculatedPay += ($hoursForThisComponent * $componentRate);
                    $remainingActualHours -= $hoursForThisComponent;
                }

                // If worked more than allocated, pay overflow at the last component's rate
                if ($remainingActualHours > 0) {
                    $lastComponent = end($wageTypes);
                    $lastRate = $lastComponent['rate'] ?? 0;
                    $calculatedPay += ($remainingActualHours * $lastRate);
                }
            } else {
                // Fallback if no types provided but rate is 0 (shouldn't happen)
                $calculatedPay = $actualHours * $rate;
            }
        } else {
            // Simple flat rate calculation
            $calculatedPay = $actualHours * $rate;
        }

        $gas = $pivot->gas_rate ?? 0;
        return round($calculatedPay + $gas, 2);
    }

    /**
     * Calculate earnings for a patroller schedule.
     */
    protected function calculatePatrollerEarnings($schedule)
    {
        $pivot = $schedule->pivot;

        if (empty($pivot->actual_start_at) || empty($pivot->actual_end_at)) {
            return 0;
        }

        $start = Carbon::parse($pivot->actual_start_at);
        $end = Carbon::parse($pivot->actual_end_at);
        $actualHours = abs($end->diffInMinutes($start)) / 60;

        $rate = $pivot->wage_rate ?? 0;
        $calculatedPay = 0;

        // If base rate is 0, we use tiered wage component logic (wage_types)
        if ($rate <= 0) {
            $wageTypes = is_string($pivot->wage_types) ? json_decode($pivot->wage_types, true) : $pivot->wage_types;

            if (is_array($wageTypes) && count($wageTypes) > 0) {
                // Ensure SIN (id=1) is first
                usort($wageTypes, function ($a, $b) {
                    $aId = $a['id'] ?? ($a['wage_type_id'] ?? 0);
                    $bId = $b['id'] ?? ($b['wage_type_id'] ?? 0);
                    $aIsSin = $aId == 1 ? 0 : 1;
                    $bIsSin = $bId == 1 ? 0 : 1;
                    return $aIsSin - $bIsSin;
                });

                $remainingActualHours = $actualHours;

                foreach ($wageTypes as $wt) {
                    if ($remainingActualHours <= 0)
                        break;

                    $allocated = $wt['allocated_hours'] ?? 0;
                    $componentRate = $wt['rate'] ?? 0;

                    $hoursForThisComponent = min($remainingActualHours, $allocated);
                    $calculatedPay += ($hoursForThisComponent * $componentRate);
                    $remainingActualHours -= $hoursForThisComponent;
                }

                // If worked more than allocated, pay overflow at the last component's rate
                if ($remainingActualHours > 0) {
                    $lastComponent = end($wageTypes);
                    $lastRate = $lastComponent['rate'] ?? 0;
                    $calculatedPay += ($remainingActualHours * $lastRate);
                }
            } else {
                $calculatedPay = $actualHours * $rate;
            }
        } else {
            // Simple flat rate calculation
            $calculatedPay = $actualHours * $rate;
        }

        return round(
            $calculatedPay +
            ($pivot->gas_rate ?? 0) +
            ($pivot->allowance ?? 0) +
            ($pivot->other_expense ?? 0),
            2
        );
    }
}
