<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class ImageService
{
    /**
     * Process an image: convert to WebP, compress to 60%, and store in specific path.
     *
     * @param UploadedFile $file
     * @param string $path
     * @param int $quality
     * @return string|null
     */
    public static function processAndStore(UploadedFile $file, $path, $quality = 60)
    {
        // Get image data
        $filePath = $file->getRealPath();
        $imageInfo = @\getimagesize($filePath);

        if (!$imageInfo) {
            return $file->store($path, 'public');
        }

        $mime = $imageInfo['mime'];
        $gdInfo = \function_exists('gd_info') ? \gd_info() : [];
        $supportsJpeg = $gdInfo['JPEG Support'] ?? false;
        $supportsWebp = $gdInfo['WebP Support'] ?? false;
        $supportsPng = $gdInfo['PNG Support'] ?? false;

        $image = null;

        // Only attempt GD processing if the format is supported for reading
        if ($mime === 'image/jpeg' && $supportsJpeg) {
            $image = @\imagecreatefromjpeg($filePath);
        } elseif ($mime === 'image/png' && $supportsPng) {
            $image = @\imagecreatefrompng($filePath);
            if ($image) {
                \imagepalettetotruecolor($image);
                \imagealphablending($image, true);
                \imagesavealpha($image, true);
            }
        } elseif ($mime === 'image/webp' && $supportsWebp) {
            $image = @\imagecreatefromwebp($filePath);
        }

        // If GD failed to load the image or doesn't support it, fallback to binary copy
        if (!$image) {
            // Fulfill the hierarchy requirement even without compression
            $filename = \pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME) . '_' . \uniqid() . '.' . $file->getClientOriginalExtension();
            return $file->storeAs($path, $filename, 'public');
        }

        // Ensure directory exists
        Storage::disk('public')->makeDirectory($path);

        $relativeStoragePath = '';

        // Save as WebP if supported, otherwise fallback to JPEG if supported, otherwise original
        if ($supportsWebp) {
            $filename = \pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME) . '_' . \uniqid() . '.webp';
            $relativeStoragePath = $path . '/' . $filename;
            $absolutePath = Storage::disk('public')->path($relativeStoragePath);
            \imagewebp($image, $absolutePath, $quality);
        } elseif ($supportsJpeg) {
            $filename = \pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME) . '_' . \uniqid() . '.jpg';
            $relativeStoragePath = $path . '/' . $filename;
            $absolutePath = Storage::disk('public')->path($relativeStoragePath);
            \imagejpeg($image, $absolutePath, $quality);
        } else {
            // Cannot compress/convert, save original extension but in the right folder
            \imagedestroy($image);
            $filename = \pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME) . '_' . \uniqid() . '.' . $file->getClientOriginalExtension();
            return $file->storeAs($path, $filename, 'public');
        }

        \imagedestroy($image);

        return $relativeStoragePath;
    }

    /**
     * Generate the folder structure requested by the user.
     * 
     * @param string $type check-in, check-out, report-incident
     * @param string $dutyId
     * @param int|null $incidentIndex
     * @return string
     */
    public static function getFolderPath($type, $dutyId, $siteName = null, $timestamp = null)
    {
        $dt = $timestamp ? Carbon::parse($timestamp) : \now();
        $formattedTimestamp = $dt->format('Y-m-d_H-i-s');

        $cleanSiteName = str_replace([' ', '/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $siteName ?? 'Job');
        $dutyFolder = "{$cleanSiteName}_{$formattedTimestamp}_{$dutyId}";

        switch ($type) {
            case 'check-in':
                return "images/{$dutyFolder}/checkin_images";
            case 'check-out':
                return "images/{$dutyFolder}/checkout_images";
            case 'report-incident':
                $incidentTime = \now()->format('H-i-s');
                $incidentFolderName = "{$cleanSiteName}_{$incidentTime}";
                return "images/{$dutyFolder}/report_incident/{$incidentFolderName}";
            default:
                return "images/{$dutyFolder}/{$type}";
        }
    }

    /**
     * Generate the folder structure for patrollers.
     * 
     * @param string $dutyId
     * @param string $siteName
     * @return string
     */
    public static function getPatrollerFolderPath($dutyId, $siteName)
    {
        $cleanSiteName = str_replace([' ', '/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $siteName);
        return "patroller_images/tickets_issued_{$dutyId}/{$dutyId}_{$cleanSiteName}";
    }
}
