<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class FcmService
{
    protected $serviceAccount;
    protected $projectId;

    public function __construct()
    {
        $json = Setting::where('key', 'firebase_service_account_json')->value('value');
        if ($json) {
            $this->serviceAccount = json_decode($json, true);
            $this->projectId = $this->serviceAccount['project_id'] ?? null;
        }
    }

    /**
     * Get OAuth2 Access Token for FCM V1
     */
    protected function getAccessToken()
    {
        if (!$this->serviceAccount) {
            Log::error('FCM Service Account JSON not found or invalid.');
            return null;
        }

        // Cache the token for 55 minutes (tokens usually last 1 hour)
        return Cache::remember('fcm_access_token', 3300, function () {
            $clientEmail = $this->serviceAccount['client_email'];
            $privateKey = $this->serviceAccount['private_key'];

            $header = json_encode(['alg' => 'RS256', 'typ' => 'JWT']);
            $now = time();
            $payload = json_encode([
                'iss' => $clientEmail,
                'sub' => $clientEmail,
                'aud' => 'https://oauth2.googleapis.com/token',
                'iat' => $now,
                'exp' => $now + 3600,
                'scope' => 'https://www.googleapis.com/auth/firebase.messaging'
            ]);

            $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
            $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));

            $signatureInput = $base64UrlHeader . "." . $base64UrlPayload;
            $signature = '';

            if (!openssl_sign($signatureInput, $signature, $privateKey, 'SHA256')) {
                Log::error('FCM: OpenSSL Signing Failed');
                return null;
            }

            $jwt = $signatureInput . "." . str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));

            $response = Http::asForm()->post('https://oauth2.googleapis.com/token', [
                'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                'assertion' => $jwt,
            ]);

            if ($response->successful()) {
                return $response->json()['access_token'];
            }

            Log::error('FCM: Failed to exchange JWT for Access Token', ['response' => $response->body()]);
            return null;
        });
    }

    /**
     * Send Notification to a specific device token
     */
    public function send($token, $title, $body, $data = [])
    {
        $accessToken = $this->getAccessToken();
        if (!$accessToken || !$this->projectId) {
            return false;
        }

        $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";

        $message = [
            'message' => [
                'token' => $token,
                'notification' => [
                    'title' => $title,
                    'body' => $body,
                ],
                'data' => (!empty($data)) ? $data : null,
                'android' => [
                    'priority' => 'high',
                ],
                'apns' => [
                    'payload' => [
                        'aps' => [
                            'sound' => 'default',
                        ],
                    ],
                ],
            ],
        ];

        // Remove data if null to avoid 400 bad request
        if (empty($message['message']['data'])) {
            unset($message['message']['data']);
        }

        $response = Http::withToken($accessToken)
            ->withHeaders(['Content-Type' => 'application/json'])
            ->post($url, $message);

        if ($response->successful()) {
            return true;
        }

        Log::warning('FCM Send Failed', [
            'token' => substr($token, 0, 10) . '...',
            'status' => $response->status(),
            'body' => $response->body()
        ]);

        // Handle Invalid Token (NotRegistered) by deleting it
        if ($response->status() === 404 || str_contains($response->body(), 'UNREGISTERED')) {
            \App\Models\FcmToken::where('token', $token)->delete();
        }

        return false;
    }

    /**
     * Send to multiple tokens (helper)
     */
    public function sendMulticast($tokens, $title, $body, $data = [])
    {
        $successCount = 0;
        foreach ($tokens as $token) {
            if ($this->send($token, $title, $body, $data)) {
                $successCount++;
            }
        }
        return $successCount;
    }
}
