<?php

namespace App\Services;

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Color;
use Carbon\Carbon;

class ExcelExportService
{
    /**
     * Generate a professional Excel report.
     *
     * @param string $reportTitle The main title of the report.
     * @param string $reportType Subtitle describing the report type.
     * @param array $headers List of column headers.
     * @param array $data 2D array of data rows.
     * @param array $options Optional settings (statusColors, etc.).
     * @return \Symfony\Component\HttpFoundation\StreamedResponse
     */
    public function generateReport($reportTitle, $reportType, array $headers, array $data, array $options = [])
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        $this->applyBrandedHeader($sheet, $reportTitle, $reportType, count($headers));
        
        // Table Headers (Row 5)
        $sheet->fromArray($headers, NULL, 'A5');
        $this->applyTableHeaders($sheet, "A5:{$this->getColLetter(count($headers))}5");

        // Data Population
        $currentRow = 6;
        $lastColLetter = $this->getColLetter(count($headers));
        foreach ($data as $rowIndex => $rowData) {
            $sheet->fromArray($rowData, NULL, 'A' . $currentRow);
            
            // Zebra Striping
            if ($currentRow % 2 == 0) {
                $sheet->getStyle("A{$currentRow}:{$lastColLetter}{$currentRow}")->getFill()
                    ->setFillType(Fill::FILL_SOLID)
                    ->getStartColor()->setARGB('FFF1F5F9'); // Slate-50
            }

            // Optional Status Coloring
            if (isset($options['status_column_index'])) {
                $statusColIndex = $options['status_column_index'];
                $statusColLetter = $this->getColLetter($statusColIndex + 1);
                $statusCell = "{$statusColLetter}{$currentRow}";
                $statusValue = strtolower($rowData[$statusColIndex] ?? '');
                
                $color = $this->getStatusColor($statusValue);
                $sheet->getStyle($statusCell)->getFont()->setBold(true)->setColor(new Color($color));
            }

            $sheet->getRowDimension($currentRow)->setRowHeight(28);
            $currentRow++;
        }

        $lastRow = $currentRow - 1;
        if (count($data) > 0) {
            $this->applyDataStyles($sheet, "A6:{$lastColLetter}{$lastRow}", count($headers), $options);
        }

        $this->applyFinalPolish($sheet, count($headers));

        $filename = ($options['filename_prefix'] ?? 'Report') . "_" . now()->format('Ymd_His') . ".xlsx";

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename);
    }

    /**
     * Apply professional header with company name and report title
     */
    public function applyBrandedHeader($sheet, $reportTitle, $reportType, $colCount)
    {
        $companyId = session('selected_company_id');
        $company = \App\Models\Company::find($companyId);
        $companyName = $company ? $company->name : 'ShiftMatrix';

        // Sanitize sheet title
        $cleanTitle = str_replace(['\\', '/', '*', '[', ']', ':', '?'], ' ', $reportTitle);
        $sheet->setTitle(substr($cleanTitle, 0, 31));

        $lastColLetter = $this->getColLetter($colCount);
        $sheet->mergeCells("A1:{$lastColLetter}1");
        $sheet->setCellValue('A1', strtoupper($companyName) . " - " . strtoupper($reportTitle));
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(16)->setColor(new Color('FF1E293B'));
        $sheet->getRowDimension(1)->setRowHeight(45);
        
        $sheet->setCellValue('A2', "Report Type: " . ucfirst($reportType));
        $sheet->setCellValue('A3', "Period/Generated: " . now()->format('M d, Y h:i A'));
        $sheet->getStyle('A2:A3')->getFont()->setItalic(true)->setColor(new Color('FF64748B'));
        $sheet->getRowDimension(2)->setRowHeight(25);
        $sheet->getRowDimension(3)->setRowHeight(25);
        $sheet->getRowDimension(4)->setRowHeight(15); // Spacer
    }

    /**
     * Apply standard Slate-700 header style
     */
    public function applyTableHeaders($sheet, $range)
    {
        $headerStyle = [
            'font' => ['bold' => true, 'color' => ['argb' => 'FFFFFFFF']],
            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['argb' => 'FF334155']],
            'borders' => ['bottom' => ['borderStyle' => Border::BORDER_MEDIUM, 'color' => ['argb' => 'FF000000']]],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
                'vertical' => Alignment::VERTICAL_CENTER,
                'indent' => 1,
            ],
        ];
        $sheet->getStyle($range)->applyFromArray($headerStyle);
        
        // Set row heights for the range
        preg_match_all('/\d+/', $range, $matches);
        if (!empty($matches[0])) {
            foreach ($matches[0] as $rowNum) {
                $sheet->getRowDimension($rowNum)->setRowHeight(35);
            }
        }
    }

    /**
     * Apply standard borders, indent, and heights to data rows
     */
    public function applyDataStyles($sheet, $range, $colCount, array $options = [])
    {
        $dataStyle = [
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['argb' => 'FFE2E8F0'], // Slate-200
                ],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
                'vertical' => Alignment::VERTICAL_CENTER,
                'indent' => 1,
            ],
        ];
        $sheet->getStyle($range)->applyFromArray($dataStyle);

        // Handle specific column alignments
        $centerColumns = $options['center_columns'] ?? [];
        for ($i = 1; $i <= $colCount; $i++) {
            $letter = $this->getColLetter($i);
            if (in_array($i - 1, $centerColumns)) {
                $sheet->getStyle("{$letter}5:{$letter}" . $sheet->getHighestRow())
                    ->getAlignment()
                    ->setHorizontal(Alignment::HORIZONTAL_CENTER)
                    ->setIndent(0);
            }
        }

        // Set row heights
        preg_match_all('/\d+/', $range, $matches);
        if (!empty($matches[0])) {
            $start = min($matches[0]);
            $end = max($matches[0]);
            for ($i = $start; $i <= $end; $i++) {
                $sheet->getRowDimension($i)->setRowHeight(28);
            }
        }
    }

    /**
     * Handle auto-sizing and extra column padding
     */
    public function applyFinalPolish($sheet, $colCount)
    {
        for ($i = 1; $i <= $colCount; $i++) {
            $letter = $this->getColLetter($i);
            $sheet->getColumnDimension($letter)->setAutoSize(true);
        }
        $sheet->calculateColumnWidths();
        for ($i = 1; $i <= $colCount; $i++) {
            $letter = $this->getColLetter($i);
            $currentWidth = $sheet->getColumnDimension($letter)->getWidth();
            $sheet->getColumnDimension($letter)->setAutoSize(false);
            $sheet->getColumnDimension($letter)->setWidth($currentWidth + 7);
        }
    }

    public function getColLetter($index)
    {
        return \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($index);
    }

    private function getStatusColor($status)
    {
        $status = strtolower($status);
        if (in_array($status, ['completed', 'active', 'success', 'paid'])) return 'FF059669'; // Emerald-600
        if (in_array($status, ['today', 'upcoming', 'pending'])) return 'FF2563EB'; // Blue-600
        if (in_array($status, ['missed', 'cancelled', 'failed', 'rejected', 'banned'])) return 'FFDC2626'; // Red-600
        return 'FF64748B'; // Slate-500
    }
}
