<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Site extends Model
{
    use HasFactory, \App\Traits\LogsActivity;

    protected $guarded = ['id'];
    protected $appends = ['address', 'lat', 'lng'];

    public function getAddressAttribute()
    {
        $address = $this->address_line_1;
        if ($this->address_line_2) {
            $address .= ' ' . $this->address_line_2;
        }
        if ($this->city) {
            $address .= ', ' . $this->city;
        }
        return $address;
    }

    public function getLatAttribute()
    {
        return (float) $this->latitude;
    }

    public function getLngAttribute()
    {
        return (float) $this->longitude;
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    public function state()
    {
        return $this->belongsTo(State::class);
    }

    public function bannedEmployees()
    {
        return $this->belongsToMany(Employee::class, 'employee_site_bans')
            ->withPivot('reason', 'banned_at')
            ->withTimestamps();
    }

    public function checkpoints()
    {
        return $this->hasMany(Checkpoint::class);
    }

    public function tourRoutes()
    {
        return $this->hasMany(TourRoute::class);
    }

    public function getLogoUrlAttribute()
    {
        if ($this->logo) {
            return asset('storage/' . $this->logo);
        }

        $path = \App\Models\Setting::where('key', 'default_site_image')->value('value');
        return $path ? asset('storage/' . $path) : asset('images/default-site.png');
    }

    public function isProfileComplete()
    {
        $required = [
            'rate',
            'address_line_1',
            'country_id',
            'state_id',
            'city',
            'zip_code',
            'latitude',
            'longitude'
        ];

        foreach ($required as $field) {
            if (empty($this->$field)) {
                return false;
            }
        }
        return true;
    }
}
